import sys
import os
import fcntl
from itertools import chain

import six
import gevent

try:
    import gevent.coros as coros
except ImportError:
    import gevent.lock as coros

import gevent.event
import yaml

from . import namespace
from .namespace import InstallationUnit
from .service import State
from ..dirutils import prepare_downloaddir
from ..fsyncqueue import Fsyncer
from ..framework.component import Component
from ..framework.greendeblock import Deblock
from ..framework.utils import ensure_dir
from ..downloader import parse_advertised_release
from ..migratetools import fixup_state

CONFIG_UPDATE_PERIOD = 60


# Sample services config:
#
#   skynet.skycore:
#       version:
#         # NOTE: BELOW IS NOT REAL GENISYS STRUCTURE REPRESENTING SANDBOX RESOURCE
#         md5: 68b329da9893e34099c7d8ad5cb9c940
#         urls:
#         skynet: "rbtorrent:11079e38ae9091d1813859fe3705181a52e2159e"
#         http:
#           - "http://lucid.build.skydev.search.yandex.net:13578/4/3/61935334/dist/skynet.bin"
#           - "http://sandbox-oxygen26.search.yandex.net:13578/4/3/61935334/dist/skynet.bin"
#         rsync:
#           - "rsync://lucid.build.skydev.search.yandex.net/sandbox-tasks/4/3/61935334/dist/skynet.bin"
#           - "rsync://sandbox-oxygen26.search.yandex.net/sandbox-tasks/4/3/61935334/dist/skynet.bin"
#       namespaces:
#         skynet_main:
#           cqudp:
#             # NOTE: BELOW IS NOT REAL GENISYS STRUCTURE REPRESENTING SANDBOX RESOURCE
#             md5: 68b329da9893e34099c7d8ad5cb9c940
#             urls:
#               skynet: "rbtorrent:11079e38ae9091d1813859fe3705181a52e2159e"
#               http:
#                 - "http://lucid.build.skydev.search.yandex.net:13578/4/3/61935334/dist/skynet.bin"
#                 - "http://sandbox-oxygen26.search.yandex.net:13578/4/3/61935334/dist/skynet.bin"
#               rsync:
#                 - "rsync://lucid.build.skydev.search.yandex.net/sandbox-tasks/4/3/61935334/dist/skynet.bin"
#                 - "rsync://sandbox-oxygen26.search.yandex.net/sandbox-tasks/4/3/61935334/dist/skynet.bin"
#           copier:
#             # NOTE: BELOW IS NOT REAL GENISYS STRUCTURE REPRESENTING SANDBOX RESOURCE
#             md5: b026324c6904b2a9cb4b88d6d61c81d1
#             urls:
#               skynet: "rbtorrent:21079e38ae9091d1813859fe3705181a52e2159e"
#               http:
#                 - "http://lucid.build.skydev.search.yandex.net:13578/4/3/61935335/dist/skynet.bin"
#                 - "http://sandbox-oxygen26.search.yandex.net:13578/4/3/61935335/dist/skynet.bin"
#               rsync:
#                 - "rsync://lucid.build.skydev.search.yandex.net/sandbox-tasks/4/3/61935335/dist/skynet.bin"
#                 - "rsync://sandbox-oxygen26.search.yandex.net/sandbox-tasks/4/3/61935335/dist/skynet.bin"
#
class ServiceUpdater(Component):
    CURRENT_STATE_VERSION = 4

    _async_listdir = Deblock.wrap_fun(os.listdir, static=True)
    _async_unlink = Deblock.wrap_fun(os.unlink, static=True)
    _async_prepare_downloaddir = Deblock.wrap_fun(prepare_downloaddir, static=True)

    class UpdateException(Exception):
        @property
        def reason(self):
            return self.args[0]

        def __str__(self):
            return '%s:\n%s' % (
                self.args[0],
                '\n'.join(self.args[1:])
            )

    def __init__(self,
                 deblock,
                 base,
                 workdir,
                 rundir,
                 linkdir,
                 apidir,
                 downloaddir,
                 statefile,
                 output_logger_factory,
                 starter,
                 config,
                 namespaces,
                 core_revision,
                 skynetdir=None,
                 supervisordir=None,
                 parent=None,
                 reporter=None,
                 paused=False,
                 ):
        """
        :param ..framework.greendeblock.Deblock deblock: deblock for i/o operations
        :param str base: root dir used to construct relative paths for state
        :param str workdir: root dir where namespaces with services are stored
        :param str rundir: root dir where meta information to be stored (liner socket, services state etc.)
        :param str linkdir: root dir where symlinks to service parts are stored
        :param str apidir: root dir where service API dicts are stored
        :param str downloaddir: dir used for temporary data such as downloaded archives
        :param str statefile: file used to dump current state
        :param ..framework.logger.HierachicalLoggerFactory output_logger_factory: factory for
                                                           creating custom loggers for service outputs
        :param skycore.service.Starter starter: starter for processes
        :param .registry.Registry config: config registry to query
        :param dict namespaces: map [str name -> Namespace ns] of available namespaces
        :param str core_revision: uniq revision (currently md5 hash of sandbox release)
                                  denoting current skycore release. If it's changed,
                                  updater will not update any services until gosky
                                  to update and restart skycore
        :param str skynetdir: path to skynet root, can be used in services as ${SKYNETDIR}
        :param str supervisordir: path to skynet supervisor root, can be used in services as ${SUPERVISORDIR}
        :param Component parent: parent component owning this one
        :param .statereporter.StateReporter reporter: object reporting services status changes to heartbeat
        """
        self._paused = paused
        self.deblock = deblock
        self.core_revision = core_revision
        self.base = base
        self.workdir = workdir
        self.rundir = rundir
        self.linkdir = linkdir
        self.apidir = apidir
        self.downloaddir = downloaddir
        self.skynetdir = skynetdir
        self.supervisordir = supervisordir
        self.statefile = statefile
        self.output_logger_factory = output_logger_factory
        self.starter = starter
        self.config = config
        self.namespaces = namespaces
        self.ctx_write_lock = coros.RLock()
        self.update_lock = coros.RLock()
        self.context_changed_event = coros.Semaphore()
        self.downloads_cache = {}
        self.locked_state = None
        self.reporter = reporter
        self.fsyncer = None

        self.stats = None

        super(ServiceUpdater, self).__init__(logname='svcupdr', parent=parent)

        self.log.info("Work directory: %r", workdir)
        self.log.info("Run directory: %r", rundir)

    @property
    def paused(self):
        return self._paused

    @paused.setter
    def paused(self, value):
        value = bool(value)
        oldval, self._paused = self._paused, value
        if oldval != value and not value:
            self.check_loop.wakeup()
            self.context_changed_event.release()

    def set_stats(self, stats):
        self.stats = stats

    def config_changed(self, path, config):
        with self.update_lock:
            self.check_all_namespaces(self.log, reason='config changed')

    @Component.green_loop(logname='ns_check')
    def check_loop(self, log):
        if self.paused:
            log.debug("updates are paused, doing nothing")
            return CONFIG_UPDATE_PERIOD

        with self.update_lock:
            try:
                self.check_all_namespaces(log, reason='regular checking every %ds' % (CONFIG_UPDATE_PERIOD, ))
            except self.UpdateException as exc:
                exc_info = sys.exc_info()
                self.config.send_report(log, False, str(exc))
                self.log.error("update loop failed: ", exc_info=exc_info)
            else:
                self.config.send_report(log, True, '')

        return CONFIG_UPDATE_PERIOD

    @Component.green_loop(logname='ns_ctxdmp')
    def ctxdumper(self, log):
        while True:
            if not self.context_changed_event.wait(30.):
                continue

            updates = self.context_changed_event.counter
            try:
                self.write_context()
            except Exception as e:
                log.exception("context write failed: %s" % (e,), exc_info=sys.exc_info())

            for _ in six.moves.xrange(updates):
                self.context_changed_event.acquire()

    def check_all_namespaces(self, log, reason, start_on_install=True):
        try:
            version = self.config.query(['skynet', 'versions'], deepcopy=False)
            namespaces = self.config.query(['skynet', 'skycore', 'namespaces'], deepcopy=False)

            skycore_ver = parse_advertised_release(version['config'])
            if self.core_revision != 'dev' and skycore_ver['svn_url'] != self.core_revision:
                log.info(
                    "Skycore revision has changed (old %r, new %r), "
                    "will not perform update until skycore updated!",
                    self.core_revision,
                    skycore_ver['svn_url']
                )
                return
            namespaces = namespaces['subsections']
        except Exception as e:
            log.warning("Cannot read config: ", exc_info=sys.exc_info())
            raise self.UpdateException("Failed to fetch new config", str(e))

        errors = []
        to_check = []
        empty = []

        for ns, cfg in namespaces.items():
            if cfg is None:
                empty.append(ns)
                continue
            to_check.append(ns)

        # special hack to update 'skynet' first of all
        to_check.sort(key=lambda val: (0 if val == 'skynet' else 1, val))

        if empty:
            log.info(
                'Got %d namespaces (%s) and %d empty (%s), will check them (reason: %s)',
                len(to_check),
                ', '.join(to_check),
                len(empty),
                ', '.join(empty),
                reason
            )
        else:
            log.info(
                'Got %d namespaces (%s), will check them (reason: %s)',
                len(to_check),
                ', '.join(to_check),
                reason
            )

        for ns in to_check:
            cfg = namespaces[ns]
            try:
                namespace = self.ensure_namespace(ns, log=log)

                # if no skybone, try to install it first
                if (
                    start_on_install
                    and ns == 'skynet'
                    and 'skybone' not in namespace.services
                    and cfg.get('subsections', {}).get('skybone', {}).get('config')
                ):
                    self._check_namespace(
                        log,
                        namespace,
                        {'subsections': {'skybone': cfg['subsections']['skybone']}},
                        start_on_install=start_on_install)
                self._check_namespace(log, namespace, cfg, start_on_install=start_on_install)
            except Exception as e:
                log.warning("Section %r is broken:" % (ns,), exc_info=True)
                errors.append("  namespace %r failed with: %s" % (ns, e))

        if errors:
            raise self.UpdateException('Failed to apply config', *errors)

    def _make_namespace(self, name, **kwargs):
        workdir = os.path.join(self.workdir, name)
        rundir = os.path.join(self.rundir, name)
        ensure_dir(rundir)
        apidir = os.path.join(self.apidir, name)
        ensure_dir(apidir)
        linkdir = os.path.join(self.linkdir, name)
        ns = namespace.Namespace(starter=self.starter,
                                 deblock=self.deblock,
                                 name=name,
                                 workdir=workdir,
                                 rundir=rundir,
                                 linkdir=linkdir,
                                 apidir=apidir,
                                 skynetdir=self.skynetdir,
                                 supervisordir=self.supervisordir,
                                 registry=self.config,
                                 output_logger_factory=self.output_logger_factory.child(name),
                                 context_changed_event=self.context_changed_event,
                                 parent=self,
                                 reporter=self.reporter,
                                 **kwargs)
        self.namespaces[name] = ns
        return ns

    def ensure_namespace(self, name, log=None):
        log = log or self.log
        if name not in self.namespaces:
            log.debug("namespace %r is new, creating", name)
            self._make_namespace(name, stats=self.stats)
            self.namespaces[name].start()
            self.context_changed_event.release()

        return self.namespaces[name]

    def _skybone_available(self):
        ns = self.namespaces.get('skynet')
        if not ns:
            return False

        svc = ns.services.get('skybone')
        if not svc:
            return False

        return svc.state in (State.RUNNING, State.PRERUNNING)

    def _download(self, log, namespace, existing_hashes, downloaded_hashes, unit):
        if unit.md5 in namespace.rhashes:
            old_unit_dirty = any(
                namespace.get_service_metainfo(service_name).dirty
                for service_name in namespace.rhashes[unit.md5]
            )
            existing_hashes.add(unit.md5)
            unit.dirty = old_unit_dirty
            # update existing meta info
            if not old_unit_dirty:
                for service_name in namespace.rhashes[unit.md5]:
                    namespace.services_meta[service_name].release = unit.release
                return

        if unit.md5 in downloaded_hashes:
            return

        unit.download(self.downloaddir, log, skybone_available=self._skybone_available())
        unit.extract(self.downloaddir, log, fsyncqueue=self.fsyncer)
        downloaded_hashes.add(unit.md5)
        unit.collect_services(log)

        return unit

    def _collect_conflicting(self, units):
        visited = {}
        conflicting = set()

        for unit in units:
            for name in unit.service_cfgs:
                if name in visited:
                    conflicting.add(name)
                    unit.conflicting = True
                    visited[name].conflicting = True
                    continue
                visited[name] = unit
        return conflicting

    def _check_namespace(self, log, namespace, cfg, start_on_install=True):
        existing_hashes = set()
        downloaded_hashes = set()

        try:
            units = []
            for section, section_cfg in cfg['subsections'].iteritems():
                try:
                    unit = InstallationUnit.from_cfg(section, section_cfg, self.deblock)
                    if unit is not None:
                        units.append(unit)
                except ValueError:
                    pass
                except Exception as e:
                    log.warning("%s", e.message)

            required_to_download = sum(unit.download_size for unit in units)
            self._async_prepare_downloaddir(log, self.downloaddir, required_to_download, self.downloads_cache)

            required_to_install = filter(
                None,
                (self._download(log, namespace, existing_hashes, downloaded_hashes, unit)
                 for unit in units)
            )

            conflicting = self._collect_conflicting(required_to_install)
            required_to_install = filter(lambda unit: not unit.conflicting, required_to_install)

            old_services = namespace.hashes
            new_services = {name: unit for unit in required_to_install for name in unit.service_cfgs}

            to_uninstall = {name for name in old_services
                            if (name not in new_services  # these ones are not in new_services
                                # we haven't got any strange config from hell
                                and old_services[name] not in existing_hashes
                                and name not in conflicting
                                )
                            }

            to_reinstall = {
                name: unit
                for name, unit in six.iteritems(new_services)
                if unit.dirty
                and name not in conflicting
            }

            to_upgrade = {
                name for name, unit in six.iteritems(new_services)
                if name in old_services
                and not unit.dirty
                and unit.md5 != old_services.get(name)
                and name not in conflicting
            }

            to_install = {
                name for name in new_services
                if name not in to_upgrade
                and name not in to_reinstall
                and name not in conflicting
            }

            todo_log_msgs = ['%s: %s' % p for p in filter(lambda p: p[1], (
                ('REMOVE', to_uninstall),
                ('INSTALL', to_install),
                ('UPGRADE', to_upgrade),
                ('REINSTALL', to_reinstall.keys()),
                ('BROKEN', conflicting),
            ))]
            if todo_log_msgs:
                namespace.log.info('Services should be changed: [%s]', '; '.join(todo_log_msgs))

            installed = []

            # uninstall everything old
            for srvc in to_uninstall:
                namespace.uninstall_service(srvc)

            # flush reinstalled dirty flag
            for unit in to_reinstall.values():
                unit.dirty = False

            # unpack everything new
            try:
                installed.extend(
                    chain.from_iterable(
                        namespace.install_services(unit)
                        for unit in required_to_install
                    )
                )
            except BaseException:
                for s, _ in installed:
                    s.stop()
                raise

            namespace.inject_services(installed, log=log, start_on_install=start_on_install)
        finally:
            for unit in units:
                unit.close()

            namespace.cleanup()

    @property
    def context(self):
        ctx = {
            name: ns.context
            for name, ns in self.namespaces.items()
        }
        ctx['__version__'] = self.CURRENT_STATE_VERSION
        return ctx

    def write_context(self):
        def _inner(ctx):
            f = open(new_path, 'wb')
            exc = None
            try:
                fcntl.flock(f.fileno(), fcntl.LOCK_EX)
                yaml.dump(
                    ctx,
                    f,
                    default_flow_style=False,
                    Dumper=getattr(yaml, 'CSafeDumper', yaml.SafeDumper)
                )
                f.flush()
                os.fsync(f.fileno())
                os.rename(new_path, self.statefile)

            except Exception:
                exc = sys.exc_info()
                f.close()
                raise
            finally:
                # unlock old state in any case so that even if write failed
                # 'skyctl check' to consider old state as inconsistent one
                if self.locked_state is not None:
                    try:
                        fcntl.flock(self.locked_state.fileno(), fcntl.LOCK_UN)
                        self.locked_state.close()
                    except ValueError as e:  # old state is already closed
                        self.log.error("failed to close old state: %s", e)
                        if exc is not None:
                            six.reraise(*exc)
                    except Exception as e:
                        self.log.error("failed to close old state: %s", e)
                        if exc is not None:
                            six.reraise(*exc)
                        else:
                            raise

            self.locked_state = f

        new_path = self.statefile + '.new'
        with self.ctx_write_lock:
            ctx = self.context
            self.deblock.apply(_inner, ctx)

    @Deblock.wrap_fun
    def _read_state(self, workdir, path):
        if not os.path.exists(path):
            return {}

        try:
            with open(path, 'rb') as f:
                data = yaml.load(f, Loader=getattr(yaml, 'CSafeLoader', yaml.SafeLoader))
        except (EnvironmentError, yaml.YAMLError) as e:
            self.log.warning("failed to read state file, will drop it and start skycore from scratch: %s", e)
            try:
                os.unlink(path)
            except EnvironmentError as ex:
                self.log.error(
                    "failed to remove corrupted state file! The skycore will lose all data after restart: %s",
                    ex
                )
            return {}

        try:
            fixup_state(workdir, data)
            return data
        except Exception as e:
            self.log.warning("failed to parse state file, will drop it and start skycore from scratch: %s", e)
            return {}

    def restore_context(self, context, restart_all=None):
        version = context.pop('__version__', 1)
        if version > self.CURRENT_STATE_VERSION:
            self.log.warning(
                "state version %s is not supported, will drop it and start from scratch",
                version
            )
        else:
            for name, ctx in context.iteritems():
                self._make_namespace(name, context=ctx, stats=self.stats, restart_all=restart_all)

    def read_context(self, restart_all=None):
        context = self._read_state(self.base, self.statefile)

        if not context:
            return

        self.restore_context(context, restart_all=restart_all)

    def cleanup_liner_sockets(self):
        alive_uuids = set(
            chain.from_iterable(
                service.proc_uuids()
                for ns in six.itervalues(self.namespaces)
                for service in six.itervalues(ns.services)
            )
        )
        socknames = [name for name in self._async_listdir(self.rundir) if name.endswith('.sock')]
        for sockname in socknames:
            uuid = sockname[:-5]
            path = os.path.join(self.rundir, sockname)
            if uuid not in alive_uuids:
                self.log.debug("unlinking unknown liner sock %s", path)
                try:
                    self._async_unlink(path)
                except Exception as e:
                    self.log.warning("failed to remove sock %s: %s", path, e)

    def report_rusage(self):
        for ns in list(self.namespaces.values()):
            ns.report_rusage()

    def start(self):
        with self.update_lock:
            super(ServiceUpdater, self).start()

            self.fsyncer = Fsyncer(self.log.get_child('fsyncer'))

        try:
            self.config.subscribe(self.config_changed, [('skynet', 'skycore')], config_only=True)
        except self.UpdateException as e:
            self.log.error("failed to apply first config: %s" % (e,), exc_info=sys.exc_info())
            self.config.send_report(self.log, False, str(e))
        else:
            self.config.send_report(self.log, True, '')

        # we need to write ctx at least once to create missing statefile or upgrade its format
        self.write_context()

        return self

    def stop(self):
        super(ServiceUpdater, self).stop()
        if self.fsyncer is not None:
            self.fsyncer.stop()
        self.config.unsubscribe(self.config_changed)
        self.write_context()
        self.output_logger_factory.shutdown()
        return self

    def __str__(self):
        return "%s (%s, revision %r)" % (
            self.__class__.__name__,
            "paused" if self.paused else "active",
            self.core_revision,
        )
