import sys
import os
import time
import gevent

try:
    from gevent.coros import RLock
except ImportError:
    from gevent.lock import RLock

import random
from ..framework.utils import detect_hostname
from ..framework.subprocess_gevent import Popen, PIPE
from ..framework.component import Component

from api.srvmngr import getRoot
from api.skycore import ServiceManager
from api.skycore.errors import SkycoreError

HB_REPORT_SEND_DELAY = 30
HB_REPORT_HTTP_TIMEOUT = 20
HB_REPORT_COLLECT_TIMEOUT = 10
HB_REPORT_VALID_TIMEOUT = 1200
HB_FULL_REPORT_MIN_PERIOD = 300
HB_FULL_REPORT_MAX_PERIOD = 900
HB_REST_API_BASE_URL = 'http://api.heartbeat.yandex.net/api/v1.0'
HB_REST_API_SERVER_ID_HDR = 'X-Heartbeat-Server'
HB_COMMAND = '''
from api.heartbeat import client
client.scheduleReport(
{0!r}, {1!r},
incremental=True,
compress=True,
sendDelay={2},
valid={3}
)'''


class StateReporter(Component):
    def __init__(self, parent=None, deblock=None, log=None):
        logname = 'statrep'
        super(StateReporter, self).__init__(logname=logname, parent=parent)

        if log:
            self.log = log.getChild(logname)

        self._deblock = deblock
        self._lock = RLock()
        self._report_available_event = gevent.event.Event()
        self._report = {}
        self._report_type = 'SkyCoreUpdate'

        delay = random.randint(HB_FULL_REPORT_MIN_PERIOD, HB_FULL_REPORT_MAX_PERIOD)
        self.log.info('Next SkyCoreInfo in {} seconds'.format(delay))
        self._next_full_report_time = time.time() + delay

    def send_state_report(self, namespace, service, state, version):
        with self._lock:
            self.log.info('Add new report: {}/{} {} {}'.format(namespace, service, state, version))
            if self._report_type == 'SkyCoreInfo':
                # we have unsent full report, update it
                self._report['services']['{}/{}'.format(namespace, service)] = {
                    'state': state,
                    'version': version
                }
            else:
                self._report['{}/{}'.format(namespace, service)] = {
                    'state': state,
                    'version': version,
                    'ts': time.time()
                }
            self._report_available_event.set()

    @Component.green_loop(logname='reporter')
    def _send_state_report_loop(self, log):
        wait_time = float(self._next_full_report_time - time.time() + 1)
        if wait_time < 0:
            wait_time = None

        if self._report_available_event.wait(wait_time):
            log.debug('Waiting reports for {} seconds...'.format(HB_REPORT_COLLECT_TIMEOUT))
            gevent.sleep(HB_REPORT_COLLECT_TIMEOUT)

        with self._lock:
            if time.time() >= self._next_full_report_time:
                # time to send full report
                self._create_full_report()
                # calculate next report time
                delay = random.randint(HB_FULL_REPORT_MIN_PERIOD, HB_FULL_REPORT_MAX_PERIOD)
                log.info('Next SkyCoreInfo in {} seconds'.format(delay))
                self._next_full_report_time = time.time() + delay

            log.info('Sending report ({}): {}'.format(self._report_type, self._report))
            sent = True
            cmd = HB_COMMAND.format(self._report_type, self._report, HB_REPORT_SEND_DELAY, HB_REPORT_VALID_TIMEOUT)
            _, err = Popen([sys.executable, '-sBtt', '-c', cmd],
                           stdout=PIPE,
                           stderr=PIPE,
                           close_fds=True).communicate()

            if err:
                log.warning('Failed to send report:')
                log.warning(err)

                url = HB_REST_API_BASE_URL + '/report'
                import json
                report = json.dumps({
                    'name': self._report_type,
                    'report': self._report,
                    'source': detect_hostname()
                })

                log.info('Sending the report of %d bytes directly via %r' % (len(report), url))
                err_msg = self._deblock.apply(
                    self._send_state_report_directly,
                    url,
                    report)

                if err_msg:
                    log.warning(err_msg)
                    # failed to send directly
                    sent = False

            if sent:
                # clear data if report has been really sent only
                self._report_available_event.clear()
                self._report = {}
                self._report_type = 'SkyCoreUpdate'
                log.info('Reports processed')

        return 1

    @staticmethod
    def _send_state_report_directly(url, report):
        err = None
        import urllib2
        import json
        try:
            req = urllib2.Request(url, report, {'Content-Type': 'application/json'})
            resp = json.loads(urllib2.urlopen(req, None, HB_REPORT_HTTP_TIMEOUT).read())
            if resp:
                err = 'Server %r respond: %r' % (resp.info().get(HB_REST_API_SERVER_ID_HDR), resp)
        except Exception as ex:
            err = 'Unable to sent a report directly: %s' % str(ex)
            if isinstance(ex, urllib2.HTTPError):
                err += '\nServer %r respond: %s' % (ex.info().get(HB_REST_API_SERVER_ID_HDR), ex.read())

        return err

    def _create_full_report(self):
        services = self._get_skycore_services_info()
        autostart = self._deblock.apply(self._check_skycore_autostart)
        self._report = {
            'services': services,
            'autostart': autostart[0],
            'services_autostart': autostart[1]
        }
        self._report_type = 'SkyCoreInfo'

    @staticmethod
    def _get_skycore_services_info():
        # get state of skycored services
        result = {}
        try:
            manager = ServiceManager()
            for ns in manager.list_namespaces():
                services = manager.list_services(ns)
                if services:
                    try:
                        # try to use new API
                        states = manager.check_services(ns, services, new_format=True)
                        for service, state in states.items():
                            result['{}/{}'.format(ns, service)] = state
                    except SkycoreError:
                        # perhaps there is old skycore API, try to use it
                        states = manager.check_services(ns, services)
                        for service, (state, version) in states.items():
                            result['{}/{}'.format(ns, service)] = {
                                'state': state,
                                'version': version
                            }
        except BaseException:
            # something is wrong, do not add these services
            pass

        return result

    def _check_skycore_autostart(self):
        autostart = False
        autoservices = False
        try:
            autostart_filename = os.path.join(getRoot(), 'skycore/skyctl.lock')
            with open(autostart_filename, 'r') as f:
                for line in f:
                    line = line.strip()
                    if line == 'autostart: enabled':
                        autostart = True
                    if line == 'autoservices: enabled':
                        autoservices = True

        except IOError:
            pass

        return autostart, autoservices
