#!/skynet/python/bin/python

from __future__ import absolute_import, print_function

from . import initialize_skycore

initialize_skycore()  # noqa

import argparse
import os
import signal
import sys

import gevent
from py import std

from .framework.logger import initialize_log, HierarchicalLoggerFactory
from .framework.utils import Path
from .components.daemon import SkycoreDaemon

from .kernel_util.console import setProcTitle


def patch_subprocess():
    # Force kernel.util.sys.user to use green subprocess module
    from .framework import subprocess_gevent

    sys.modules['subprocess'] = subprocess_gevent
    for name, mod in sys.modules.iteritems():
        if name.startswith('skycore.kernel_util.sys.user._'):
            if hasattr(mod, 'subprocess'):
                mod.subprocess = subprocess_gevent


def unpatch_subprocess():
    from .framework import subprocess_gevent

    vanilla_subprocess = subprocess_gevent._subprocess

    sys.modules['subprocess'] = vanilla_subprocess
    for name, mod in sys.modules.iteritems():
        if name.startswith('skycore.kernel_util.sys.user._'):
            if hasattr(mod, 'subprocess'):
                mod.subprocess = vanilla_subprocess


def bind_singletone(log, port):
    try:
        singletone_sock = std.socket.socket(std.socket.AF_INET, std.socket.SOCK_STREAM)
        singletone_sock.setsockopt(std.socket.SOL_SOCKET, std.socket.SO_REUSEADDR, 1)
        singletone_sock.bind(('127.0.0.1', port))
        singletone_sock.listen(1)
        return singletone_sock
    except std.socket.error as err:
        import errno
        if errno.errorcode.get(err.errno, None) == 'EADDRINUSE':
            error_message = 'ERROR: Address 127.0.0.1:%d already in use! Quitting...' % port
            print(error_message, file=sys.stderr)
            log.critical(error_message)
            sys.exit(1)
        raise


def start_dowser(port):
    if not port:
        return

    from pkg_resources import require
    require('pil')
    import cherrypy
    import dowser

    cherrypy.tree.mount(dowser.Root())
    cherrypy.config.update({
        'environment': 'embedded',
        'server.socket_host': '0.0.0.0',
        'server.socket_port': port,
    })

    cherrypy.server.start()
    cherrypy.engine.start()


def start_yappi(mode):
    if mode is None:
        return

    import yappi
    import greenlet

    yappi.set_clock_type('cpu')
    yappi.set_context_id_callback(
        lambda: greenlet and id(greenlet.getcurrent() or 0)
    )
    yappi.set_context_name_callback(
        lambda: greenlet and type(greenlet.getcurrent()).__name__ or ''
    )
    yappi.start(True, True)


def get_app_path():
    # Determine absolute app path
    fpath = Path(__file__)

    for part in reversed(fpath.parts()):
        # /skynet/skycore/lib/skycore
        if part.basename == 'skycore':
            skycore_lib = part.dirpath()

            # /skynet/skycore/lib
            if skycore_lib.basename == 'lib':
                # /skynet/skycore
                possible_app_path = skycore_lib.dirpath()

                # /skynet/skycore/bin
                possible_bin_path = possible_app_path.join('bin')

                if possible_bin_path.check(dir=1, exists=1):
                    # /skynet/skycore/bin/skycore
                    if possible_bin_path.join('skycore').check(exists=1, file=1):
                        # use /skynet/skycore
                        return possible_app_path


def main():
    parser = argparse.ArgumentParser()
    parser.add_argument(
        '-d', '--workdir', required=False, default=os.path.abspath('var'),
        help='Working directory'
    )
    parser.add_argument(
        '--skynetdir',
        help='Skynet root directory'
    )
    parser.add_argument(
        '--supervisordir',
        help="Skynet supervisor directory"
    )
    parser.add_argument(
        '-p', '--singletone_port', type=int, required=True,
        help='Singleton port'
    )
    parser.add_argument('-u', '--user', required=False)
    parser.add_argument(
        '-r', '--revision', type=str, required=True,
        help='Skycore revision',
    )
    parser.add_argument(
        '--oneshot', action='store_true', default=False,
        help='just update config and services and exit'
    )
    parser.add_argument(
        '--restart-all-services', dest='restart_all', action='store_true', default=False,
        help='restart all existing services'
    )
    parser.add_argument(
        '--skydeps-changed', dest='skydeps_changed', action='store_true', default=False,
        help='restart all skydeps dependent services'
    )
    parser.add_argument(
        '--freeze-updates-i-know-what-i-am-doing', action='store_true', default=False,
        help='you should not use this option',
    )
    parser.add_argument(
        '--dowser', default=0, type=int, required=False
    )
    parser.add_argument(
        '--yappi-summary', dest='yappi', default=None, action='store_true', required=False,
    )
    parser.add_argument(
        '--yappi-period', dest='yappi', type=int, required=False,
    )

    args = parser.parse_args()

    if (
        not args.freeze_updates_i_know_what_i_am_doing
        and Path(args.workdir).join('freeze_updates_SKYDEV-1556').check(exists=1, file=1)
    ):
        args.freeze_updates_i_know_what_i_am_doing = True

    start_dowser(args.dowser)

    if args.user:
        from .kernel_util.sys.user import userPrivileges as user_privileges  # noqa
    else:
        import contextlib

        @contextlib.contextmanager
        def user_privileges(*args, **kwargs):
            yield

    patch_subprocess()

    curnofile = std.resource.getrlimit(std.resource.RLIMIT_NOFILE)

    if curnofile[1] < 4096:
        # If hard < 4096 -- forcibly set soft/hard to 4096
        # FIXME (torkve) we will die here if hard<4096 and we are in container (no CAP_SYS_RESOURCE),
        #                good news is that we mightn't have <4096 in container by default.
        limits = {std.resource.RLIMIT_NOFILE: (4096, 4096)}
    else:
        # If hard > 4096 -- forcibly set only soft to 4096
        # This is very important part -- in porto containers even root can NOT
        # raise up limits. Thus if we set hard limit to 4096 here, any other process will
        # not be able to set it bigger than 4096.
        #
        # Thus, we set only soft limit here.
        limits = {std.resource.RLIMIT_NOFILE: (4096, curnofile[1])}

    app_path = get_app_path()

    with user_privileges(user=args.user, limit=False, minimalLimits=limits):
        privileged_main(args, app_path, limits)

    os.killpg(0, signal.SIGKILL)
    return 0


def privileged_main(args, app_path, limits):
    unpatch_subprocess()

    setProcTitle('skycore')

    start_yappi(args.yappi)

    # Dont rely on umask we have, because we will play with
    # chmodding files.
    os.umask(0o22)

    for limit, value in limits.items():
        std.resource.setrlimit(limit, value)

    # subprocess.MAXFD is set to softlimit on process start by default,
    # and it is used as upper limit in os.closerange() on forks,
    # so we tune it down to save a bit of time and cpu
    sys.modules['subprocess'].MAXFD = std.resource.getrlimit(std.resource.RLIMIT_NOFILE)[0]

    write_stderr_log = not os.getenv('SKYNET_LOG_STDOUT', False)

    # from library import config
    # log_cfg = config.query('skynet.skycore.config', 'service').get('logging', {})
    # max_bytes = log_cfg.get('max_bytes', 10 * 1024 * 1024)
    # backup_count = log_cfg.get('backup_count', 5)

    logdir = Path(args.workdir).join("log")
    logdir.ensure(dir=True)
    log = initialize_log(logdir, 'skycore', 20 * 1024 * 1024, 14)
    initialize_log(logdir, 'skycore-rpc', 20 * 1024 * 1024, 14, 'rpc')
    log = log.getChild('main')

    if not app_path:
        log.critical('Unable to detect app path, exiting!')
        sys.stderr.write('Unable to detect app path, exiting!\n')
        return 1

    logger_factory = HierarchicalLoggerFactory(logdir, 'out')

    singletone_sock = bind_singletone(log, args.singletone_port)  # noqa (keep until exit)

    try:
        for pidfile_name in (os.path.join(args.workdir, 'skycore.pid'), '/run/skycore.pid'):
            if os.path.exists(os.path.dirname(pidfile_name)):
                with open(pidfile_name, 'w') as pidfile:
                    pidfile.write(str(os.getpid()))
                    pidfile.flush()
                    os.fsync(pidfile.fileno())

        if write_stderr_log:
            stderr_log = logdir.join('skycore-stderr.log').open('wb')
            os.dup2(stderr_log.fileno(), 2)
            stderr_log.close()

        restart = None
        if args.restart_all:
            restart = 'ALL'
        elif args.skydeps_changed:
            restart = 'SKYDEPS'

        daemon = SkycoreDaemon(
            app_path=app_path,
            workdir=args.workdir,
            revision=args.revision,
            output_logger_factory=logger_factory,
            skynetdir=args.skynetdir,
            supervisordir=args.supervisordir,
            yappi_mode=args.yappi,
            restart_all=restart,
            paused=args.freeze_updates_i_know_what_i_am_doing,
        )
        if not args.oneshot:
            daemon.start()
        else:
            gevent.spawn(daemon._iterate_mailbox)  # FIXME dangerous, we break component by manual spawn
            try:
                daemon.service_updater.check_all_namespaces(log, 'requested by run-mode', start_on_install=False)
            finally:
                daemon.service_updater.write_context()
            log.debug("oneshot init succeeded")

    except KeyboardInterrupt:
        log.debug('interrupted!')
        print('interrupted!', file=sys.stderr)
        sys.stderr.flush()
        os.killpg(0, signal.SIGKILL)
        os._exit(1)
    except Exception as e:
        import traceback
        message = 'Unhandled daemon exception: %s--\nQuit immediately!' % traceback.format_exc(e)
        log.debug(message)
        print(message, file=sys.stderr)
        sys.stderr.flush()

        gevent.sleep(1)

        os.killpg(0, signal.SIGKILL)
        os._exit(1)
    else:
        if args.dowser:
            import cherrypy
            cherrypy.engine.stop()


if __name__ == '__main__':
    # Run skycore/bin/skycore instead of this!
    sys.exit(main())
