import os
import sys
import bz2
import gzip
import errno
from cStringIO import StringIO
from collections import defaultdict

from . import tartools as tarfile
from .framework.utils import Path, human_size
from .framework.version import LooseVersion


def is_supported_extension(filename):
    return any(filename.endswith(extension) for extension in ('.tgz', '.tar.gz', '.tbz2', '.tar.bz2', '.tar'))


def unpack_archive_fileobj(filename):
    if filename.endswith('.tgz') or filename.endswith('.tar.gz'):
        fn = gzip.open

    elif filename.endswith('.tbz2') or filename.endswith('.tar.bz2'):
        fn = bz2.open

    else:
        fn = open

    with fn(filename, 'rb') as f:
        return StringIO(f.read())


def cleanup_downloads_junk(log, root):
    for child in os.listdir(root):
        item = os.path.join(root, child)
        if (
            not os.path.isfile(item)
            and not os.path.islink(item)
            or not is_supported_extension(child)
        ):
            log.debug("will remove junk file: %r", item)
            Path(item).remove(ignore_errors=True)


def get_removal_candidates(log, root, stat_cache):
    paths = os.listdir(root)
    stats = []
    services = defaultdict(list)

    for path in paths:
        fullpath = os.path.join(root, path)
        svcs_in_archive = []
        try:
            st = os.stat(fullpath)
        except EnvironmentError as e:
            if e.errno != errno.ENOENT:
                log.warning("cleanup: failed to stat %r, you need to investigate it manually: %s", fullpath, e)
            continue

        if fullpath in stat_cache and stat_cache[fullpath][0] >= st.st_mtime:
            for svc in stat_cache[fullpath][1]:
                services[svc].append((LooseVersion(path), st, fullpath))
        else:
            if not os.path.isfile(fullpath) or not is_supported_extension(fullpath):
                continue

            try:
                with tarfile.open(fullpath, 'r|*', log=log.getChild('extract')) as arc:
                    for name in arc.getnames():
                        name = os.path.normpath(name)
                        if '/' in name or not name.endswith('.scsd'):
                            continue
                        svcs_in_archive.append(os.path.splitext(name)[0])
            except Exception as e:
                # log.warning("cleanup: removing broken archive %r: %s", fullpath, e)
                stats.append((sys.maxint, st, fullpath))
                continue

            for svc in svcs_in_archive:
                services[svc].append((LooseVersion(path), st, fullpath))

            stat_cache[fullpath] = (st.st_mtime, list(svcs_in_archive))

    for svc in services:
        services[svc].sort(key=lambda x: x[0], reverse=True)

        stats.extend((number, stat2, fullpath)
                     for (number, (version, stat2, fullpath))
                     in enumerate(services[svc]))

    stats.sort(key=lambda x: (-x[0],  # starting from the Nth of a kind
                              -x[1].st_size,  # then by size, most fat first
                              x[1].st_mtime,  # then by mtime, ye oldest first
                              ),
               reverse=True,
               )

    return stats


def prepare_downloaddir(log, downloaddir, required, stat_cache):
    cleanup_downloads_junk(log, downloaddir)

    st = os.statvfs(downloaddir)

    free_space = st.f_bsize * st.f_bavail
    total_space = st.f_frsize * st.f_blocks

    cache_limit = 1000 * 1024 * 1024  # 1GB
    cache_threshold = 100 * 1024 * 1024  # 100MB
    downloads_size = sum(os.stat(os.path.join(downloaddir, item)).st_size for item in os.listdir(downloaddir))
    if free_space + downloads_size < required:
        raise Exception("Cannot download, there's only %s free space, %s used in cache, and %s required" % (
            human_size(free_space),
            human_size(downloads_size),
            human_size(required)
        ))

    candidates = get_removal_candidates(log, downloaddir, stat_cache)

    while candidates:
        copy_number, path_stat, path = candidates.pop()
        if downloads_size > cache_threshold and free_space * 2 < total_space:
            log.debug("download cache triggered by free space amount (dl_size=%s, free_space=%s, total_space=%s)",
                      human_size(downloads_size),
                      human_size(free_space),
                      human_size(total_space)
                      )
        elif downloads_size > cache_limit:
            log.debug("download cache triggered by cache_limit (dl_size=%s, cache_limit=%s)",
                      human_size(downloads_size),
                      human_size(cache_limit)
                      )
        elif copy_number > 3:
            log.debug("download cache triggered by too many versions service of service (count=%s)", copy_number)
        else:
            break

        log.debug("will remove %s", path)
        try:
            os.unlink(path)
        except EnvironmentError as e:
            log.exception("failed to remove %r" % (e,), exc_info=sys.exc_info())
        else:
            downloads_size -= path_stat.st_size
            free_space += path_stat.st_size
            stat_cache.pop(path, None)
