import sys
import os
import errno
import random
import shutil
import signal
import socket
import subprocess
import urllib
from collections import OrderedDict

import gevent
import six

from py.path import local as Path  # noqa
from .kernel_util import logging
from .framework.subprocess_gevent import Popen
from .procs.subproc import Subprocess, ProcStartException
from .procs import waitpid

urllib_parse = six.moves.urllib_parse
CYGWIN = (sys.platform == 'cygwin')


SKY_GET_TIMEOUT = 15 * 60
RSYNC_GET_TIMEOUT = 15 * 60
HTTP_GET_TIMEOUT = 15 * 60
SKYNETBIN_RUN_TIMEOUT = 20 * 60
SKYNETBIN_RESHARE_TIMEOUT = 2 * 60

if CYGWIN:
    BSD = True
else:
    BSD = os.uname()[0].startswith('Darwin')

log = logging.getLogger('downldr')


class DownloadSubprocLoggerAdapter(logging.LoggerAdapter):
    def process(self, msg, kwargs):
        msg = '[%-5s]  %s' % (
            self.extra.get('type', '?????'),
            msg
        )
        return msg, kwargs


def is_we_fastbonized():
    host = socket.getfqdn()

    # First try if we have hostname.fb.yandex.ru
    try:
        if host.endswith('.yandex.ru'):
            test_name = host[:-len('.yandex.ru')] + '.fb.yandex.ru'
        else:
            test_name = host + '.fb.yandex.ru'

        ip = socket.getaddrinfo(test_name, 0)[0][4][0]
        if socket.gethostbyaddr(ip)[0] != 'any.yandex.ru':
            log.info('Fastbone mode: %s', test_name)
            return True
        else:
            log.warning('%s -> %s -> any.yandex.ru', test_name, ip)
    except Exception:
        pass

    # Next try if we have fb-hostname
    try:
        test_name = 'fb-' + host
        ip = socket.getaddrinfo(test_name, 0)[0][4][0]
        if socket.gethostbyaddr(ip)[0] != 'any.yandex.ru':
            log.info('Fastbone mode: %s', test_name)
            return True
        else:
            log.warning('%s -> %s -> any.yandex.ru', test_name, ip)
    except Exception:
        pass

    log.warning('Backbone mode!')
    return False


def parse_advertised_release(config):
    try:
        svn_url = config['attributes'].get('svn_url')
        attributes = config['attributes']
        size = config['size']

        try:
            filename = os.path.basename(config['file_name'])
            urls = OrderedDict((
                ('skynet', config['skynet_id']),
                ('http', {100: config['http']['links']}),
                ('rsync', config['rsync']['links']),  # deprioritize rsync
            ))

            for key, value in attributes.items():
                if key.startswith('http_') and 'proxy.sandbox.yandex-team.ru' not in value:
                    if key.endswith('_weight'):
                        continue
                    weight = int(attributes.get(key + '_weight', 5))
                    urls['http'].setdefault(weight, []).append(value)

            for key, value in urls.items():
                if isinstance(value, str):
                    urls[key] = (value, )
                elif isinstance(value, list):
                    random.shuffle(value)
                    urls[key] = tuple(value)
                elif isinstance(value, dict):
                    sorted_by_weight_and_shuffled = []
                    for weight in sorted(value, reverse=True):
                        urls_copied = list(value[weight][:])
                        random.shuffle(urls_copied)
                        sorted_by_weight_and_shuffled.extend(urls_copied)
                    urls[key] = tuple(sorted_by_weight_and_shuffled)

            md5 = config['md5']
        except KeyError as ex:
            raise KeyError('config:%s' % (ex.args[0], ))

    except KeyError as ex:
        raise ValueError('Fail to analyze genisys packet: missing %r key' % ex.args[0])

    version = '[{}] {} ({})'.format(config['md5'][:8], config['description'], config['id'])

    # Parse http and rsync urls
    for key, urllist in urls.items():
        if key in ('rsync', 'http'):
            urls[key] = tuple(urllib_parse.urlparse(url) for url in urllist)

    return {
        'svn_url': svn_url,
        'version': version,
        'urls': urls,
        'md5': md5,
        'filename': filename,
        'size': size
    }


def system(cmd, timeout, **env):
    logger = DownloadSubprocLoggerAdapter(log, {'type': 'proc'})

    try:
        proc_env = {'PATH': os.getenv('PATH')}
        proc_env.update(env)

        proc = Subprocess(
            log=logger,
            out_log=logger,
            uuid='',
            args=cmd,
            cwd='/',
            env=proc_env,
        )
    except ProcStartException as e:
        waitpid.pidwaiter().wait(e.pid)
        raise

    logger.debug('Will wait for finish %d mins', timeout / 60)

    if proc.wait(timeout=timeout):
        ret = proc.exitstatus
        return ret['exitstatus'] if not ret['signaled'] else -ret['termsig']

    logger.warning('Timed out, killing with SIGKILL...')

    proc.send_signal(signal.SIGKILL)
    gevent.sleep(1)

    if not proc.wait(0):
        proc.send_signal(signal.SIGKILL)

    return -9


def validate_binary(binary, md5):
    if binary.check(exists=1) or binary.check(link=1):
        if binary.check(file=1):
            if binary.computehash() == md5:
                return True
        else:
            binary.remove()
    return False


def move_binary(binary_tmp, binary):
    if binary.check(exists=1):
        binary.remove()
    binary_tmp.move(binary)


def download_via_skynet(binary, resid, net, deblock, skybone_available):
    if net == 'backbone':
        speedlim = ['--max-dl-speed', '8mbps', '--max-ul-speed', '8mbps']
    else:
        speedlim = []

    cmd = [
        '/skynet/tools/sky',
        'get', '-w', '-p', '-u',
        '--progress-format', 'json', '--progress-report-freq', '500',
        '-P', 'High',
        '-N', 'Backbone' if net == 'backbone' else 'Fastbone',
        '-d', binary.dirpath().strpath,
    ] + speedlim + [
        resid,
    ]

    rc = system(cmd, SKY_GET_TIMEOUT)
    if rc == 0:
        return binary

    if rc:
        log.warning('')
        log.warning('sky get failed with exit code %d', rc)
    return None


def download_via_rsync(binary, url, _, deblock, skybone_available):
    rsync = Path.sysfind('rsync')
    if rsync is None:
        log.warning("'rsync' tool is not available, download via rsync failed")
        return None

    proc = Popen([rsync.strpath, '--help'], stdout=subprocess.PIPE, stderr=subprocess.PIPE)
    stdout, _ = proc.communicate()
    proc.wait()

    append_verify = []
    contimeout = []
    if proc.returncode == 0:
        if b'--append-verify' in stdout:
            append_verify = ['--append-verify']
        if b'--contimeout' in stdout:
            contimeout = ['--contimeout=10']

    binary_tmp = binary.dirpath().join(binary.basename + '.rsync-download')
    cmd = [
        rsync.strpath,
        '--append'
    ] + append_verify + [
        '--checksum',
        '--inplace',
        '--copy-links',
    ] + contimeout + [
        '--timeout=60',
        '--temp-dir=/var/tmp',
        '--progress',
        '--bwlimit=1000',  # 1mb/s
        url, binary_tmp.strpath
    ]

    rc = system(cmd, RSYNC_GET_TIMEOUT)
    if rc == 0:
        deblock.apply(move_binary, binary_tmp, binary)
        if skybone_available:
            reshare(binary)
        return binary
    if rc:
        log.warning('')
        log.warning('%r failed with exit code %d', cmd[0], rc)
    return None


def download_via_http(binary, url, _, deblock, skybone_available):
    binary_tmp = binary.dirpath().join(binary.basename + '.http-download')
    if 'storage-int.mds.yandex.net/get-skynet' in url:
        auth = ['--header', 'Authorization: Basic c2t5bmV0OjNhMjEzODMzYjJlMzY1ZDZiNzQwYjNlMTY0NzVkMWU1']
        fetch_env = {'HTTP_AUTH': "basic:*:skynet:3a213833b2e365d6b740b3e16475d1e5"}
    else:
        auth = []
        fetch_env = {}

    wget = Path.sysfind('wget')
    env = {}

    if wget is None:
        log.info("'wget' tool is not available. Using 'fetch' instead.")

        fetch = Path.sysfind('fetch')
        if fetch is None:
            log.warning("'fetch' tool is not available, download via HTTP failed")
            return None

        cmd = [fetch.strpath, '-r', '-o', binary_tmp.strpath, '-T', '30', '-a', url]
        env = fetch_env

    else:
        cmd = [
            wget.strpath,
            '-c', '-O', binary_tmp.strpath,
            '--progress=dot:mega',
            '--connect-timeout=10',
            '--read-timeout=30',
            '--retry-connrefused',
            '--tries=5',
            '--limit-rate=1000000',  # 1mb/s
        ] + auth + [
            url
        ]

    rc = system(cmd, HTTP_GET_TIMEOUT, **env)
    if rc == 0:
        deblock.apply(move_binary, binary_tmp, binary)
        if skybone_available:
            reshare(binary)
        return binary
    else:
        log.warning('')
        log.warning('%r failed with exit code %d', cmd[0], rc)
    return None


def fix_parsed_url(url, net):
    # Extract pure hostname from url
    components = list(url)
    # FIXME six.moves.urllib_parse should be used here,
    # but current skynet 'six' version doesn't support it yet
    user, host = urllib.splituser(components[1])
    host, port = urllib.splitport(host)

    if net == 'fastbone':
        if host.endswith('.yandex.ru'):
            host = host[:-len('.yandex.ru')]
        host = host + '.fb.yandex.ru'
    elif net == 'fastbone2':
        host = 'fb-' + host

    # Now put components with modified hostname back
    if port is not None:
        host = '%s:%s' % (host, port)
    if user is not None:
        host = '%s@%s' % (user, host)

    components[1] = host
    return urllib_parse.urlunparse(components)


def download(dest, filename, urls, md5, deblock, skybone_available=True):
    log.info('Attemping to download skynet binary...')
    for net, links in six.iteritems(urls):
        for link in links:
            log.debug(
                '   %s candidate: %s',
                net,
                (
                    fix_parsed_url(link, net)
                    if isinstance(link, urllib_parse.ParseResult)
                    else link
                )
            )

    dirpath = Path(dest)
    binary = dirpath.join(filename)
    deblock.apply(dirpath.ensure, dir=1)

    # Check for free space before attemping to download
    try:
        binary_parent_free = deblock.apply(shutil.disk_usage, dest).free
    except AttributeError:
        # Does not work on CYGWIN
        pass
    except OSError as ex:
        if ex.errno == 75:
            # too much blocks
            pass
        else:
            raise
    else:
        if binary_parent_free < 300 << 20:
            raise OSError(
                errno.ENOSPC,
                'There is not enough free space in "%s": only %dMiB free (at least 300MiB required).' % (
                    dirpath, binary_parent_free >> 20
                ),
                dirpath
            )

    if deblock.apply(validate_binary, binary, md5):
        log.info('binary already downloaded and md5 looks sane')
        return binary

    if is_we_fastbonized():
        possible_networks = ('fastbone', 'fastbone2', 'backbone')
    else:
        possible_networks = ('backbone', )

    possible_tries = 0
    for key, urllist in urls.items():
        for url in urllist:
            if key == 'skynet':
                # skynet transport does not support 'fastbone2' mode
                nets = set(possible_networks)
                nets.discard('fastbone2')
                possible_tries += len(nets)
            else:
                possible_tries += len(possible_networks)

    global_tryout = 0
    for tryout, net in enumerate(possible_networks):
        for key, urllist in urls.items():
            if key == 'skynet' and not skybone_available:
                log.warning("skip urls %r: no skybone available", urllist)
                continue

            func = globals().get('download_via_%s' % (key, ), None)
            if func:
                for url in urllist:
                    if isinstance(url, urllib_parse.ParseResult):
                        url = fix_parsed_url(url, net)
                    elif url.startswith('rbtorrent'):
                        # For rbtorrent urls dont try 'fastbone2' mode
                        if net in ('fastbone2', ):
                            continue
                    try:
                        global_tryout += 1
                        log.info('-' * 80)
                        log.info(
                            '[%d/%d] Trying to download with %r from %r',
                            global_tryout, possible_tries, func, url
                        )
                        result = func(binary, url, net, deblock, skybone_available=skybone_available)
                        log.info('-' * 80)
                    except Exception as ex:
                        log.info('-' * 80)
                        log.warning('failed with unhandled exception: %s', ex)
                        result = None

                    if result:
                        if deblock.apply(result.computehash) != md5:
                            log.warning("md5 hash doesn't match, probably need to redownload")
                        else:
                            return result
            else:
                log.warning('skip url type "%s": dont know how to download it', key)


def reshare(path):
    cmd = [
        '/skynet/tools/sky',
        'share',
        '-d', path.dirpath().strpath,
        path.basename
    ]
    return system(cmd, SKYNETBIN_RESHARE_TIMEOUT)
