import collections
import functools
import weakref
import inspect
import logging
import traceback
import types

import gevent

from .logger import adapt_logger, SmartLoggerAdapter


class ComponentLoggerAdapter(SmartLoggerAdapter):
    def process(self, msg, kwargs):
        return self.extra.get('prefix', '') + msg, kwargs


class ComponentLoopLoggerAdapter(ComponentLoggerAdapter):
    def process(self, msg, kwargs):
        return '[%s]  %s' % (self.extra.get('loop_name', ''), msg), kwargs


class Component(object):
    __default = object()

    class WakeUp(BaseException):
        pass

    def __init__(
        self,
        parent=None,
        logname=__default,
        log_msg_prefix=__default
    ):
        self.loops = collections.OrderedDict()
        self.childs = []
        self.parent = weakref.ref(parent) if parent is not None else None

        if parent:
            parent.add_child(self)

        self._logname = (
            self.__class__.__name__
            if logname is self.__default
            else logname
        )
        self._log_msg_prefix = '' if log_msg_prefix is self.__default else log_msg_prefix
        self._make_logger()
        self._init_loops()

    def _make_logger(self):
        parent = self.parent and self.parent()
        if not parent:
            base_logger = logging.getLogger('')
            base_logger.name = 'main'
            base_logger = adapt_logger(base_logger)
        else:
            base_logger = parent.log

        if self._logname:
            base_logger = base_logger.getChild(self._logname)

        if self._log_msg_prefix:
            prefix = '[%s]  ' % (self._log_msg_prefix, )
        else:
            prefix = ''

        self.log = ComponentLoggerAdapter(
            base_logger,
            {'prefix': prefix}
        )

    def _init_loops(self):
        for key in dir(self):
            if key.startswith('__'):
                continue
            try:
                meth = getattr(self, key)
            except AttributeError:
                # ignore complex properties
                continue
            if getattr(meth, 'is_a_loop', False) is True:
                if meth not in self.loops:
                    self.loops[meth] = None

    def start(self):
        for meth, loop in self.loops.iteritems():
            assert loop is None, '%r %r already running' % (meth, loop)
            self.loops[meth] = gevent.spawn(meth, **meth.extra)
            self.log.debug("Started loop %r %r", meth, self.loops[meth])

        for child in self.childs:
            child.start()

        return self

    def stop(self):
        for child in self.childs:
            child.stop().join()

        for _, loop in self.loops.iteritems():
            if loop is not None and not loop.ready():
                loop.kill(gevent.GreenletExit)

        return self

    def join(self):
        if self.childs:
            joiners = []
            for child in self.childs:
                joiners.append(gevent.spawn(child.join))
            joiners.extend(filter(bool, self.loops.values()))
            gevent.joinall(joiners)
        else:
            gevent.joinall(filter(bool, self.loops.values()))
        return self

    def add_child(self, component):
        assert isinstance(component, Component)
        self.childs.append(component)

    def add_loop(self, meth, **kwargs):
        loop = type(self).green_loop(meth, **kwargs)
        self.loops[loop] = None
        return loop

    @classmethod
    def green_loop(cls, meth=None, logname=None, **kwargs):
        if meth is None:
            return functools.partial(cls.green_loop, logname=logname, **kwargs)

        ism = isinstance(meth, types.MethodType)

        @functools.wraps(meth)
        def _loop(self, *args, **kwargs):
            margs, _, _, _ = inspect.getargspec(meth)

            prefix = logname or meth.__name__
            if self._log_msg_prefix:
                prefix = '%s, %s' % (self._log_msg_prefix, prefix)

            log = ComponentLoggerAdapter(self.log, {'prefix': '[oo][%s]: ' % (prefix, )})
            log.debug('Started')

            if 'log' in margs:
                kwargs['log'] = log

            pass_wait = 'waited' in margs
            pass_waking = 'waking' in margs
            wait = 0
            waking = False

            waking_name = '_loop__%s__waking' % (meth.__name__,)
            inited_name = '_loop__%s__inited' % (meth.__name__,)

            setattr(self, inited_name, True)

            while True:
                try:
                    try:
                        if pass_wait:
                            kwargs['waited'] = wait

                        if pass_waking:
                            kwargs['waking'] = waking
                        waking = False

                        setattr(self, waking_name, False)

                        wait = meth(*args, **kwargs) if ism else meth(self, *args, **kwargs)
                    except self.WakeUp:
                        log.info('Woke up (not handled by loop directly)')
                        raise
                    gevent.sleep(wait or 0)
                except gevent.GreenletExit:
                    log.info('Caught GreenletExit, exiting...')
                    return
                except self.WakeUp:
                    waking = True
                    continue
                except BaseException:
                    log.error('Unhandled exception in %r (%r)' % (meth, logname))
                    log.error(traceback.format_exc())
                    try:
                        gevent.sleep(1)  # avoid busy loops
                    except self.WakeUp:
                        waking = True
                        continue
                    wait = 0

        _loop = cls.make_loop(_loop, kwargs)

        if ism:
            _loop = types.MethodType(_loop, meth.__self__)

        return _loop

    @classmethod
    def make_loop(cls, meth, extra):
        class _LoopInternal(object):
            def __get__(self, instance, owner):
                loop_name = '_loop__%s' % (meth.__name__)
                loop = getattr(instance, loop_name, None)
                if loop is not None:
                    return loop

                waking_name = '_loop__%s__waking' % (meth.__name__,)
                inited_name = '_loop__%s__inited' % (meth.__name__,)

                @functools.wraps(meth)
                def _meth(*args, **kwargs):
                    return meth(instance, *args, **kwargs)

                def _wake_up():
                    old_val = getattr(instance, waking_name, False)
                    if old_val:
                        return

                    inited = getattr(instance, inited_name, False)
                    grn = instance.loops.get(_meth)
                    if not grn or not inited:
                        return

                    grn.kill(cls.WakeUp, block=False)
                    setattr(instance, waking_name, True)

                _meth.is_a_loop = True
                _meth.extra = extra
                _meth.wakeup = _wake_up

                setattr(instance, loop_name, _meth)
                setattr(instance, inited_name, False)

                return _meth

        return _LoopInternal()
