from __future__ import absolute_import

from functools import wraps

import gevent
from gevent.greenlet import Greenlet as GeventGreenlet

from ..kernel_util.errors import formatException, saveTraceback


class Greenlet(GeventGreenlet):
    def __init__(self, *args, **kwargs):
        GeventGreenlet.__init__(self, *args, **kwargs)
        if getattr(self, '_run', None) is not None:
            self._run = self._safe_run(self._run)

        try:
            self.slocal = gevent.getcurrent().slocal
        except AttributeError:
            self.slocal = {}

    def _safe_run(self, func):
        @wraps(func)
        def _wrapper(*args, **kwargs):
            try:
                return func(*args, **kwargs)
            except BaseException as err:
                # Dont log some types of errors
                if isinstance(err, (
                    gevent.GreenletExit,
                )):
                    raise

                if not getattr(err, '_logged', False):
                    import logging
                    log = logging.getLogger('').getChild('greenlet')
                    log.critical('Unhandled error during coroutine run: %s' % formatException())
                    err._logged = True

                raise
        return _wrapper

    def _report_error(self, exc_info):
        exception = exc_info[1]
        if isinstance(exception, gevent.GreenletExit):
            self._report_result(exception)
            return

        saveTraceback(exception, blocktitle='GreenLet._report_error()', excInfo=exc_info)
        self._exception = exception

        if self._links and self._notifier is None:
            self._notifier = gevent.core.active_event(self._notify_links)
