from __future__ import absolute_import, division

import os
import time
import stat as osstat
import shutil
import socket
import hashlib

import gevent
import gevent.socket

import py
from py import std
import six

from ..kernel_util.sys.gettime import monoTime


class Path(py.path.local):
    strpath = None

    def ensure(self, *args, **kwargs):
        if kwargs.get('nolink', 0):
            p = self.join(*args)
            if p.check(link=1):
                p.remove()

        try:
            return super(Path, self).ensure(*args, **kwargs)
        except (py.error.EEXIST, py.error.EISDIR):
            if kwargs.get('force', 0):
                self.join(*args).remove()
                return super(Path, self).ensure(*args, **kwargs)
            else:
                raise

    def join(self, *args, **kwargs):
        return super(Path, self).join(*args, **kwargs)

    @classmethod
    def sysfind(cls, *args, **kwargs):
        return py.path.local.sysfind(*args, **kwargs)

    def chmod(self, *args, **kwargs):
        return super(Path, self).chmod(*args, **kwargs)

    def ensure_perms(self, uid, gid, chmod, mask=None):
        stat = self.lstat()
        errors = []

        # avoid second stat() call
        islink = osstat.S_ISLNK(stat.mode)

        if not islink:
            if isinstance(chmod, (list, tuple)):
                chmod_x, chmod_n = chmod
            else:
                chmod_x, chmod_n = (chmod, chmod)

            if mask is not None:
                chmod_x = chmod_x - (chmod_x & mask)
                chmod_n = chmod_n - (chmod_n & mask)

        if stat.uid != uid or stat.gid != gid:
            try:
                if islink:
                    os.lchown(self.strpath, uid, gid)
                else:
                    self.chown(uid, gid)
            except Exception as err:
                errors.append(
                    '%s has invalid owner/group %d:%d, cant set to %d:%d: %s' % (
                        self, stat.uid, stat.gid, uid, gid, err
                    )
                )

        if not islink:
            if stat.mode & 0o100:
                chmod = chmod_x
            else:
                chmod = chmod_n

            if (stat.mode & 0o777) ^ chmod != 0:
                try:
                    self.chmod(chmod)
                except Exception as err:
                    errors.append('%s has invalid mode %04o, cant set to %04o: %s' % (stat.mode, chmod, err))

        if errors:
            raise Exception('\n'.join(errors))

        return self

    def hash_contents(self, paths):
        md5 = hashlib.md5()
        for name in paths:
            path = self.join(name)
            if not path.check(exists=1):
                continue
            stat = path.stat()
            islink = os.path.islink(path.strpath)

            if not islink and not stat.isfile() and not stat.isdir():
                continue

            md5.update(name)
            if islink:
                md5.update(path.readlink())
            elif not stat.isdir():
                with open(path.strpath) as f:
                    while True:
                        data = f.read(1 << 14)  # 16KB
                        if not data:
                            break
                        md5.update(data)
            if not islink:
                mode = str(stat.mode & 0o7777)
                mtime = str(int(stat.mtime))
                md5.update(mode)
                md5.update(mtime)

        return md5.hexdigest()


def getaddrinfo(*args, **kwargs):
    this = getaddrinfo

    meth = kwargs.pop('getaddrinfo', std.socket.getaddrinfo)

    ips = meth(*args, **kwargs)
    anyips, gentime = getattr(this, 'anyips', (None, 0))
    now = std.time.time()

    if now - gentime > 3600:
        try:
            anyips = set([ip[4][0] for ip in meth('any.yandex.ru', 0)])
        except std.socket.gaierror as ex:
            if ex.errno in (
                getattr(std.socket, 'EAI_NONAME', None),
                getattr(std.socket, 'EAI_NODATA', None),
            ):
                # any.yandex.ru was not found
                anyips = set()
            else:
                raise
        this.anyips = (anyips, now)

    for ipinfo in ips:
        ip = ipinfo[4][0]
        if ip in anyips:
            raise std.socket.gaierror(std.socket.EAI_NONAME, 'Name or service not known (resolved to any.yandex.ru)')

    return ips


def getaddrinfo_g(*args, **kwargs):
    try:
        rpipe, wpipe = std.os.pipe()

        result_q = []

        def _runner(*args, **kwargs):
            try:
                result_q.append((True, getaddrinfo(*args, **kwargs)))
            except BaseException:
                result_q.append((False, std.sys.exc_info()))
            finally:
                gevent.socket.wait_write(wpipe, 300.0)
                std.os.write(wpipe, '1')

        thr = std.threading.Thread(target=_runner, args=args, kwargs=kwargs, name='getaddrinfo')
        thr.daemon = True
        thr.start()

        gevent.socket.wait_read(rpipe, 300.0)
        assert std.os.read(rpipe, 1) == '1'

        complete, result = result_q[0]

        if complete:
            return result
        else:
            six.reraise(result[0], result[1], result[2])
    finally:
        std.os.close(rpipe)
        std.os.close(wpipe)


def gethostbyaddr_g(*args, **kwargs):
    meth = kwargs.pop('gethostbyaddr', std.socket.gethostbyaddr)

    try:
        rpipe, wpipe = std.os.pipe()

        result_q = []

        def _runner(*args, **kwargs):
            try:
                result_q.append((True, meth(*args, **kwargs)))
            except BaseException:
                result_q.append((False, std.sys.exc_info()))
            finally:
                gevent.socket.wait_write(wpipe, 300.0)
                std.os.write(wpipe, '1')

        thr = std.threading.Thread(target=_runner, args=args, kwargs=kwargs, name='gethostbyaddr')
        thr.daemon = True
        thr.start()

        gevent.socket.wait_read(rpipe, 300.0)
        assert std.os.read(rpipe, 1) == '1'

        complete, result = result_q[0]

        if complete:
            return result
        else:
            six.reraise(result[0], result[1], result[2])
    finally:
        std.os.close(rpipe)
        std.os.close(wpipe)


def fastbonize_ip(ip, log):
    """
    Returns: hostname, fastbone hostname, [[ipv4 fb ips], [ipv6 fb ips]]
    """

    import _socket

    try:
        hostname = gethostbyaddr_g(ip, gethostbyaddr=_socket.gethostbyaddr)[0]
    except _socket.error as ex:
        log.warning('Unable to grab hostname: %s', ex)
        return None, None, None

    log.debug('Resolved hostname %s', hostname)

    fb_hostname, fb_ips = fastbonize_hostname(hostname, log)
    return hostname, fb_hostname, fb_ips


def fastbonize_hostname(hostname, log):
    import _socket

    fb_ips = [[], []]  # ipv4, ipv6

    log.debug('fastbonize_hostname: %s', hostname)

    for variant in 1, 2, 3:
        if variant == 1:
            fb_hostname = 'fb-' + hostname
        elif variant == 2:
            fb_hostname = 'fastbone.' + hostname
        else:
            if hostname.endswith('.yandex.ru'):
                fb_hostname = hostname[:-len('.yandex.ru')] + '.fb.yandex.ru'
            elif '.' not in hostname:
                fb_hostname = hostname + '.fb.yandex.ru'
            else:
                continue

        try:
            log.debug('  resolving %s...', fb_hostname)
            ips = getaddrinfo_g(fb_hostname, 0, 0, gevent.socket.SOCK_STREAM, getaddrinfo=_socket.getaddrinfo)

            for family, _, _, _, ipinfo in ips:
                if family == gevent.socket.AF_INET:
                    fb_ips[0].append(ipinfo[0])
                    log.debug('    detected ipv4 fastbone: %s', ipinfo[0])
                elif family == gevent.socket.AF_INET6:
                    fb_ips[1].append(ipinfo[0])
                    log.debug('    detected ipv6 fastbone: %s', ipinfo[0])

            if variant == 2:
                # fastbone.* may be the same as backbone address, so we resolve
                # backbone here and recheck if they are equal. See SKYDEV-459.
                bb_ips = []
                try:
                    log.debug('    resolving bb ips %s...', hostname)
                    bb_ips_raw = getaddrinfo_g(
                        hostname, 0, 0, gevent.socket.SOCK_STREAM, getaddrinfo=_socket.getaddrinfo
                    )

                    for family, _, _, _, ipinfo in bb_ips_raw:
                        log.debug('      detected ip: %s', ipinfo[0])
                        bb_ips.append(ipinfo[0])
                except gevent.socket.gaierror as ex:
                    if ex.errno in (
                        getattr(gevent.socket, 'EAI_NONAME', None),
                        getattr(gevent.socket, 'EAI_NODATA', None)
                    ):
                        log.debug('      %s not found (errno: %d)', hostname, ex.errno)

                bb_same_as_fb = False
                for ip_by_family in fb_ips:
                    for ip in ip_by_family:
                        if ip in bb_ips:
                            bb_same_as_fb = True
                            break
                    if bb_same_as_fb:
                        break

                if bb_same_as_fb:
                    log.debug('    detected fastbone address matches backbone address, ignoring')
                    continue

            log.info(
                '  detected fastbone address %s with ipv4 ips %s and ipv6 ips %s',
                fb_hostname, fb_ips[0], fb_ips[1]
            )
            return fb_hostname, fb_ips

        except gevent.socket.gaierror as ex:
            if ex.errno in (
                getattr(gevent.socket, 'EAI_NONAME', None),
                getattr(gevent.socket, 'EAI_NODATA', None),
            ):
                log.debug('    %s not found (errno: %d)', fb_hostname, ex.errno)
                continue
            else:
                raise

    log.info('  fastbone address not found')
    return None, None


class SlottedDict(dict):
    def __getattr__(self, key):
        value = self[key]
        if isinstance(value, dict):
            value = self[key] = type(self)(value)
        self.__dict__[key] = value
        return value

    def __setattr__(self, key, value):
        self[key] = self.__dict__[key] = value


def human_size_sep(b):
    kb = 1024
    mb = kb * 1024
    gb = mb * 1024
    tb = gb * 1024

    def mkfl(fl, k):
        return ('%%.%df' % (k, )) % (fl, )

    if b >= tb * 100:  # 123 TiB
        return str(int(b / tb)), 'TiB'
    if b >= tb * 10:   # 12.23 TiB
        return mkfl(b / tb, 2), 'TiB'
    if b >= tb:        # 1.12 TiB
        return mkfl(b / tb, 2), 'TiB'
    if b >= gb * 100:  # 123 GiB
        return str(int(b / gb)), 'GiB'
    if b >= gb * 10:   # 12.23 GiB
        return mkfl(b / gb, 2), 'GiB'
    if b >= gb:        # 1.123 GiB
        return mkfl(b / gb, 2), 'GiB'
    if b >= mb:        # 123.23 MiB
        return mkfl(b / mb, 2), 'MiB'
    if b >= kb:        # 123.23 KiB
        return mkfl(b / kb, 2), 'KiB'
    if b > 0:          # 123 B
        return str(b), 'B'
    else:
        return '0', ''


def human_size(b):
    return ''.join(human_size_sep(b))


def human_time(s):
    if s < 1:
        return '%dms' % (s * 1000, )

    days, hours = divmod(s, 86400)
    hours, minutes = divmod(hours, 3600)
    minutes, seconds = divmod(minutes, 60)

    if not days and not hours and not minutes:
        return '%ds' % (seconds, )
    elif not days and not hours:
        return '%dm%.2ds' % (minutes, seconds)
    elif not days:
        return '%dh%.2dm%.2ds' % (hours, minutes, seconds)
    else:
        return '%dd%.2dh%.2dm%.2ds' % (days, hours, minutes, seconds)


def human_speed_sep(b):
    bps = b * 8
    mib = 10 ** 6
    kib = 10 ** 3

    if bps >= mib:
        return str(int(bps // mib)), 'Mbps'
    if bps >= kib:
        return str(int(bps // kib)), 'kbps'
    if bps >= 10:
        return '%.2f' % (bps / kib, ), 'kbps'
    else:
        return '0', ''


def human_speed(b):
    return ''.join(human_speed_sep(b))


def from_monotime(t):
    """convert in-process monotime to absolute timestamp"""
    if t <= 0:
        return t

    cur_time = time.time()
    cur_monotime = monoTime()
    return t - cur_monotime + cur_time


def to_monotime(t):
    """convert absolute timestamp to monotime"""
    if t <= 0:
        return t

    cur_time = time.time()
    cur_monotime = monoTime()
    return t - cur_time + cur_monotime


def has_root():
    res = []
    for attr in ('getresuid', 'getreuid'):
        if hasattr(os, attr):
            res = getattr(os, attr)()
            return 0 in res

    for attr in ('geteuid', 'getuid'):
        if hasattr(os, attr):
            if getattr(os, attr)() == 0:
                return True

    return False


def detect_hostname():
    hostname = socket.gethostname().lower()
    ips = set()

    not_found_errs = (
        getattr(socket, 'EAI_NONAME', None),
        getattr(socket, 'EAI_NODATA', None),
    )

    try:
        addrinfo = socket.getaddrinfo(hostname, 0, 0, socket.SOCK_STREAM, 0)
    except socket.gaierror as ex:
        if ex.errno in not_found_errs:
            # Unable to resolve ourselves
            return hostname
        raise
    else:
        for ipinfo in addrinfo:
            ips.add(ipinfo[4][0])

        fqdns = set()

        for ip in ips:
            try:
                fqdns.add(socket.gethostbyaddr(ip)[0])
            except socket.gaierror as ex:
                if ex.errno in not_found_errs:
                    continue
                raise
            except socket.herror as ex:
                # python has no error constants for gethostbyaddr
                if ex.strerror == 'Unknown host' or ex.errno == 1:
                    continue
                raise

        fqdns = list(fqdns)

        if hostname in fqdns:
            # Found hostname in fqdns
            return hostname
        elif len(fqdns) == 1 and fqdns[0].startswith(hostname):
            # Got only 1 fqdn
            return fqdns[0]
        else:
            # Got many fqdns, dont know how to choose one
            return hostname


def ensure_dir(name):
    parent = os.path.dirname(os.path.abspath(name))
    if parent != name and parent != '/' and not os.path.isdir(parent):
        ensure_dir(parent)
    if os.path.islink(name):
        os.unlink(name)
    if os.path.exists(name) and not os.path.isdir(name):
        os.unlink(name)
    if not os.path.exists(name):
        os.mkdir(name)


def ensure_link(source, target):
    ensure_dir(os.path.dirname(os.path.abspath(target)))
    if os.path.islink(target):
        dst = os.readlink(target)
        if dst != source:
            os.unlink(target)
    elif os.path.isdir(target):
        shutil.rmtree(target)
    elif os.path.exists(target):
        os.unlink(target)
    if not os.path.islink(target):
        os.symlink(source, target)
