import os
import sys
import logging
import threading

import six


class Fsyncer(object):
    def __init__(self, logger):
        self.log = logger or logging.getLogger('fsync')
        self._lock = threading.Lock()
        self._queue = six.moves.queue.Queue()
        self._stopped = False
        self._thread = None

    def put(self, path):
        if self._stopped:
            raise RuntimeError("Fsyncer object is stopped")

        if self._thread is None or not self._thread.is_alive():
            with self._lock:
                if self._stopped:
                    raise RuntimeError("Fsyncer object is stopped")

                if self._thread is None or not self._thread.is_alive():
                    self._run_thread()

        self._queue.put(path)

    def _run_thread(self):
        self.log.debug('Run new thread')
        self._thread = threading.Thread(
            target=self._thread_loop,
            name='Fsyncer',
        )
        self._thread.daemon = True
        self._thread.start()

    def _thread_loop(self):
        self.log.debug('Loop started')
        fsync_read_binary_mode = 'rb+' if sys.platform == 'cygwin' else 'rb'

        try:
            while not self._stopped:
                path = self._queue.get()
                if path is None:
                    break

                try:
                    with open(path, fsync_read_binary_mode) as fh:
                        os.fsync(fh.fileno())
                except EnvironmentError as e:
                    self.log.warning("fsync %r failed: %s", path, e)
        except BaseException as ex:
            self.log.warning('Loop finished (err:%s)', str(ex))
            raise
        else:
            self.log.debug('Loop finished (ok)')

    def stop(self):
        with self._lock:
            self._stopped = True
            self._queue.put(None)
            self._queue = six.moves.queue.Queue()

        thread = self._thread
        if thread is not None:
            thread.join(timeout=60)
