import cStringIO
import gzip
import requests
import msgpack


# GENISYS_URL = 'http://beta.genisys.yandex-team.ru/v2/'
# GENISYS_URL = 'http://genisys-angri.yandex-team.ru/v2/'
GENISYS_URL = 'http://api.genisys.yandex-team.ru/v2/'

GENISYS_API_URL = GENISYS_URL + 'hosts/{}?fmt=msgpack'
GENISYS_API_URL_WITH_HASH = GENISYS_URL + 'hosts/{}?config_hash={}&fmt=msgpack'
GENISYS_API_REPORT_URL = GENISYS_URL + 'host-status/{}/{}/{}'
GENISYS_API_CALL_TIMEOUT = 60


def download_config(hostname, log, last_modified=None, config_hash=None):
    try:
        # try to load new config
        headers = {'User-Agent': 'SkyCore/0.1',
                   'Accept-encoding': 'gzip'}

        if last_modified:
            headers['If-Modified-Since'] = last_modified

        if config_hash:
            url = GENISYS_API_URL_WITH_HASH.format(hostname, config_hash)
        else:
            url = GENISYS_API_URL.format(hostname)

        log.debug('Requesting genisys API via %r', url)
        r = requests.get(url,
                         headers=headers,
                         timeout=GENISYS_API_CALL_TIMEOUT)
        if r.status_code == 304:
            log.debug('Configuration has not been modified since %r. (%s)', last_modified, config_hash)
        elif r.status_code == requests.codes.ok:
            # process new received config data
            data = r.content
            if r.encoding == 'gzip':
                data = gzip.GzipFile(fileobj=cStringIO.StringIO(data)).read()
            data = msgpack.loads(data)
            last_modified = r.headers.get('Last-Modified')
            return last_modified, data
        else:
            r.raise_for_status()
    except requests.exceptions.Timeout:
        log.warning('Failed to connect to genisys. Timeout occurred')
    except requests.exceptions.RequestException as ex:
        log.error('Failed to connect to genisys. Error %s', ex)

    return None, None


def send_report(hostname, log, snapshot_revision, success, description=''):
    if not snapshot_revision:
        # old genisys does not support snapshot and reports
        return

    url = GENISYS_API_REPORT_URL.format(hostname, snapshot_revision, 'ok' if success else 'failed')
    headers = {'User-agent': 'SkyCore/1.0'}

    log.debug('Sending report: %r', url)
    reason = description.split('\n')[0]
    r = requests.post(url, headers=headers, data={'reason': reason, 'description': description})
    if r.status_code != requests.codes.ok:
        log.error('Failed to send report %s' % r.status_code)
