from cStringIO import StringIO
import os
import resource
import functools
import socket
import types

from .framework.greendeblock import Deblock
from .kernel_util import logging

try:
    from gevent.coros import RLock
except ImportError:
    from gevent.lock import RLock

import gevent

from porto.exceptions import SocketError, SocketTimeout

__all__ = ['build_env', 'build_limits_string', 'PortoAdapter']


def _quote_str(s):
    io = StringIO()
    io.write("'")
    for char in s:
        if char == "'":
            io.write("'\"'\"'")
        else:
            io.write(char)
    io.write("'")
    return io.getvalue()


def build_env(env=None):
    if env is None:
        env = {k: v for k, v in os.environ.iteritems() if k in ('SHELL', 'TERM', 'PATH', 'PYTHONPATH')}

    return '; '.join(
        '{}={}'.format(k, v.replace(';', r'\;'))
        for k, v in env.iteritems()
    )


def _limit_value(lim):
    return "unlim" if lim == resource.RLIM_INFINITY else str(lim)


def build_limits_string(limits):
    return '; '.join(
        "{}: {} {}".format(lim, _limit_value(soft), _limit_value(hard))
        for lim, (soft, hard)
        in limits.iteritems()
    )


class PortoAdapter(object):
    def __init__(self, slave, lock=None, log=None, deblock=None):
        self.__slave = slave
        self.__lock = lock or RLock()
        self._log = log or logging.getLogger('portoa')
        self.__deblock = deblock or Deblock(logger=self._log.getChild('dblk'), name='portoadapter')

    def stop(self):
        self.__deblock.stop()

    @staticmethod
    def _is_porto(obj):
        if 'porto.api' not in getattr(obj, '__module__', ''):
            return False
        if not hasattr(obj, '__class__'):
            return False
        if obj.__class__.__name__ == '_RPC':
            return False
        return True

    def __wrap(self, item):
        if callable(item):
            return self.__wrapper(item)
        elif self._is_porto(item):
            return PortoAdapter(item, lock=self.__lock, log=self._log, deblock=self.__deblock)
        elif isinstance(item, (list, types.GeneratorType)):
            return [
                (PortoAdapter(x, lock=self.__lock, log=self._log, deblock=self.__deblock) if self._is_porto(x) else x)
                for x in item
            ]
        else:
            return item

    def __disconnect(self):
        slave = self.__slave
        if hasattr(slave, 'conn'):
            slave = slave.conn
        if not hasattr(slave, 'rpc'):
            return
        self.__deblock.apply(slave.Disconnect)

    def __try_connect(self):
        slave = self.__slave
        if hasattr(slave, 'conn'):
            slave = slave.conn
        if not hasattr(slave, 'rpc'):
            return
        if slave.rpc.sock is None or slave.rpc.sock_pid is None:
            self.__deblock.apply(slave.TryConnect)

    def __wrapper(self, cb):
        @functools.wraps(cb)
        def _fn(*args, **kwargs):
            with self.__lock:
                for attempt in range(3):
                    try:
                        self.__try_connect()
                        result = self.__deblock.apply(cb, *args, **kwargs)
                        return self.__wrap(result)
                    except (socket.error, SocketError, SocketTimeout) as e:
                        self._log.warning(
                            "porto call %s failed: %s",
                            cb, e
                        )
                        self.__disconnect()
                        if attempt == 2:
                            raise
        return _fn

    def __getattr__(self, item):
        res = getattr(self.__slave, item)
        return self.__wrap(res)
