import os
import sys
import errno
import signal

import six
import psutil
import msgpack
import gevent
from gevent import socket

from ..kernel_util.sys.gettime import monoTime

from . import baseproc, proctools, fdtools, waitpid
from ..framework.utils import Path
from ..framework import event


LINER_PATH = os.path.join(os.path.dirname(sys.executable), 'liner')


class LinerProcess(baseproc.Proc):
    class LinerVersionMismatch(Exception):
        pass

    LINER_VERSION = 1           # Current liner version.
    LINER_START_TIMEOUT = 10    # Wait 10 seconds till the control socket will be created by the liner.

    restorable = True

    def __init__(self, log, out_log,
                 args=None,
                 cwd=None,
                 env=None,
                 limits=None,
                 cgroups=None,
                 rundir=None,
                 uuid=None,
                 pid=None,
                 username=None,
                 waitpid=None,
                 tags=None,
                 liner=None,
                 raw_args=None,
                 context=None):
        self.uuid = uuid
        self.log = log
        self.psproc = None
        self.pid = pid
        self.waitpid = waitpid
        self.liner = liner
        self.exitstatus = None
        self._max_rss = 0
        self._cpu_usage = 0

        self.finish_event = event.Event()

        self.args = args
        self.raw_args = raw_args
        self.env = env or {}
        self.cwd = cwd
        self.username = username
        self.limits = limits or {}
        self.cgroups = cgroups
        self.tags = tags or []

        if context is not None:
            for k, v in context.iteritems():
                if hasattr(self, k):
                    setattr(self, k, v)

        self.log = log
        self.rundir = rundir

        self.stdout = baseproc.Stream(log=out_log, mark='out')
        self.stderr = baseproc.Stream(log=out_log, mark='err')

        if liner is None:
            sockreader = self.start_liner(os.path.join(rundir, uuid + '.sock'))
        else:
            sockreader = self.adopt()

        gevent.spawn(self._process_liner_messages, sockreader)

    @property
    def pid(self):
        return self._pid

    @pid.setter
    def pid(self, pid):
        self._pid = pid
        if pid is not None:
            try:
                self.psproc = psutil.Process(pid)
            except psutil.NoSuchProcess:
                self.psproc = None
        else:
            self.psproc = None

    def context(self, base):
        return {
            'type': 'liner',
            'rundir': Path(self.rundir).relto(base),
            'uuid': self.uuid,
            'pid': self.pid,
            'tags': self.tags,
            'raw_args': self.raw_args,
        }

    def send_signal(self, num):
        pid = self.pid
        if pid is not None:
            self.log.debug('requested to send signal %s to %s', num, pid)
            try:
                os.kill(pid, num)
            except EnvironmentError as e:
                if e.errno == errno.ESRCH:
                    self.log.info("send signal %s to pid %s failed, process is dead", num, pid)
                else:
                    raise
        else:
            self.log.debug('not sending signal %s to already dead', num)

    def kill(self):
        self.send_signal(signal.SIGKILL)

    def executing(self):
        return bool(self.pid)

    @property
    def repr(self):
        if self.pid:
            return "pid %d" % self.pid
        return None

    def start_liner(self, sockname):
        log = self.log
        log.debug('Starting liner (%r) on the socket %r.', LINER_PATH, sockname)

        lpid = proctools.runproc(
            log,
            [LINER_PATH, sockname],
            self.env, self.cwd, self.username,
            limits=self.limits,
            cgroups=self.cgroups
        )

        self.waitpid = lpid

        log.debug('Liner (PID %d) spawned.', lpid)
        sock, sockreader = self._connect_liner_with_timeout(sockname, log)

        # Now we can close liner's out/err pipes and switch into normal functional state.
        self.liner = sock
        self.update_context(log)
        return sockreader

    def adopt(self):
        self.log.debug("Existing liner adopting.")
        self.waitpid = None

        # Connecting already running liner
        sockreader = self._socket_reader(self.liner)
        # Switch connected liner into the normal mode
        self.liner.send(msgpack.dumps("adopt"))
        # I wanna to be a paranoid and update liner's context here,
        # BUT, it can be a zombie actually, which already sent a message with exit status
        # and terminated normally. So we have to go socket messages read here first.
        return sockreader

    def update_context(self, log=None):
        log = log or self.log
        if not self.liner or not isinstance(self.liner, socket.socket):
            return

        log.debug("Update context storage in the liner.")
        self._send({'context': msgpack.dumps(self.context(''))})

    def _send(self, msg):
        self.liner.send(msgpack.dumps(msg))

    @staticmethod
    def _socket_reader(sock, timeout=None):
        sock.settimeout(timeout)
        unpacker = msgpack.Unpacker()
        while True:
            buf = sock.recv(0x100)
            if not buf:
                raise Exception('Liner unexpectedly closed the connection.')
            unpacker.feed(buf)
            for o in unpacker:
                yield o

    @staticmethod
    def _connect_liner(name, log):
        sock = socket.socket(socket.AF_UNIX, socket.SOCK_STREAM)
        fdtools.setcloexec(sock.fileno())
        timeout = 60
        log.debug('Connect socket %r (timeout %ds)', name, timeout)
        old_timeout = sock.gettimeout()
        sock.settimeout(timeout)
        sock.connect(name)
        sock.settimeout(old_timeout)
        return sock

    def _connect_liner_with_timeout(self, sockname, log=None):
        log = log or self.log
        wait_for = self.LINER_START_TIMEOUT
        start = monoTime()
        while monoTime() - start <= wait_for:
            if os.path.exists(sockname):
                try:
                    sock = self._connect_liner(sockname, log)
                except Exception as ex:
                    log.debug('Failed to connect liner: %s', ex)
                else:
                    break
            gevent.sleep(.05)
        else:
            # loop was not breaked -- we failed to connect
            raise Exception('Liner start timed out.')

        sockreader = self._socket_reader(sock)
        log.debug('Liner connected. Starting sub-process.')
        sock.send(msgpack.dumps('info'))
        assert sockreader.next()['version'] == self.LINER_VERSION

        sock.send(msgpack.dumps(['%s (%s)' % (' '.join(self.tags), self.uuid)] + self.args))
        self.pid = sockreader.next()
        log.info('Sub-process PID is %d. Going to normal functional mode.', self.pid)

        return sock, sockreader

    def _terminatepg(self, tries):
        for i in range(tries):
            try:
                self.log.debug('Sending terminatepg command to liner')
                self.liner.send(msgpack.dumps('terminatepg'))
            except BaseException as ex:
                if isinstance(ex, socket.error) and ex.errno == errno.EPIPE:
                    # Probably liner died. We will not be able to connect him after procman reboot anyway
                    # so, just post an error and continue
                    break

                try:
                    self.log.error('Failed with:', exc_info=sys.exc_info())
                except BaseException:
                    pass

                if i == (tries - 1):
                    # We are not able to kill liner gracefully. There is nothing to do here, just
                    # emergency exit :(
                    self.log.critical('Unable to stop liner normally, emergency exit')
                    os._exit(1)
                else:
                    gevent.sleep(1)
            else:
                break

    def _waitsockclose(self, timeout):
        with gevent.Timeout(timeout) as timeout_instance:
            try:
                assert self.liner.recv(1) == ''
                return True
            except gevent.Timeout as timeout_instance_received:
                if timeout_instance_received != timeout_instance:
                    raise
        return False

    def _process_liner_messages(self, sockreader):
        try:
            try:
                for data in sockreader:
                    for label in ('OUT', 'ERR'):
                        key = 'stdout' if label == 'OUT' else 'stderr'
                        chunk = data.get(key, None)
                        if not chunk:
                            continue
                        stream = getattr(self, key)
                        stream.feed(chunk)

                    if 'exited' in data:
                        # Process process termination event
                        self.log.info('Exited %r', data)
                        self.exitstatus = data

                        # Tell liner to terminate itself and his process group
                        self.log.debug('Sending terminatepg command to liner')
                        self.liner.send(msgpack.dumps('terminatepg'))
                        return
            except BaseException:
                excinfo = sys.exc_info()
                try:
                    self.log.error("Unhandled exception caught:", exc_info=excinfo)
                except BaseException:
                    pass

                # Since we possible have communication error -- try to forcibly send
                # terminatepg to liner. If we dont do this -- liner communication loop may stuck forever, while
                # liner will be function normally. In that case we will wait for liner exit forever in finally block.
                self._terminatepg(tries=60)
                six.reraise(*excinfo)
            finally:
                self.stdout.finish()
                self.stderr.finish()
                if self.waitpid:
                    self.log.debug('Waiting liner process for exit (pid: %d)', self.waitpid)
                    status, _ = waitpid.pidwaiter().wait(self.waitpid)
                    if self.exitstatus is not None:
                        self.exitstatus.update(liner_exited=1, liner_exitstatus=status)
                    else:
                        self.exitstatus = dict(liner_exited=1,
                                               liner_exitstatus=status,
                                               exitstatus=-1,
                                               exited=1,
                                               signaled=0,
                                               note="liner died prematurely")
                    self.log.debug('Liner process exited with status %r', status)
                else:
                    self.log.debug('Dont waiting process exit for adopted liner')

                self.log.debug('Waiting liner socket to be closed')

                if not self._waitsockclose(60):
                    self.log.error('Liner not exited normally, killing his process group')
                    for _ in six.moves.xrange(60):
                        if self._waitsockclose(1):
                            self.log.debug('Liner socket closed')
                            break
                    else:
                        self.log.critical('Failed to kill liner!!!')
                else:
                    self.log.debug('Liner socket closed')
        finally:
            self.finish_event.set()
            self.log.debug("Liner runner loop finished.")
            self.liner = None
            self.pid = None

    @classmethod
    def reconnect(cls, log, out_log, base, context):
        rundir = context['rundir'] if os.path.isabs(context['rundir']) else os.path.join(base, context['rundir'])
        sockname = os.path.join(rundir, context['uuid'] + '.sock')
        try:
            sock = cls._connect_liner(sockname, log)
        except Exception as ex:
            log.warning("Connect failed: %s. Unlinking socket file...", str(ex))

            try:
                os.unlink(sockname)
            except EnvironmentError as e:
                if e.errno != errno.ENOENT:
                    raise

            return None

        try:
            sock.send(msgpack.dumps('info'))
            reader = cls._socket_reader(sock, cls.LINER_START_TIMEOUT)

            info = reader.next()
            if info['version'] != cls.LINER_VERSION:
                raise cls.LinerVersionMismatch('We want %r we got %r' % (
                    cls.LINER_VERSION, info['version']
                ))
            context['pid'] = info['pid']
            if not context['pid']:
                raise Exception("Liner subprocess pid not available")
            proc = cls(log=log, out_log=out_log, rundir=rundir, liner=sock, context=context)
            proc.log.debug("Successfully connected")
            return proc
        except Exception as ex:
            if isinstance(ex, cls.LinerVersionMismatch):
                log.warning('Liner version mismatch: %s', ex)
            else:
                log.error('Got unhandled exception while reconnecting liner:', exc_info=sys.exc_info())

            log.debug('Sending terminatepg and waiting for 5 seconds for socket to be closed')

            sock.send(msgpack.dumps('terminatepg'))
            with gevent.Timeout(5) as timeout:
                try:
                    while sock.recv(1) != '':
                        pass
                    log.info('Old liner exited!')
                except gevent.Timeout as ex:
                    if ex != timeout:
                        raise
                    log.error('Old liner refuses to die =(. Will leave him orphan, removing sock file')
                    os.unlink(sockname)

            sock.close()
            return None

    def wait(self, timeout=None):
        return self.finish_event.wait(timeout=timeout)

    def link(self, cb):
        if isinstance(cb, event.Event):
            self.finish_event.link_event(cb)
        else:
            self.finish_event.rawlink(lambda _: cb(self))

    @property
    def ready(self):
        return self.exitstatus is not None

    @property
    def succeeded(self):
        return (
            self.ready
            and self.exitstatus['exitstatus'] == 0
            and self.exitstatus['signaled'] == 0
        )

    @property
    def failed(self):
        return self.ready and (
            self.exitstatus['exitstatus'] != 0
            or self.exitstatus['signaled'] != 0
        )

    @property
    def cpu_usage(self):
        if self.psproc is None:
            return self._cpu_usage
        try:
            u = self.psproc.get_cpu_times()
            self._cpu_usage = u.user + u.system
        except psutil.NoSuchProcess:
            pass

        return self._cpu_usage

    @property
    def rss(self):
        if self.psproc is not None:
            try:
                rss = self.psproc.get_memory_info().rss
                self._max_rss = max(self._max_rss, rss)
                return rss
            except psutil.NoSuchProcess:
                pass
        return 0

    @property
    def max_rss(self):
        # FIXME (torkve) we need to teach liner to send max_rss along with exitstatus when process dies
        return self._max_rss
