import os
import signal
import subprocess

from .. import portoutils
from ..interval import IntervalController
from ..framework import event
from . import baseproc, proctools

import gevent

try:
    from porto.exceptions import (
        ContainerDoesNotExist, ContainerAlreadyExists, InvalidState, EError, SocketError, SocketTimeout
    )
except ImportError:
    pass


class PortoProcess(baseproc.Proc):
    def __init__(self, log, out_log, conn,
                 watcher=None,
                 args=None,
                 cwd=None,
                 root_container=None,
                 uuid=None,
                 username=None,
                 limits=None,
                 env=None,
                 meta_options=None,
                 options=None,
                 raw_args=None,
                 tags=None,
                 context=None,
                 job=None):
        self.log = log
        self.conn = conn
        self.watcher = watcher
        self.job = job
        self.uuid = uuid
        self.username = username
        self.root_container = root_container

        self._cpu_usage = 0
        self._max_rss = 0
        self.exitstatus = None
        self.finish_event = event.Event()
        self.wake_event = event.Event()

        self.respawn = False
        self.cwd = cwd
        self.args = args
        self.raw_args = raw_args
        self.tags = tags
        self.meta_options = meta_options or {}
        self.options = options or {}
        self.options.setdefault('ulimit', portoutils.build_limits_string(proctools.build_limits(limits or {})))
        self.options.setdefault('env', portoutils.build_env(env or {}))

        self.poll_interval = IntervalController(initial=0.25, maximum=30.0)
        self.stdout = baseproc.Stream(log=out_log, mark='out')
        self.stderr = baseproc.Stream(log=out_log, mark='err')

        self.stdout_offset = -1
        self.stderr_offset = -1

        if context is not None:
            for k, v in context.iteritems():
                if hasattr(self, k):
                    setattr(self, k, v)

        if job is None:
            self.start_container()

        self.watcher.procs.add(self.uuid)
        gevent.spawn(self.runner)

    def ensure_root_container(self):
        container = self.root_container
        if not container:
            return

        parts = container.split('/')
        target = ''
        for part in parts:
            if target:
                target += '/' + part
            else:
                target = part

            try:
                self.conn.Create(target)
            except ContainerAlreadyExists:
                c = self.conn.Find(target)
                state = c.GetData('state')
                if state == 'dead':
                    c.Stop()
                    c.Start()
                elif state == 'stopped':
                    c.Start()

        cont = self.conn.Find(container)
        try:
            for k, v in self.meta_options.iteritems():
                cont.SetProperty(k, v)
        except InvalidState:
            pass  # Container is already configured and started

    def start_container(self):
        try:
            container_name = self._container_name(self.root_container, self.uuid)
            self.log.debug('Creating porto container %s (%s)', container_name, self.tags)
            self.ensure_root_container()
            try:
                self.conn.Stop(container_name)  # if it's running
            except ContainerDoesNotExist:
                pass

            try:
                self.conn.Destroy(container_name)  # if it already exists
            except ContainerDoesNotExist:
                pass
            self.job = self.conn.Create(container_name)

            props = dict(isolate=False,  # TODO set by option  # noqa
                         respawn=self.respawn,
                         command=subprocess.list2cmdline(self.args),
                         stdin_path='/dev/null',
                         stdout_limit=(1 << 19),  # 1M
                         user=self.username,
                         cwd=self.cwd,
                         private='SKYCORE',
                         aging_time=600,
                         )
            props.update(self.options)

            for k, v in props.iteritems():
                self.job.SetProperty(k, v)
        except Exception as e:
            self.log.error("container creation failed: %s", e)
            try:
                if self.job is not None:
                    self.job.Destroy()  # if it already exists
            except ContainerDoesNotExist:
                pass
            finally:
                self.job = None
            raise e

    @property
    def repr(self):
        if self.job:
            return "porto %s" % self.job.name

    def send_signal(self, num):
        self.log.debug('requested to send signal %s to %s', num, self.uuid)
        try:
            self.job.Kill(num)
        except InvalidState:
            self.log.info("send signal %s to uuid %s failed, process is dead", num, self.uuid)
        except ContainerDoesNotExist:
            self.log.info("send signal %s to uuid %s failed, container already destroyed", num, self.uuid)

    def kill(self):
        self.send_signal(signal.SIGKILL)

    def context(self, base):
        return {
            'type': 'porto',
            'uuid': self.uuid,
            'root_container': self.root_container,
            'stdout_offset': self.stdout_offset,
            'stderr_offset': self.stderr_offset,
            'raw_args': self.raw_args,
        }

    def executing(self):
        return self.job and self.job.GetData("state") == "running"

    @staticmethod
    def _container_name(root_container, uuid):
        name = "skycore-{}".format(uuid)
        if root_container is not None:
            name = '{}/{}'.format(root_container, name)
        return name

    @staticmethod
    def stop_container(conn, name):
        try:
            container = conn.Find(name)
        except ContainerDoesNotExist:
            return

        container.Stop()

    def _run(self):
        try:
            state = self.job.GetData('state')
            if state == 'stopped':
                self.job.Start()
            elif state == 'dead' and self.respawn:
                self.job.Stop()
                self.job.Start()
            elif state == 'paused':
                self.job.Resume()
        except EError as e:
            raise Exception("Cannot start porto container: %s" % (e,))

        try:
            while True:
                try:
                    state = self.job.GetData('state')  # get early, check lately, so you lose no output
                    if state != 'stopped':
                        self._check_output('stdout')
                        self._check_output('stderr')
                        self.poll_interval.schedule_next()
                    if state in ('dead', 'stopped'):
                        # if we stop parent container to kill service, this one will also be stopped
                        break
                    self.wake_event.wait(self.poll_interval.interval)
                    self.wake_event.clear()
                except SocketTimeout:
                    gevent.sleep(5.)  # let give porto some time to recover
                    continue
        finally:
            self.watcher.procs.discard(self.uuid)

        exitstatus = int(self.job.GetData('exit_status'))
        ret = {
            'exited': int(os.WIFEXITED(exitstatus)),
            'exitstatus': os.WEXITSTATUS(exitstatus),
            'signaled': int(os.WIFSIGNALED(exitstatus)),
        }
        if ret['signaled']:
            ret['termsig'] = os.WTERMSIG(exitstatus)
            ret['coredump'] = int(os.WCOREDUMP(exitstatus))

        return exitstatus, ret

    def runner(self):
        exitstatus = None
        ret = None

        try:
            try:
                exitstatus, ret = self._run()
            except Exception:
                self.log.exception("uuid %s runner failed: " % (self.uuid,))
                raise
            finally:
                try:
                    self.job.Stop()
                except (EError, SocketError, SocketTimeout):
                    pass

                try:
                    self._cpu_usage = float(self.job.GetData('cpu_usage')) / 1e9
                except (EError, SocketError, SocketTimeout):
                    pass
                try:
                    self._max_rss = int(self.job.GetData('max_rss'))
                except (EError, SocketError, SocketTimeout):
                    pass

                try:
                    self.job.Destroy()
                except ContainerDoesNotExist:
                    pass
                except BaseException:
                    self.log.exception("Container %s destroy failed:" % (self.uuid,))
                    raise
        finally:
            self.stdout.finish()
            self.stderr.finish()

            if exitstatus is None:
                exitstatus = -1
                ret = {"exited": 1, "exitstatus": -1, "signaled": 0, "note": "exit info is lost"}

            self.exitstatus = ret
            self.log.info("Exited %r", ret)

            self.finish_event.set()

        return exitstatus

    def _check_output(self, outtype):
        stream = getattr(self, outtype)

        offset = int(self.job.GetData(outtype + '_offset'))
        begin = getattr(self, outtype + '_offset')
        end = begin + len(stream.last_val)
        if offset < begin:
            return

        if offset >= end:
            data = self.job.GetData("%s[%d]" % (outtype, offset))
            setattr(self, outtype + '_offset', offset)
        else:
            # read only diff
            data = self.job.GetData("%s[%d]" % (outtype, end))
            setattr(self, outtype + '_offset', end)

        stream.feed(data)

        if data:
            self.poll_interval.reset()

    @classmethod
    def reconnect(cls, porto, log, out_log, context, watcher):
        cont = cls._container_name(context['root_container'], context['uuid'])
        try:
            job = porto.Find(cont)
        except ContainerDoesNotExist:
            log.debug('[%s] process already disappeared', context['uuid'])
            return None
        except (SocketError, SocketTimeout) as e:
            log.debug("[%s] porto not available: %s", context['uuid'], e)
            return None

        return cls(conn=porto, log=log, out_log=out_log, context=context, job=job, watcher=watcher)

    def wait(self, timeout=None):
        return self.finish_event.wait(timeout=timeout)

    def link(self, cb):
        if isinstance(cb, event.Event):
            self.finish_event.link_event(cb)
        else:
            self.finish_event.rawlink(lambda _: cb(self))

    @property
    def ready(self):
        return self.exitstatus is not None

    @property
    def succeeded(self):
        return (
            self.ready
            and self.exitstatus['exitstatus'] == 0
            and self.exitstatus['signaled'] == 0
        )

    @property
    def failed(self):
        return self.ready and (
            self.exitstatus['exitstatus'] != 0
            or self.exitstatus['signaled'] != 0
        )

    @property
    def cpu_usage(self):
        if self.job is not None and not self.finish_event.is_set():
            try:
                self._cpu_usage = float(self.job.GetData('cpu_usage')) / 1e9
            except Exception:
                pass
        return self._cpu_usage

    @property
    def rss(self):
        if self.job is not None and not self.finish_event.is_set():
            try:
                return int(self.job.GetData('memory_usage'))
            except Exception:
                pass
        return 0

    @property
    def max_rss(self):
        if self.job is not None and not self.finish_event.is_set():
            try:
                self._max_rss = int(self.job.GetData('max_rss'))
            except Exception:
                pass
        return self._max_rss

    in_porto = True
    restorable = True
