from __future__ import absolute_import

import os
import logging

from ..kernel_util.uuid import genuuid

from . import waitpid
from .subproc import Subprocess
from .linerproc import LinerProcess
from .portoproc import PortoProcess
from .exceptions import ProcStartException


class StarterLoggerAdapter(logging.LoggerAdapter):
    def __init__(self, logger, extra):
        super(StarterLoggerAdapter, self).__init__(logger, extra)

    def process(self, msg, kwargs):
        msg = '[%-5s]  {%-8s}  %s' % (
            self.extra.get('type', '?????'),
            self.extra.get('uuid', 'unknown')[:8],
            msg
        )

        return msg, kwargs


class Starter(object):
    def __init__(self, porto, portowatcher, workdir, process_lock, cgroup_controller, default_cgroup=None):
        self.porto = porto
        self.portowatcher = portowatcher
        self.workdir = workdir
        self.process_lock = process_lock
        self.cgroup_controller = cgroup_controller
        self.default_cgroup = default_cgroup if default_cgroup is not None else 'skycore/services'

    def config_changed(self, _, new_config):
        try:
            starter_cfg = new_config['config']['starter']
            cgroup = starter_cfg['default_cgroup']
            self.default_cgroup = cgroup
        except (KeyError, TypeError):
            # this exception means that some section is not resolved yet
            # hence we can just do nothing, and leave previous default.
            # NB: if config applicance never succeeded, default 'skycore/services' will remain.
            pass

        return True

    def reconnect(self, base, log, out_log, ctx):
        if ctx['type'] == 'porto':
            logger = StarterLoggerAdapter(log, {'uuid': ctx['uuid'], 'type': 'porto'})
            return PortoProcess.reconnect(self.porto, logger, out_log=out_log, context=ctx, watcher=self.portowatcher)
        elif ctx['type'] == 'liner':
            logger = StarterLoggerAdapter(log, {'uuid': ctx['uuid'], 'type': 'liner'})
            return LinerProcess.reconnect(logger, out_log=out_log, base=base, context=ctx)
        else:
            raise TypeError('Unsupported process type')

    def stop_container(self, name):
        if self.porto is None:
            return

        PortoProcess.stop_container(self.porto, name)

    def get_valid_cgroups(self, name):
        return self.cgroup_controller and self.cgroup_controller.get_valid_cgroups(name if name else self.default_cgroup)

    def run(
        self, log, out_log, args, service_root, porto,
        user, cgroups, limits, env, fast, root_container,
        porto_meta_options, porto_options, tags, raw_args,
    ):
        uuid = genuuid()
        if porto in ('yes', True) and self.porto is None:
            raise RuntimeError("Cannot start process, no porto on host")

        if (
            porto in ('yes', True) or
            (porto == 'auto' and self.porto is not None and os.path.exists('/run/portod.socket'))
        ):
            try:
                logger = StarterLoggerAdapter(log, {'uuid': uuid, 'type': 'porto'})
                return PortoProcess(logger,
                                    out_log,
                                    self.porto,
                                    watcher=self.portowatcher,
                                    root_container=root_container,
                                    uuid=uuid,
                                    username=user,
                                    cwd=service_root,
                                    args=args,
                                    limits=limits,
                                    env=env,
                                    options=porto_options,
                                    meta_options=porto_meta_options,
                                    tags=tags,
                                    raw_args=raw_args,
                                    )
            except Exception:
                if porto in ('yes', True):
                    raise

        try:
            with self.process_lock:
                if fast:
                    logger = StarterLoggerAdapter(log, {'uuid': uuid, 'type': 'proc'})
                    return Subprocess(
                        logger,
                        out_log,
                        args=args,
                        uuid=uuid,
                        cwd=service_root,
                        env=env,
                        limits=limits,
                        cgroups=cgroups,
                        username=user
                    )
                else:
                    logger = StarterLoggerAdapter(log, {'uuid': uuid, 'type': 'liner'})
                    return LinerProcess(
                        logger,
                        out_log,
                        rundir=self.workdir,
                        uuid=uuid,
                        username=user,
                        cwd=service_root,
                        args=args,
                        cgroups=cgroups,
                        limits=limits,
                        env=env,
                        tags=tags,
                        raw_args=raw_args
                    )
        except ProcStartException as e:
            waitpid.pidwaiter().wait(e.pid)
            raise
