import os
import errno
import signal
import subprocess
import logging

import psutil
import gevent.select

from ..framework import event
from .exceptions import ProcStartException
from . import proctools, fdtools, baseproc, waitpid


class PidLoggerAdapter(logging.LoggerAdapter):
    def __init__(self, logger, extra):
        super(PidLoggerAdapter, self).__init__(logger, extra)

    def process(self, msg, kwargs):
        msg = '[PID=%-5s]  %s' % (self.extra.get('pid', None) or '', msg)
        return msg, kwargs


class Subprocess(baseproc.Proc):
    def __init__(self, log, out_log,
                 args,
                 uuid,
                 cwd=None,
                 env=None,
                 limits=None,
                 cgroups=None,
                 username=None):
        self.log = PidLoggerAdapter(log, {'pid': None})

        self.pid = None
        self.psproc = None
        self._cpu_usage = 0
        self._max_rss = 0
        self.exitstatus = None

        self.finish_event = event.Event()

        self.args = args
        self.env = env or {}
        self.cwd = cwd
        self.username = username
        self.limits = limits or {}
        self.cgroups = cgroups
        self.stdout = baseproc.Stream(log=out_log, mark='out')
        self.stderr = baseproc.Stream(log=out_log, mark='err')

        try:
            self.pid, self.sout, self.serr = proctools.runproc(
                log=log,
                args=self.args,
                env=self.env,
                cwd=self.cwd,
                username=self.username,
                limits=self.limits,
                cgroups=self.cgroups,
                catch_outs=True,
            )
            try:
                self.psproc = psutil.Process(self.pid)
            except psutil.NoSuchProcess:
                self.psproc = None
            self.log.extra['pid'] = self.pid
            self.log.debug('Started process (%s)', subprocess.list2cmdline(self.args))
        except ProcStartException as ex:
            self.log.error(
                'Unable to start process (%s): %s: %s',
                subprocess.list2cmdline(self.args), type(ex).__name__, ex
            )
            raise
        except Exception as ex:
            self.log.error(
                'Unable to start process (%s): %s: %s',
                subprocess.list2cmdline(self.args), type(ex).__name__, ex
            )
            raise

        gevent.spawn(self.process_outs)
        gevent.spawn(self.watch)

    def process_outs(self):
        out = self.sout
        err = self.serr

        fdtools.setnonblocking(out)
        fdtools.setnonblocking(err)

        fds = [out, err]
        while fds:
            r, _, _ = gevent.select.select(fds, [], [], 3.0)
            if out in r:
                try:
                    o = os.read(out, 16384)
                    if not o:
                        fds.remove(out)
                        self.stdout.finish()
                    else:
                        self.stdout.feed(o)
                except EnvironmentError as e:
                    if e.errno == errno.EBADF:
                        fds.remove(out)
                        self.stdout.finish()
                    elif e.errno not in (errno.EWOULDBLOCK, errno.EAGAIN):
                        raise

            if err in r:
                try:
                    o = os.read(err, 16384)
                    if not o:
                        fds.remove(err)
                        self.stderr.finish()
                    else:
                        self.stderr.feed(o)
                except EnvironmentError as e:
                    if e.errno == errno.EBADF:
                        fds.remove(err)
                        self.stderr.finish()
                    elif e.errno not in (errno.EWOULDBLOCK, errno.EAGAIN):
                        raise

    def watch(self):
        log = self.log

        try:
            exitstatus, rusage = waitpid.pidwaiter().wait(self.pid)
            ret = {
                'exited': int(os.WIFEXITED(exitstatus)),
                'exitstatus': os.WEXITSTATUS(exitstatus),
                'signaled': int(os.WIFSIGNALED(exitstatus)),
            }
            if ret['signaled']:
                ret['termsig'] = os.WTERMSIG(exitstatus)
                ret['coredump'] = int(os.WCOREDUMP(exitstatus))
            self._max_rss = rusage.ru_maxrss * 1024
            self._cpu_usage = rusage.ru_utime + rusage.ru_stime
            self.psproc = None
            self.exitstatus = ret
        except Exception as ex:
            log.warning('Unable to watch process: %s: %s', type(ex).__name__, ex)
            self.send_signal(signal.SIGKILL)
        else:
            if ret['exitstatus'] == 0:
                log.info('Exited with 0')
            else:
                log.info("Exited with %d (%r)", ret['exitstatus'], ret)
            self.finish_event.set()

    def context(self, base):
        return None

    def link(self, cb):
        if isinstance(cb, event.Event):
            self.finish_event.link_event(cb)
        else:
            self.finish_event.rawlink(lambda _: cb(self))

    def wait(self, timeout=None):
        return self.finish_event.wait(timeout=timeout)

    def send_signal(self, num):
        pid = self.pid
        if pid is not None and self.exitstatus is None:
            self.log.debug('requested to send signal %s to %s', num, pid)
            try:
                os.kill(self.pid, num)
            except EnvironmentError as e:
                if e.errno == errno.ESRCH:
                    self.log.info("send signal %s to pid %s failed, process is dead", num, self.pid)
                else:
                    raise
        else:
            self.log.debug('not sending signal %s to already dead %s', num, pid)

    def kill(self):
        self.send_signal(signal.SIGKILL)

    @property
    def ready(self):
        return self.exitstatus is not None

    @property
    def succeeded(self):
        return (
            self.ready
            and self.exitstatus['exitstatus'] == 0
            and self.exitstatus['signaled'] == 0
        )

    @property
    def failed(self):
        return self.ready and (
            self.exitstatus['exitstatus'] != 0
            or self.exitstatus['signaled'] != 0
        )

    @property
    def cpu_usage(self):
        if self.psproc is None:
            return self._cpu_usage

        try:
            u = self.psproc.get_cpu_times()
            self._cpu_usage = u.user + u.system
        except psutil.NoSuchProcess:
            pass

        return self._cpu_usage

    @property
    def rss(self):
        if self.psproc is not None:
            try:
                return self.psproc.get_memory_info().rss
            except psutil.NoSuchProcess:
                pass

        return 0

    @property
    def max_rss(self):
        return self._max_rss
