import os
import time
import logging
from functools import partial

import gevent.event

from ..kernel_util.functional import singleton
from ..kernel_util.errors import formatException
from ..kernel_util.misc import daemonthr

from . import mailbox


class WaitPid(object):
    """
    Don't you ever dare to remove this and use other ways
    to handle SIGCHLD. pg@ claims that he managed to miss some signals
    using sighandler + waitpid(-1, WNOHANG).
    """
    def __init__(self, mbox=None):
        self.pids = {}
        self.mbox = mbox or mailbox.mailbox()
        daemonthr(self.run)

    def run(self):
        tout = 0

        while True:
            try:
                self.mbox.put(partial(self.onexit, *os.wait3(0)))
                tout = 0
            except OSError:
                time.sleep(tout)
                tout = min((tout + 0.1) * 1.2, 1)
            except Exception as e:
                logging.getLogger('waitpid').info(str(formatException(e)))

    def onexit(self, pid, status, rusage):
        e = self.event(pid)
        e.set_result((status, rusage))

    def wait(self, pid, timeout=None):
        e = self.event(pid)
        finished = e.wait(timeout)
        del self.pids[pid]
        if finished:
            return e.get_nowait()
        return None

    def event(self, pid):
        if pid not in self.pids:
            self.pids[pid] = gevent.event.AsyncResult()
        return self.pids[pid]


@singleton
def pidwaiter():
    return WaitPid()
