import os
import re
import sys
import signal

import six
import yaml

from .framework.utils import Path


# Typical service description example::
#   conf:
#     - skynet.services.MyService
#     - skynet.base.cgroups
#     - sandbox.client_conf
#   conf_format: yaml  # optional, possible values: json, yaml, default: yaml
#   conf_action: None # possible values: None, signal (int or SIGNAME), or script
#   require:
#     - copier
#   exec:
#     - bin/myservice-server --config ${CFGFILE} --ns ${NAMESPACE} ${cfg:my.super.section:my.super.key}
#     - /skynet/python/bin/python bin/myservice-client
#   max_check_interval: 600.0  # optional
#   check: bin/check ${NAMESPACE}
#   stop: bin/stop-server --grace --all ${NAMESPACE}  # optional, SIGINT by default
#   install_script: bin/install ${MODE}  # optional, variable MODE takes forms 'install', 'upgrade'
#   uninstall_script: bin/uninstall ${MODE}  # optional, variable MODE takes forms, 'preupgrade', 'uninstall'
#   install_as_privileged: false  # optional, if set (un)install scripts will be called with root privileges
#   restart_on_upgrade: true  # optional, if false, only (un)install scripts will be invoked
#   porto: auto  # allowed: 'yes', 'no' or 'auto'
#   porto_options:          # optional, dictionary or porto container options,
#     isolate: true         # will be applied to the root container of the service
#     cpu_guarantee: 0.5c
#   porto_container: ISS    # optional, will be used as root meta-container for all processes started in it
#   user: root
#   cgroup: None  # optional
#   limits:       # optional
#     nofile: 1024  # single value sets both soft and hard limits
#     nproc: [100, 120]
#   env:  # optional
#     CONFIG: ${CFGFILE}
#     HOME: ${RUNDIR}
#   api:          # optional, map of api type to corresponding API options
#     python:
#       import_paths:
#         - "${CURDIR}/api"
#       requires:  # pkg_resource requirements
#         - skynet-example-service==1.0
#       module: org.services.myservice.api
#       object: MyObject  # or e.g. function get_new_api_object
#       call: true
#       args: [1, 2, 3]
#       kwargs:
#         socket: ${RUNDIR}/service.sock
#         c: d
#   # added in version 2:
#   scsd_version: 2
#   aliases:  # optional
#     common: bin/runner --namespace=${NAMESPACE}  # you can use it in other places like ${alias:common}
#     some_other: ${alias:common} --another-arg=something-else
#   scripts:  # optional, scripts can be retrieved using 'skyctl get'
#     show_stats: ${alias:some_other} show-stats


def _as_iterable(x):
    if isinstance(x, six.string_types):
        return [x]
    else:
        return x


def parse_porto_options(options):
    if not options or options == 'None':
        return
    elif not isinstance(options, dict):
        raise TypeError('`porto_options` must be null or dictionary, got: %r' % (type(options),))

    unknown_keys = set(options.keys()) - {'meta', 'exec', 'check', 'stop', 'install', 'uninstall', 'notify'}
    if unknown_keys:
        raise TypeError('unknown keys found in `porto_options`: %s' % (list(unknown_keys),))

    result = {}
    for key, value in options.items():
        if not value or value == 'None':
            result[key] = None
        elif isinstance(value, dict):
            result[key] = value
        else:
            raise TypeError("`porto_options[%r]` must be null or dictionary, got: %r" % (key, type(options)))

    return result


class ServiceConfig(object):
    @property
    def dependencies(self):
        raise NotImplementedError  # pragma: nocoverage

    @property
    def configs(self):
        raise NotImplementedError  # pragma: nocoverage

    @property
    def registry_filename(self):
        raise NotImplementedError  # pragma: nocoverage

    @classmethod
    def from_path(cls, path, log):
        filename = os.path.basename(path)
        name, ext = os.path.splitext(filename)
        if ext != '.scsd':
            raise ValueError("Incorrect service filename: %r" % (filename,))

        try:
            data = yaml.load(open(path, 'rb'), Loader=getattr(yaml, 'CSafeLoader', yaml.SafeLoader))
        except yaml.error.YAMLError as e:
            log.exception("Couldn't load service description: %s", e, exc_info=sys.exc_info())
            raise TypeError("Not valid YAML description: %r" % (filename,))

        version = data.get('scsd_version', 1)
        scsd_cls = globals().get('ServiceConfigV' + str(version))
        if not version or not scsd_cls:
            raise TypeError("Unknown service description version")

        return scsd_cls.from_data(name, path, data, log)

    @classmethod
    def from_data(cls, name, path, data, log):
        version = data.get('scsd_version', 1)
        scsd_cls = globals().get('ServiceConfigV' + str(version))
        if not version or not scsd_cls:
            raise TypeError("Unknown service description version")

        return scsd_cls.from_data(name, path, data, log)

    @classmethod
    def parse_action(cls, action):
        if not action or action == 'None':
            return None
        elif isinstance(action, int):
            return action

        if action.startswith('SIG'):
            return getattr(signal, action)

        return action

    @classmethod
    def from_context(cls, **kwargs):
        version = kwargs.pop('version', 1)
        scsd_cls = globals().get('ServiceConfigV' + str(version))
        if not version or not scsd_cls:
            raise TypeError("Unknown service description version")

        return scsd_cls(**kwargs)

    def as_dict(self, base=None):
        raise NotImplementedError  # pragma: nocoverage

    def format_item(self, item, **additional_opts):
        return item

    def take(self, var, **additional_opts):
        return self.format_item(getattr(self, var), **additional_opts)

    def get_porto_options(self, kind):
        raise NotImplementedError

    def get_field(self, field, raw):
        if field not in ('dependencies', 'configs', 'registry_filename'):
            raise AttributeError("No such field: %r" % (field,))
        if raw:
            return getattr(self, field)
        else:
            return self.take(field)

    def cfg_version(self):
        raise NotImplementedError

    def __eq__(self, other):
        return False


class ServiceConfigV1(ServiceConfig):
    arg_re = re.compile(r'\$\{(?:([^:}]+):)?([^}]+)\}', re.UNICODE)

    # mappings format variants:
    # - NAME: None         - required raw value, NAME in scsd equal to NAME in class
    # - NAME: VALUE        - required raw value, name VALUE is scsd is parsed as NAME in class
    # - NAME: (V1, V2)     - required value, name V1 in scsd is parsed with function V2
    # - NAME: (V1, V2, V3) - optional value, name V1 in scsd is parsed with function V2, default value V3
    field_mappings = {
        'require': ('requirements', _as_iterable),
        'conf': ('conf_sections', _as_iterable),
        'conf_format': (None, None, 'yaml'),
        'conf_action': (None, ServiceConfig.parse_action, None),
        'exec': ('executables', _as_iterable),
        'max_check_interval': (None, float, 600.0),
        'check': (None, None, None),
        'stop': (None, ServiceConfig.parse_action, signal.SIGINT),
        'porto': None,
        'porto_options': (None, None, None),
        'porto_container': (None, None, None),
        'user': None,
        'cgroup': (None, None, None),
        'limits': (None, None, None),
        'env': (None, None, None),
        'install_script': (None, None, None),
        'uninstall_script': (None, None, None),
        'restart_on_upgrade': (None, None, True),
        'install_as_privileged': (None, None, False),
        'api': (None, None, None),
    }

    def __init__(self,
                 name,
                 base,
                 basepath,
                 porto,
                 check,
                 stop,
                 conf_sections,
                 conf_format,
                 conf_action,
                 executables,
                 user,
                 max_check_interval=45.0,
                 requirements=None,
                 cgroup=None,
                 limits=None,
                 env=None,
                 install_script=None,
                 uninstall_script=None,
                 restart_on_upgrade=True,
                 install_as_privileged=False,
                 porto_options=None,
                 porto_container=None,
                 api=None,
                 **kwargs
                 ):
        super(ServiceConfigV1, self).__init__()
        self.name = name
        self.basepath = basepath if os.path.isabs(basepath) else os.path.join(base, basepath)
        self.requirements = frozenset(requirements or ())
        self.conf_sections = frozenset(conf_sections)
        self.conf_format = conf_format
        self.conf_action = conf_action
        self.executables = executables
        self.porto = porto
        self.porto_options = porto_options or {}
        self.porto_container = porto_container
        self.max_check_interval = max_check_interval
        self.check = check
        self.stop = stop
        self.user = user
        self.cgroup = cgroup
        self.limits = limits
        self.env = env
        self.install_script = install_script
        self.uninstall_script = uninstall_script
        self.restart_on_upgrade = restart_on_upgrade
        self.install_as_privileged = install_as_privileged
        self.api = api
        self._var_type_handlers = {}
        self._static_var_handlers = {}
        self._var_handlers = {}

        self.setup_default_handlers()

    def setup_default_handlers(self):
        self._var_type_handlers.update(
            env=self.format_env,
        )
        self._static_var_handlers.update(
            CURDIR=self.basepath,
            SKYNET_PYTHON=sys.executable,
        )
        self._var_handlers.update(
            CFGFILE=lambda: self.registry_filename,
        )

    def set_type_handler(self, typename, handler):
        self._var_type_handlers[typename] = handler

    def set_var_handler(self, varname, handler):
        if callable(handler):
            self._var_handlers[varname] = handler
            self._static_var_handlers.pop(varname, None)
        else:
            self._var_handlers.pop(varname, None)
            self._static_var_handlers[varname] = str(handler)

    def _get_var(self, var_name, additional_vars={}):
        if var_name in additional_vars:
            return additional_vars[var_name]
        elif var_name in self._var_handlers:
            return self._var_handlers[var_name]()
        elif var_name in self._static_var_handlers:
            return self._static_var_handlers[var_name]
        else:
            # self.log.warning("unknown variable used: ${%s}", var_name)
            return ''

    def _get_typed_var(self, var_type, var_name):
        if var_type in self._var_type_handlers:
            return self._var_type_handlers[var_type](var_type, var_name)
        # self.log.warning("unknown variable type: ${%s:%s}", var_type, var_name)
        return ''

    def format_str(self, val, **additional_vars):
        match = self.arg_re.search(val)
        while match is not None:
            var_type = match.group(1)
            var_name = match.group(2)
            ret = ''
            if not var_type:
                ret = str(self._get_var(var_name, additional_vars))
            else:
                ret = str(self._get_typed_var(var_type, var_name))

            val = val.replace(match.group(0), ret)
            match = self.arg_re.search(val)

        return val

    def format_env(self, var_type, var_name):
        if var_name not in os.environ:
            # self.log.warning("unknown environment variable: ${env:%s}", var_name)
            return ''
        else:
            return os.getenv(var_name)

    def format_item(self, item, **additional_opts):
        if isinstance(item, six.string_types):
            return self.format_str(item, **additional_opts)
        elif isinstance(item, dict):
            return {
                self.format_item(k, **additional_opts): self.format_item(v, **additional_opts)
                for k, v in item.iteritems()
            }
        elif isinstance(item, (tuple, list, set, frozenset)):
            return type(item)(self.format_item(x, **additional_opts) for x in item)
        else:
            return item

    @property
    def dependencies(self):
        return self.format_item(self.requirements)

    @property
    def configs(self):
        return self.format_item(self.conf_sections)

    @property
    def registry_filename(self):
        return 'configuration.' + self.format_str(self.conf_format)

    def get_porto_options(self, kind):
        if kind == 'meta':
            return self.take('porto_options')
        else:
            return {}

    def get_field(self, field, raw):
        if field not in self.field_mappings:
            raise AttributeError("No such field: %r" % (field,))
        new_name = self.field_mappings[field]
        new_name = new_name[0] if isinstance(new_name, tuple) else new_name
        new_name = field if new_name is None else new_name
        if raw:
            return getattr(self, new_name)
        else:
            return self.take(new_name)

    def cfg_version(self):
        name = self.__class__.__name__
        prefix = 'ServiceConfigV'
        assert name.startswith(prefix)
        return int(name[len(prefix):])

    @classmethod
    def from_data(cls, name, path, data, log):
        args = {'name': name, 'base': None, 'basepath': os.path.dirname(path)}
        try:
            args.update(cls.parse_data(data))
        except KeyError as e:
            raise TypeError("Service description lack required field: `%s`" % e.args[0])

        return cls(**args)

    @classmethod
    def parse_data(cls, data):
        args = {}
        for field, mapping in cls.field_mappings.iteritems():
            if mapping is None:
                args[field] = data[field]
            elif not isinstance(mapping, tuple):
                args[mapping] = data[field]
            else:
                new_name = mapping[0] if mapping[0] is not None else field
                val = data.get(field, mapping[2]) if len(mapping) > 2 else data[field]
                val = val if mapping[1] is None else mapping[1](val)
                args[new_name] = val

        return args

    def as_dict(self, base=None):
        return {
            'name': self.name,
            'basepath': self.basepath if base is None else Path(self.basepath).relto(base),
            'requirements': list(self.requirements),
            'conf_sections': list(self.conf_sections),
            'conf_format': self.conf_format,
            'conf_action': self.conf_action,
            'executables': list(self.executables),
            'porto': self.porto,
            'porto_options': self.porto_options,
            'porto_container': self.porto_container,
            'max_check_interval': self.max_check_interval,
            'check': self.check,
            'stop': self.stop,
            'user': self.user,
            'cgroup': self.cgroup,
            'limits': self.limits,
            'env': self.env,
            'install_script': self.install_script,
            'uninstall_script': self.uninstall_script,
            'install_as_privileged': self.install_as_privileged,
            'restart_on_upgrade': self.restart_on_upgrade,
            'api': self.api,
            'version': self.cfg_version(),
        }

    def __eq__(self, other):
        return (
            type(self) is type(other)
            and self.name == other.name
            and self.basepath == other.basepath
            and sorted(list(self.requirements)) == sorted(list(other.requirements))
            and sorted(list(self.conf_sections)) == sorted(list(other.conf_sections))
            and self.conf_format == other.conf_format
            and self.conf_action == other.conf_action
            and sorted(list(self.executables)) == sorted(list(other.executables))
            and self.porto == other.porto
            and self.porto_options == other.porto_options
            and self.porto_container == other.porto_container
            and self.max_check_interval == other.max_check_interval
            and self.check == other.check
            and self.stop == other.stop
            and self.user == other.user
            and self.cgroup == other.cgroup
            and self.limits == other.limits
            and self.env == other.env
            and self.install_script == other.install_script
            and self.uninstall_script == other.uninstall_script
            and self.install_as_privileged == other.install_as_privileged
            and self.restart_on_upgrade == other.restart_on_upgrade
            and self.api == other.api
        )


class ServiceConfigV2(ServiceConfigV1):
    field_mappings = ServiceConfigV1.field_mappings.copy()
    field_mappings['aliases'] = (None, None, {})
    field_mappings['scripts'] = (None, None, {})

    def __init__(self, scripts=None, aliases=None, **kwargs):
        self.aliases = aliases or {}
        self.scripts = scripts or {}
        super(ServiceConfigV2, self).__init__(**kwargs)

    def setup_default_handlers(self):
        super(ServiceConfigV2, self).setup_default_handlers()
        self.set_type_handler('alias', self.format_alias)
        self.set_type_handler('script', self.format_alias)

    def format_alias(self, var_type, var_name):
        aliases = self.aliases if var_type == 'alias' else self.scripts
        if var_name not in aliases:
            # self.log.warning("unknown alias: %r", var_name)
            return ''

        return self.format_item(aliases[var_name])

    @property
    def registry_filename(self):
        return os.path.join(
            str(self._get_var('RUNDIR')),
            'conf', 'configuration.' + self.format_str(self.conf_format)
        )

    def as_dict(self, base=None):
        data = super(ServiceConfigV2, self).as_dict(base=base)
        data['aliases'] = self.aliases
        data['scripts'] = self.scripts
        return data

    def __eq__(self, other):
        return (
            super(ServiceConfigV2, self).__eq__(other)
            and self.aliases == other.aliases
            and self.scripts == other.scripts
        )


class ServiceConfigV3(ServiceConfigV2):
    format_srvenv = ServiceConfigV2.format_env

    field_mappings = ServiceConfigV2.field_mappings.copy()
    del field_mappings['require']
    field_mappings['after'] = (None, _as_iterable, None)

    def __init__(self, after=None, **kwargs):
        self.after = frozenset(after or ())
        super(ServiceConfigV3, self).__init__(**kwargs)

    def format_cfg_env(self, var_type, var_name):
        env = self.take('env')
        if env and var_name in env:
            return env[var_name]
        return ''

    def setup_default_handlers(self):
        super(ServiceConfigV3, self).setup_default_handlers()
        self.set_type_handler('srvenv', self.format_srvenv)
        self.set_type_handler('env', self.format_cfg_env)

    def as_dict(self, base=None):
        data = super(ServiceConfigV3, self).as_dict(base=base)
        data.pop('require', None)
        return data

    def __eq__(self, other):
        return (
            super(ServiceConfigV3, self).__eq__(other)
            and self.after == other.after
        )


class ServiceConfigV4(ServiceConfigV3):
    def setup_default_handlers(self):
        super(ServiceConfigV4, self).setup_default_handlers()
        self.set_type_handler('cfgfile', self.format_cfgfile)

    @property
    def registry_filename_dir(self):
        return os.path.join(str(self._get_var('RUNDIR')), 'conf')

    def format_cfgfile(self, var_type, var_name):
        return os.path.join(
            self.registry_filename_dir,
            var_name + '.' + self.format_str(self.conf_format)
        )


class ServiceConfigV5(ServiceConfigV4):
    # Just version bump to notify Service that it can process conf_action='RESTART' differently
    pass


class ServiceConfigV6(ServiceConfigV5):
    field_mappings = ServiceConfigV5.field_mappings.copy()
    field_mappings['restart_on_skydeps_upgrade'] = (None, None, False)

    def __init__(self, restart_on_skydeps_upgrade=False, **kwargs):
        self.restart_on_skydeps_upgrade = restart_on_skydeps_upgrade
        super(ServiceConfigV6, self).__init__(**kwargs)

    def as_dict(self, base=None):
        data = super(ServiceConfigV6, self).as_dict(base=base)
        data['restart_on_skydeps_upgrade'] = self.restart_on_skydeps_upgrade
        return data

    def __eq__(self, other):
        return (
            super(ServiceConfigV6, self).__eq__(other)
            and self.restart_on_skydeps_upgrade == other.restart_on_skydeps_upgrade
        )


class ServiceConfigV7(ServiceConfigV6):
    field_mappings = ServiceConfigV6.field_mappings.copy()
    field_mappings['porto_options'] = (None, parse_porto_options, None)

    def get_porto_options(self, kind):
        return (self.take('porto_options') or {}).get(kind)


def main():
    import argparse
    import logging
    from pprint import pprint

    parser = argparse.ArgumentParser(description='validate service descriptor file')
    parser.add_argument('path', type=str,
                        help='path to .scsd file')

    log = logging.getLogger()
    log.addHandler(logging.StreamHandler())
    pprint(ServiceConfig.from_path(os.path.abspath(parser.parse_args().path), log).as_dict())
    print("Service descriptor appears correct.")


if __name__ == '__main__':
    main()
