from collections import defaultdict


class Stats(object):
    def __init__(self, db):
        self.db = db
        self.inmemory_values = defaultdict(int)
        self.inmemory_service_values = defaultdict(int)

    def main_inc_num(self, key, inc=1, inmemory=False):
        if inmemory:
            self.inmemory_values[key] += inc
            return

        changed = self.db.query(
            'UPDATE stats '
            'SET value = value + ? '
            'WHERE key = ?',
            (inc, key), get_changed=True
        )

        if changed == 0:
            self.db.query(
                'INSERT INTO stats (key, value) VALUES (?, ?)',
                (key, inc)
            )

    def main_dec_num(self, key, dec=1, inmemory=False):
        return self.main_inc_int(key, -dec, inmemory=inmemory)

    def main_set_val(self, key, val, inmemory=False):
        if inmemory:
            self.inmemory_values[key] = val
            return

        self.db.query(
            'REPLACE INTO stats '
            '(key, value) '
            'VALUES (?, ?)',
            (key, val)
        )

    def main_get_val(self, key, inmemory=False):
        if inmemory:
            return self.inmemory_values.get(key)

        for (val,) in self.db.query(
            'SELECT value FROM stats '
            'WHERE key = ?',
            (key,),
            log=False,
        ):
            return val

    def main_remove_key(self, key, inmemory=False):
        if inmemory:
            self.inmemory_values.pop(key, None)
            return

        self.db.query(
            'DELETE FROM stats '
            'WHERE key = ?',
            (key,)
        )

    def service_inc_num(self, ns, service, key, inc=1, inmemory=False):
        if inmemory:
            self.inmemory_service_values[(ns, service, key)] += inc
            return

        changed = self.db.query(
            'UPDATE service_stats '
            'SET value = value + ? '
            'WHERE namespace = ? AND service = ? AND key = ?',
            (inc, ns, service, key),
            get_changed=True
        )

        if changed == 0:
            self.db.query(
                'INSERT INTO service_stats (namespace, service, key, value) '
                'VALUES (?, ?, ?, ?)',
                (ns, service, key, inc)
            )

    def service_dec_num(self, ns, service, key, dec, inmemory=False):
        return self.service_inc_int(ns, service, key, -dec, inmemory=inmemory)

    def service_set_val(self, ns, service, key, val, inmemory=False):
        if inmemory:
            self.inmemory_service_values[(ns, service, key)] = val
            return

        self.db.query(
            'REPLACE INTO service_stats '
            '(namespace, service, key, value) '
            'VALUES (?, ?, ?, ?)',
            (ns, service, key, val)
        )

    def service_get_val(self, ns, service, key, inmemory=False):
        if inmemory:
            return self.inmemory_service_values.get((ns, service, key))

        for (val,) in self.db.query(
            'SELECT value FROM service_stats '
            'WHERE namespace = ? AND service = ? AND key = ?',
            (ns, service, key),
            log=False,
        ):
            return val

    def service_remove_inmemory(self, ns, service):
        for nskey, servicekey, key in list(self.inmemory_service_values.keys()):
            if nskey == ns and servicekey == service:
                self.inmemory_service_values.pop((nskey, servicekey, key), None)

    def delete_service_stats(self, ns, service):
        self.db.query('DELETE FROM service_stats WHERE namespace = ? AND service = ?', (ns, service))
        self.service_remove_inmemory(ns, service)

    def get_all(self):
        stats = {
            'main': dict(self.db.query('SELECT key, value FROM stats')),
            'svcs': {}
        }

        for k, v in list(self.inmemory_values.items()):
            stats['main'][k] = v

        for ns, service, key, value in self.db.query(
            'SELECT namespace, service, key, value FROM service_stats'
        ):
            stats['svcs'].setdefault(ns, {}).setdefault(service, {})[key] = value

        for (ns, service, key), value in list(self.inmemory_service_values.items()):
            stats['svcs'].setdefault(ns, {}).setdefault(service, {})[key] = value

        return stats
