import time
import logging
from functools import wraps

import ujson as json
from six.moves import urllib


__all__ = ['fetch_json', 'fetch_lines']


class RetryWithSleep(Exception):
    pass


def retried(attempts, sleep):
    def _retried(func):
        @wraps(func)
        def _wrapper(*args, **kwargs):
            attempt = attempts
            while attempt > 0:
                try:
                    return func(*args, **kwargs)
                except RetryWithSleep:
                    time.sleep(sleep)
                except Exception:
                    attempt -= 1
                    if attempt == 0:
                        raise
        return _wrapper
    return _retried


@retried(attempts=3, sleep=1.0)
def _fetch(urls, source, kind, data=None, headers={}, log=None, json_response=False):
    log = log or logging.getLogger('skylib.http_tools')

    if isinstance(urls, str):
        urls = (urls,)

    for i, url in enumerate(urls):
        res = None
        try:
            req = urllib.request.Request(url, data=data, headers=headers)
            res = urllib.request.urlopen(req, timeout=10)
            if res.getcode() == 429:
                raise RetryWithSleep()

            if res.getcode() != 200:
                raise Exception("non-200 response from %s %r (%s)" % (source, url, res.code,))

            try:
                if json_response:
                    return json.load(res)
                else:
                    return list(res.readlines())
            except Exception:
                log.warning("cannot parse %s from %s!", kind, source)
                raise
        except urllib.error.URLError:
            if i + 1 == len(urls):
                raise
        finally:
            if res is not None:
                res.close()


def fetch_json(urls, source, kind, data=None, headers={}, log=None):
    return _fetch(
        urls=urls,
        source=source,
        kind=kind,
        data=data,
        headers=headers,
        log=log,
        json_response=True,
    )


def fetch_lines(urls, source, kind, data=None, headers={}, log=None):
    return _fetch(
        urls=urls,
        source=source,
        kind=kind,
        data=data,
        headers=headers,
        log=log,
        json_response=False,
    )
