import os
import grp
import pwd

import six
from porto import exceptions


def get_container_root(porto_connection, container_name):
    return porto_connection.ConvertPath("/", container_name, "self")


def get_parent_container(porto_connection, container_name):
    parent = container_name.rsplit('/', 1)[0] if '/' in container_name else '/'
    if not parent:
        raise RuntimeError("cannot detect parent for container %r" % (container_name,))

    cparent = porto_connection.Find(parent)
    return cparent


def is_memory_limited(porto_connection, container):
    while True:
        try:
            lim = int(container.GetProperty('memory_limit'))
        except (ValueError, TypeError, exceptions.NoValue):
            lim = 0

        if lim:
            return True

        container = get_parent_container(porto_connection, container.name)

        if container.name == '/':
            break

    return False


def get_container_user_and_group(porto_connection, container, predefined_user=None):
    if predefined_user:
        gid = pwd.getpwnam(predefined_user).pw_gid
        group = grp.getgrgid(gid).gr_name
        return predefined_user, group

    if isinstance(container, six.string_types + (bytes,)):
        container = porto_connection.Find(container)

    if '/' != porto_connection.ConvertPath("/", container.name, "self"):
        # container is isolated, we can do anything
        return "root", 'root'

    user = container.GetProperty("user")
    group = container.GetProperty("group")

    return user, group


def set_capabilities(porto_connection, container, user, own_user):
    parent = container.name.rsplit('/', 1)[0]
    parent = porto_connection.Find(parent)

    isolate = parent.GetProperty("isolate")
    parent_pid = int(parent.GetProperty('root_pid'))
    net = os.readlink('/proc/self/ns/net') != os.readlink('/proc/%d/ns/net' % (parent_pid,))

    if user == 'nobody':
        caps = set()
    elif own_user != 'root':
        # we are in container and we are limited to what we're given
        caps = set(filter(None, container.GetProperty('capabilities').split(';')))
    else:
        caps = set(filter(None, parent.GetProperty("capabilities").split(';')))

        caps.update((
            # tcpdump
            'NET_RAW',
        ))

        if isolate:
            caps.update((
                # gdb, strace
                'SYS_PTRACE',
                # sudo-like features
                'KILL',
            ))

        if user == "root":
            caps.update((
                'SETUID',
                'SETGID',
                'AUDIT_WRITE',
                'CHOWN',
                'DAC_OVERRIDE',
                'FOWNER',
            ))

        if net:
            caps.update((
                # iptables
                'NET_ADMIN',
                # bind < 1024
                'NET_BIND_SERVICE',
            ))

    ambient_caps = caps.copy()

    if get_container_root(porto_connection, container.name) == '/':
        # full bounding set for non-chroot
        ambient_caps &= {
            'IPC_LOCK',
            'SYS_BOOT',
            'KILL',
            'SYS_PTRACE',
            'NET_ADMIN',
            'NET_BIND_SERVICE',
            'NET_RAW',
        }

        if not net:
            ambient_caps.discard('NET_ADMIN')

        if not is_memory_limited(porto_connection, container):
            ambient_caps.discard('IPC_LOCK')

        if not isolate:
            ambient_caps.difference_update((
                'SYS_BOOT',
                'KILL',
                'SYS_PTRACE',
            ))

    caps = ';'.join(caps)
    try:
        container.SetProperty("capabilities", caps)  # shouldn't have more than this
    except (exceptions.InvalidProperty, exceptions.NotSupported, exceptions.UnknownError):
        pass

    try:
        ambient_caps_limit = container.GetProperty('capabilities_ambient_allowed')
    except (exceptions.InvalidProperty, exceptions.NoValue, exceptions.NotSupported, exceptions.UnknownError):
        pass
    else:
        ambient_caps_limit = set(filter(None, ambient_caps_limit.split(';')))
        ambient_caps &= ambient_caps_limit

    ambient_caps = ';'.join(ambient_caps)
    try:
        container.SetProperty("capabilities_ambient", ambient_caps)
    except (exceptions.InvalidProperty, exceptions.NotSupported, exceptions.UnknownError):
        pass
