package errorbooster

import (
	"context"
	"encoding/json"
	"fmt"
	"time"

	"a.yandex-team.ru/kikimr/public/sdk/go/persqueue"
	"a.yandex-team.ru/kikimr/public/sdk/go/ydb"
)

type Config struct {
	Project  string `yaml:"project"`
	Env      string `yaml:"env"`
	Topic    string `yaml:"topic"`
	SourceID string `yaml:"source_id"`
}

type Data struct {
	Message   string `json:"message"`
	Timestamp int64  `json:"timestamp"`
	Project   string `json:"project"`
	Env       string `json:"env"`
	Level     string `json:"level"`
}

type Client struct {
	project string
	env     string
	writer  persqueue.Writer
}

func NewClient(config *Config, logbrokerToken string, ctx context.Context) (*Client, error) {
	writer := persqueue.NewWriter(persqueue.WriterOptions{
		Endpoint:       "logbroker.yandex.net",
		Credentials:    ydb.AuthTokenCredentials{AuthToken: logbrokerToken},
		TLSConfig:      nil,
		Database:       "/Root",
		Topic:          config.Topic,
		SourceID:       []byte(config.SourceID),
		Codec:          persqueue.Raw,
		RetryOnFailure: true,
	})
	if _, err := writer.Init(ctx); err != nil {
		return nil, err
	}
	return &Client{
		writer:  writer,
		project: config.Project,
		env:     config.Env,
	}, nil
}

func (c *Client) sendMessage(message []byte) error {
	err := c.writer.Write(&persqueue.WriteMessage{Data: message})
	if err != nil {
		return err
	}

	rsp := <-c.writer.C()
	switch m := rsp.(type) {
	case *persqueue.Ack:
		return nil
	case *persqueue.Issue:
		return m.Err
	default:
		return fmt.Errorf("unknown writer response")
	}
}

func (c *Client) Send(message string) error {
	data := &Data{
		Message:   message,
		Timestamp: time.Now().UnixNano() / 1000000,
		Project:   c.project,
		Env:       c.env,
		Level:     "error",
	}
	binData, err := json.Marshal(data)
	if err != nil {
		return err
	}
	if err := c.sendMessage(binData); err != nil {
		return err
	}

	return nil
}

func (c *Client) Close() error {
	return c.writer.Close()
}
