package nanny

import (
	"context"
	"fmt"
	"net/http"
	"strconv"

	"github.com/go-resty/resty/v2"

	pb "a.yandex-team.ru/infra/nanny/go/proto/nanny_repo"
	"a.yandex-team.ru/infra/nanny2/pkg/rpc"
)

type Client struct {
	restClient *resty.Client
	rpcClient  pb.RepoServiceClient
}

func NewClient(URL string, oauthToken string) *Client {
	rpcConfig := rpc.ClientConfig{
		RPCURL:     fmt.Sprintf("%s/api/repo", URL),
		OauthToken: oauthToken,
	}
	return &Client{
		restClient: resty.New().
			SetBaseURL(URL).
			SetAuthScheme("OAuth").
			SetAuthToken(oauthToken).
			SetHeader("Content-Type", "application/json").
			SetHeader("Accept", "application/json"),
		rpcClient: pb.NewRepoServiceClient(rpc.NewClient(&rpcConfig)),
	}
}

type Annotaions struct {
	DeployEngine string `json:"deploy_engine"`
}

type MetaInfo struct {
	Annotaions *Annotaions `json:"annotations"`
}

type ChangeInfo struct {
	Author string `json:"author"`
}

type RuntimeAttrs struct {
	MetaInfo   *MetaInfo   `json:"meta_info"`
	ChangeInfo *ChangeInfo `json:"change_info"`
}

type ActiveSnapshot struct {
	State string `json:"state"`
}

type Summary struct {
	Value string `json:"value"`
}

type CurrentState struct {
	ActiveSnapshots []*ActiveSnapshot `json:"active_snapshots"`
	Summary         *Summary          `json:"summary"`
}

type DutySchedule struct {
	ABCServiceID int `json:"abc_service_id"`
	ID           int `json:"id"`
}

type InfoAttrs struct {
	EnvType      string       `json:"env_type,omitempty"`
	DutySchedule DutySchedule `json:"duty_schedule,omitempty"`
	AbcGroup     int          `json:"abc_group"`
}

type InfoAttrsContent struct {
	Content *InfoAttrs `json:"content"`
}

type Service struct {
	ID           string            `json:"_id"`
	RuntimeAttrs *RuntimeAttrs     `json:"runtime_attrs"`
	InfoAttrs    *InfoAttrsContent `json:"info_attrs"`
	CurrentState *CurrentState     `json:"current_state"`
}

type GetServicesResponse struct {
	Result []*Service `json:"result"`
}

func (c *Client) GetServices(ctx context.Context, batchSize int) ([]*Service, error) {
	var services []*Service
	skip := 0
	for {
		resp, err := c.restClient.R().
			SetResult(&GetServicesResponse{}).
			SetQueryParams(
				map[string]string{
					"skip":  strconv.Itoa(skip),
					"limit": strconv.Itoa(batchSize),
				},
			).
			SetContext(ctx).
			Get("v2/services/")

		if err != nil {
			return nil, err
		}

		if resp.StatusCode() != http.StatusOK {
			return nil, fmt.Errorf("unsupported response code %d: %s", resp.StatusCode(), resp.String())
		}

		respServices := resp.Result().(*GetServicesResponse).Result
		if len(respServices) == 0 {
			break
		}

		services = append(services, respServices...)
		skip += batchSize
	}

	return services, nil
}

func (c *Client) GetReplicationPolicies(ctx context.Context, batchSize int) (map[string]*pb.ReplicationPolicy, error) {
	req := &pb.ListReplicationPoliciesRequest{
		Skip:  0,
		Limit: int32(batchSize),
	}

	policies := map[string]*pb.ReplicationPolicy{}
	for {
		response, err := c.rpcClient.ListReplicationPolicies(ctx, req)
		if err != nil {
			return nil, err
		}

		if len(response.Policy) == 0 {
			break
		}

		for _, policy := range response.Policy {
			policies[policy.Meta.Id] = policy
		}
		req.Skip += int32(batchSize)
	}

	return policies, nil
}

type GetServiceRuntimeAttrsHistoryResponse struct {
	Result []*RuntimeAttrs `json:"result"`
}

func (c *Client) GetServiceRuntimeAttrsHistory(serviceID string, limit int) ([]*RuntimeAttrs, error) {
	resp, err := c.restClient.R().
		SetResult(&GetServiceRuntimeAttrsHistoryResponse{}).
		Get(fmt.Sprintf("v2/services/%s/history/runtime_attrs_infos/?limit=%d&skip=0", serviceID, limit))

	if err != nil {
		return nil, err
	}

	if resp.StatusCode() != http.StatusOK {
		return nil, fmt.Errorf("unsupported response code from service: %d: %s", resp.StatusCode(), resp.String())
	}

	return resp.Result().(*GetServiceRuntimeAttrsHistoryResponse).Result, nil
}
