package yp

import (
	"context"
	"fmt"
	"strings"
	"time"

	"google.golang.org/protobuf/proto"

	"a.yandex-team.ru/yp/go/proto/ypapi"
	"a.yandex-team.ru/yp/go/yp"
)

type YpClient struct {
	ypClientMap map[string]*yp.Client
	batchSize   int32
	clusters    []string
}

type YpPod struct {
	Pod     *ypapi.TPod
	Cluster string
}

func NewClient(clusters []string, batchSize int32, timeout time.Duration, ypToken string) (ypClient *YpClient, err error) {
	options := []yp.Option{
		yp.WithAuthToken(ypToken),
		yp.WithRequestTimeout(timeout),
	}
	ypClientMap := make(map[string]*yp.Client)

	for _, cluster := range clusters {
		client, err := yp.NewClient(cluster, options...)
		if err != nil {
			return nil, err
		}
		ypClientMap[strings.ToLower(cluster)] = client
	}
	return &YpClient{
		ypClientMap: ypClientMap,
		batchSize:   batchSize,
		clusters:    clusters,
	}, nil
}

type ListObjectsReciever interface {
	MakeEmptyItem() proto.Message
	AddItem(proto.Message, string) error
}

func (c *YpClient) Close() {
	for _, client := range c.ypClientMap {
		client.Close()
	}
}

type ListObjectsOptions struct {
	Selectors  []string
	Filter     string
	Cluster    string
	ObjectType yp.ObjectType
}

func (c *YpClient) ListObjects(ctx context.Context, options *ListObjectsOptions, rcv ListObjectsReciever) error {
	client := c.ypClientMap[options.Cluster]
	ts, err := client.GenerateTimestamp(ctx)
	if err != nil {
		return fmt.Errorf("timestamp generation failed: %w", err)
	}
	request := yp.SelectObjectsRequest{
		Format:          yp.PayloadFormatProto,
		ObjectType:      options.ObjectType,
		Filter:          options.Filter,
		Selectors:       options.Selectors,
		Timestamp:       ts,
		Limit:           c.batchSize,
		FetchRootObject: true,
	}
	for {
		response, err := client.SelectObjects(ctx, request)
		if err != nil {
			return err
		}
		for response.Next() {
			obj := rcv.MakeEmptyItem()
			if err := response.Fill(obj); err != nil {
				return err
			}
			if err := rcv.AddItem(obj, options.Cluster); err != nil {
				return err
			}
		}

		if int32(response.Count()) < c.batchSize {
			break
		}
		request.ContinuationToken = response.ContinuationToken()
	}

	return nil
}
