package evictionpods

import (
	"bytes"
	"fmt"
	"sort"
	"strings"
	"text/template"

	"a.yandex-team.ru/infra/spnotifier/processors"
)

const EvictionPodsTemplateText = `<#
<table>
<tr>
	<td><strong>Имя пода</strong></td>
	<td><strong>Время запроса эвакуации</strong></td>
	<td><strong>Время принудительного выключения</strong></td>
</tr>
{{ range .Table }}
<tr>
<td><a href="{{ .DeployURL }}/yp/{{ .Cluster }}/pods/{{ .PodID }}" target="_blank">{{ .HostName }}</a></td>
<td style="text-align: center;">{{ .EvictionRequestedTime }}</td>
<td style="text-align: center;">{{ .ForcedEvictionTime }}</td>
</tr>
{{ end }}
</table>
#>
{{ if gt (len .List) 0 }}
<{ И еще несколько подов ({{ len .List }})
	{{ .RenderedList }}
}>
{{ end }}`

const EvictionPodsTemplateName = "EvictionPods"

var EvictionPodsTemplate = template.Must(template.New(EvictionPodsTemplateName).Parse(EvictionPodsTemplateText))

type Pod struct {
	DeployURL             string
	Cluster               string
	PodID                 string
	HostName              string
	EvictionRequestedTime string
	ForcedEvictionTime    string
}

type Data struct {
	Table        []*Pod
	List         []string
	RenderedList string
}

type Config struct {
	MaxTableSize int
}

type Processor struct {
	Pods []*Pod
	Cfg  Config

	data Data
}

func (p *Processor) getDesciption() (string, error) {
	buf := new(bytes.Buffer)
	err := EvictionPodsTemplate.Execute(buf, p.data)
	if err != nil {
		return "", err
	}
	return buf.String(), nil
}

func (p *Processor) Process() (*processors.NotificationProcessorResult, error) {
	if len(p.Pods) == 0 {
		return &processors.NotificationProcessorResult{NeedsRendering: false}, nil
	}

	sort.Slice(p.Pods, func(i, j int) bool {
		return p.Pods[i].HostName < p.Pods[j].HostName
	})
	for _, pod := range p.Pods {
		if len(p.data.Table) < p.Cfg.MaxTableSize {
			p.data.Table = append(p.data.Table, pod)
		} else {
			p.data.List = append(p.data.List, pod.HostName)
		}
	}
	p.data.RenderedList = strings.Join(p.data.List, ", ")

	description, err := p.getDesciption()
	if err != nil {
		return nil, fmt.Errorf("eviction pods template render failed: %w", err)
	}

	return &processors.NotificationProcessorResult{
		NeedsRendering: true,
		Description:    description,
	}, nil
}
