package evictionrequested

import (
	"bytes"
	"fmt"
	"text/template"
	"time"

	"a.yandex-team.ru/infra/spnotifier/processors"
	"a.yandex-team.ru/infra/spnotifier/processors/common/evictionpods"
	"a.yandex-team.ru/infra/spnotifier/providers/pods"
)

const TemplateText = `В вашем (({{ .DeployURL }}/stages/{{ .StageID }}/ стейдже)) есть поды, эвакуация для которых запрошена более {{ .EvictionExpiredThresholdHours }} часов назад. Переселение не удалось подтвердить автоматически. Возможные причины следующие:
- Слишком много сломанных подов и не хватает бюджета на переселение
- Включено ручное подтверждение всех переселений
- Есть незавершённая выкладка (переселение работает только при отсутствии выкладки)
{{ .PodsTable }}
`

type Data struct {
	StageID                       string
	DeployURL                     string
	EvictionExpiredThresholdHours int
	PodsTable                     string

	pods []*evictionpods.Pod
}

type ProcessorConfig struct {
	WhiteList                       []string
	DeployURL                       string
	MaxEvictionExpiredPodsTableSize int
	ManualEvictionPeriod            time.Duration
	EvictionExpiredThreshold        time.Duration
}

type Processor struct {
	StageID string
	Cfg     *ProcessorConfig
	Pods    map[string]*pods.DeployPod

	data *Data
}

const TemplateName = "EvictionRequested"

var Template = template.Must(template.New(TemplateName).Parse(TemplateText))

func (p *Processor) setEvictionRequestedPodsData(pods map[string]*pods.DeployPod) {
	loc, err := time.LoadLocation("Europe/Moscow")
	if err != nil {
		panic(err)
	}

	for hostName, pod := range pods {
		evictionRequestedTS := pod.Eviction.LastUpdated.In(loc)
		evictionForcedTS := evictionRequestedTS.Add(p.Cfg.ManualEvictionPeriod)
		p.data.pods = append(p.data.pods, &evictionpods.Pod{
			HostName:              hostName,
			Cluster:               pod.Info.Cluster,
			PodID:                 pod.Info.PodID,
			DeployURL:             p.Cfg.DeployURL,
			EvictionRequestedTime: evictionRequestedTS.Format("02 Jan 15:04 MST"),
			ForcedEvictionTime:    evictionForcedTS.Format("02 Jan 15:04 MST"),
		})
	}
}

func (p *Processor) getDescription() (string, error) {
	buf := new(bytes.Buffer)
	err := Template.Execute(buf, p.data)
	if err != nil {
		return "", err
	}
	return buf.String(), nil
}

func (p *Processor) getInvocationText() string {
	text := fmt.Sprintf(
		"В стейдже обнаружено %d подов, у которых эвакуация запрошена больше %d часов назад. ",
		len(p.data.pods),
		p.data.EvictionExpiredThresholdHours,
	)

	return text
}

func (p Processor) Process() (*processors.NotificationProcessorResult, error) {
	evictionExpiredPods := map[string]*pods.DeployPod{}

	for podName, pod := range p.Pods {
		if pod.IsEvictionRequested() && pod.Eviction.LastUpdated.Add(p.Cfg.EvictionExpiredThreshold).Before(time.Now()) {
			evictionExpiredPods[podName] = pod
		}
	}

	if len(evictionExpiredPods) == 0 {
		return &processors.NotificationProcessorResult{
			NeedsRendering: false,
		}, nil
	}

	p.data = &Data{
		StageID:                       p.StageID,
		DeployURL:                     p.Cfg.DeployURL,
		EvictionExpiredThresholdHours: int(p.Cfg.EvictionExpiredThreshold.Hours()),
	}
	p.setEvictionRequestedPodsData(evictionExpiredPods)

	podsTableProcessor := evictionpods.Processor{
		Pods: p.data.pods,
		Cfg:  evictionpods.Config{MaxTableSize: p.Cfg.MaxEvictionExpiredPodsTableSize},
	}
	podsTableResult, err := podsTableProcessor.Process()
	if err != nil {
		return nil, err
	}
	p.data.PodsTable = podsTableResult.Description

	description, err := p.getDescription()
	if err != nil {
		return nil, fmt.Errorf("eviction requested template render failed: %w", err)
	}
	return &processors.NotificationProcessorResult{
		NeedsRendering:  true,
		NeedsInvocation: true,
		Name:            TemplateName,
		Header:          "Не сработало штатное переселение подов",
		Description:     description,
		InvocationText:  p.getInvocationText(),
		IsProblem:       true,
	}, nil
}

func (p *Processor) IsEnabled() bool {
	return processors.ValueInWhiteList(p.Cfg.WhiteList, p.StageID)
}
