package evictionrequested

import (
	"bytes"
	"fmt"
	"text/template"
	"time"

	"a.yandex-team.ru/infra/spnotifier/clients/nanny"
	"a.yandex-team.ru/infra/spnotifier/processors"
	"a.yandex-team.ru/infra/spnotifier/processors/common/evictionpods"
	"a.yandex-team.ru/infra/spnotifier/providers/pods"
)

const EvictionExpiredTemplateText = `В вашем (({{ .NannyURL }}/ui/#/services/catalog/{{ .ServiceID }}/ сервисе)) есть поды, которые не удалось переселить штатным способом в течение {{ .EvictionExpiredThresholdHours }} часов. По истечении приведённого ниже таймаута они будут выключены форсированно для проведения регламентных работ. Возможные причины следующие:
- Слишком много сломанных подов и не хватает бюджета на переселение
- Включено ручное подтверждение всех переселений
- Есть незавершённая выкладка (переселение работает только при отсутствии выкладки)
- В сервисе есть невыкаченный снэпшот и используется блокирующаяся об него ((https://docs.yandex-team.ru/nanny/reference/yp/replication-policy legacy-политика переселения))
{{ .PodsTable }}`

const EvictionExpiredTemplateName = "EvictionExpired"

var EvictionExpiredTemplate = template.Must(template.New(EvictionExpiredTemplateName).Parse(EvictionExpiredTemplateText))

type EvictionExpiredConfig struct {
	NannyURL                          string
	DeployURL                         string
	MaxEvictionRequestedPodsTableSize int
	EvictionExpiredThreshold          time.Duration
	ManualEvictionPeriod              time.Duration
}

type EvictionExpiredData struct {
	NannyURL                      string
	ServiceID                     string
	EvictionExpiredThresholdHours int
	PodsTable                     string

	pods []*evictionpods.Pod
}

type EvictionExpiredProcessor struct {
	Service *nanny.Service
	Cfg     EvictionExpiredConfig

	data EvictionExpiredData
}

func (p *EvictionExpiredProcessor) AddIfFits(pod *pods.NannyPod) bool {
	if pod.Eviction.LastUpdated.Add(p.Cfg.EvictionExpiredThreshold).Before(time.Now()) {
		podData := makePod(pod, p.Cfg.DeployURL, p.Cfg.ManualEvictionPeriod)
		p.data.pods = append(p.data.pods, podData)
		return true
	}

	return false
}

func (p *EvictionExpiredProcessor) getDesciption() (string, error) {
	buf := new(bytes.Buffer)
	err := EvictionExpiredTemplate.Execute(buf, p.data)
	if err != nil {
		return "", err
	}
	return buf.String(), nil
}

func (p *EvictionExpiredProcessor) getInvocationText() string {
	return fmt.Sprintf("c эвакуацией, запрошенной более %d часов назад (%d)",
		p.data.EvictionExpiredThresholdHours,
		len(p.data.pods))
}

func (p *EvictionExpiredProcessor) Process() (*processors.NotificationProcessorResult, error) {
	if len(p.data.pods) == 0 {
		return &processors.NotificationProcessorResult{NeedsRendering: false}, nil
	}

	p.data.NannyURL = p.Cfg.NannyURL
	p.data.ServiceID = p.Service.ID
	p.data.EvictionExpiredThresholdHours = int(p.Cfg.EvictionExpiredThreshold.Hours())

	podsTableProcessor := evictionpods.Processor{
		Pods: p.data.pods,
		Cfg:  evictionpods.Config{MaxTableSize: p.Cfg.MaxEvictionRequestedPodsTableSize},
	}
	podsTableResult, err := podsTableProcessor.Process()
	if err != nil {
		return nil, err
	}
	p.data.PodsTable = podsTableResult.Description

	description, err := p.getDesciption()
	if err != nil {
		return nil, fmt.Errorf("eviction expired template render failed: %w", err)
	}

	return &processors.NotificationProcessorResult{
		NeedsRendering:  true,
		NeedsInvocation: true,
		Name:            EvictionExpiredTemplateName,
		Header:          "Не сработало штатное переселение подов",
		Description:     description,
		InvocationText:  p.getInvocationText(),
		IsProblem:       true,
	}, nil
}

func (p *EvictionExpiredProcessor) IsEnabled() bool {
	return p.Service.InfoAttrs.Content.EnvType != "TESTING"
}
