package evictionrequested

import (
	"bytes"
	"fmt"
	"text/template"
	"time"

	pb "a.yandex-team.ru/infra/nanny/go/proto/nanny_repo"
	"a.yandex-team.ru/infra/spnotifier/processors"
	"a.yandex-team.ru/infra/spnotifier/processors/common/evictionpods"
	"a.yandex-team.ru/infra/spnotifier/providers/pods"
)

const HandsupPolicyTemplateText = `В (({{ .NannyURL }}/ui/#/services/catalog/{{ .ServiceID }}/ сервисе)) включен ((https://wiki.yandex-team.ru/runtime-cloud/nanny/howtos/replication-policy/#shutdown-pod-policy призыв пользователя)) при невозможности выключить под для переселения. Для следующих подов было запрошено выключение, но из-за проблем с их серверами не удалось убедиться, что оно успешно завершено:
{{ .PodsTable }}`

const HandsupPolicyTemplateName = "HandsupPolicy"

var HandsupPolicyTemplate = template.Must(template.New(HandsupPolicyTemplateName).Parse(HandsupPolicyTemplateText))

type HandsupPolicyConfig struct {
	NannyURL                          string
	DeployURL                         string
	MaxEvictionRequestedPodsTableSize int
	HandsupPolicyThreshold            time.Duration
	ManualEvictionPeriod              time.Duration
}

type HandsupPolicyData struct {
	NannyURL                    string
	ServiceID                   string
	HandsupPolicyThresholdHours int
	PodsTable                   string

	pods []*evictionpods.Pod
}

type HandsupPolicyProcessor struct {
	ServiceID               string
	Cfg                     HandsupPolicyConfig
	UncontrollablePodPolicy pb.ReplicationPolicySpec_UncontrollablePodPolicy

	data HandsupPolicyData
}

func (p *HandsupPolicyProcessor) AddIfFits(pod *pods.NannyPod) bool {
	if pod.ShutdownLock == nil || p.UncontrollablePodPolicy != pb.ReplicationPolicySpec_UPP_HANDSUP {
		return false
	}

	if pod.ShutdownLock.StartedAt.Add(p.Cfg.HandsupPolicyThreshold).Before(time.Now()) {
		podData := makePod(pod, p.Cfg.DeployURL, p.Cfg.ManualEvictionPeriod)
		p.data.pods = append(p.data.pods, podData)
		return true
	}

	return false
}

func (p *HandsupPolicyProcessor) getDesciption() (string, error) {
	buf := new(bytes.Buffer)
	err := HandsupPolicyTemplate.Execute(buf, p.data)
	if err != nil {
		return "", err
	}
	return buf.String(), nil
}

func (p *HandsupPolicyProcessor) getInvocationText() string {
	return fmt.Sprintf("ожидающие подтверждения эвакуации при включенной политике HANDSUP (%d)",
		len(p.data.pods))
}

func (p *HandsupPolicyProcessor) Process() (*processors.NotificationProcessorResult, error) {
	if len(p.data.pods) == 0 {
		return &processors.NotificationProcessorResult{NeedsRendering: false}, nil
	}

	p.data.NannyURL = p.Cfg.NannyURL
	p.data.ServiceID = p.ServiceID
	p.data.HandsupPolicyThresholdHours = int(p.Cfg.HandsupPolicyThreshold.Hours())

	podsTableProcessor := evictionpods.Processor{
		Pods: p.data.pods,
		Cfg:  evictionpods.Config{MaxTableSize: p.Cfg.MaxEvictionRequestedPodsTableSize},
	}
	podsTableResult, err := podsTableProcessor.Process()
	if err != nil {
		return nil, err
	}
	p.data.PodsTable = podsTableResult.Description

	description, err := p.getDesciption()
	if err != nil {
		return nil, fmt.Errorf("handsup policy template render failed: %w", err)
	}

	return &processors.NotificationProcessorResult{
		NeedsRendering:  true,
		NeedsInvocation: true,
		Name:            HandsupPolicyTemplateName,
		Header:          "Поды не удалось выключить для эвакуации",
		Description:     description,
		InvocationText:  p.getInvocationText(),
		IsProblem:       true,
	}, nil
}

func (p *HandsupPolicyProcessor) IsEnabled() bool {
	return true
}
