package evictionrequested

import (
	"bytes"
	"fmt"
	"text/template"
	"time"

	"a.yandex-team.ru/infra/spnotifier/processors"
	"a.yandex-team.ru/infra/spnotifier/processors/common/evictionpods"
	"a.yandex-team.ru/infra/spnotifier/providers/pods"
)

const NotificationPolicyTemplateText = `В вашем сервисе (({{ .NannyURL }}/ui/#/services/catalog/{{ .ServiceID }}/replication_policies включена политика уведомлений)) о подах, на которых запрошено переселение. Обычно эти нотификации включают, если переселение подов требует ручных действий, или чтобы просто быть в курсе о них. Если вам не нужны эти уведомления и ручное подтверждение переселения, вы всегда можете отключить их по ((https://wiki.yandex-team.ru/runtime-cloud/nanny/howtos/replication-policy/#notification-policy инструкции)).
{{ .PodsTable }}`

const NotificationPolicyTemplateName = "NotificationPolicy"

var NotificationPolicyTemplate = template.Must(template.New(NotificationPolicyTemplateName).Parse(NotificationPolicyTemplateText))

type NotificationPolicyConfig struct {
	NannyURL                          string
	DeployURL                         string
	MaxEvictionRequestedPodsTableSize int
	ManualEvictionPeriod              time.Duration
}

type NotificationPolicyData struct {
	NannyURL  string
	ServiceID string
	PodsTable string

	pods []*evictionpods.Pod
}

type NotificationPolicyProcessor struct {
	ServiceID            string
	Cfg                  NotificationPolicyConfig
	NotificationsEnabled bool

	data NotificationPolicyData
}

func (p *NotificationPolicyProcessor) AddIfFits(pod *pods.NannyPod) bool {
	if p.NotificationsEnabled {
		podData := makePod(pod, p.Cfg.DeployURL, p.Cfg.ManualEvictionPeriod)
		p.data.pods = append(p.data.pods, podData)
		return true
	}

	return false
}

func (p *NotificationPolicyProcessor) getDesciption() (string, error) {
	buf := new(bytes.Buffer)
	err := NotificationPolicyTemplate.Execute(buf, p.data)
	if err != nil {
		return "", err
	}
	return buf.String(), nil
}

func (p *NotificationPolicyProcessor) getInvocationText() string {
	return fmt.Sprintf("по причине ((%s/ui/#/services/catalog/%s/replication_policies включенных уведомлений)) (%d)",
		p.Cfg.NannyURL,
		p.ServiceID,
		len(p.data.pods))
}

func (p *NotificationPolicyProcessor) Process() (*processors.NotificationProcessorResult, error) {
	if len(p.data.pods) == 0 {
		return &processors.NotificationProcessorResult{NeedsRendering: false}, nil
	}

	p.data.NannyURL = p.Cfg.NannyURL
	p.data.ServiceID = p.ServiceID

	podsTableProcessor := evictionpods.Processor{
		Pods: p.data.pods,
		Cfg:  evictionpods.Config{MaxTableSize: p.Cfg.MaxEvictionRequestedPodsTableSize},
	}
	podsTableResult, err := podsTableProcessor.Process()
	if err != nil {
		return nil, err
	}
	p.data.PodsTable = podsTableResult.Description

	description, err := p.getDesciption()
	if err != nil {
		return nil, fmt.Errorf("notification policy template render failed: %w", err)
	}

	return &processors.NotificationProcessorResult{
		NeedsRendering:  true,
		NeedsInvocation: true,
		Name:            NotificationPolicyTemplateName,
		Header:          "Запрошено, но ещё не просрочено переселение подов",
		Description:     description,
		InvocationText:  p.getInvocationText(),
		IsProblem:       false,
	}, nil
}

func (p *NotificationPolicyProcessor) IsEnabled() bool {
	return true
}
