package evictionrequested

import (
	"bytes"
	"fmt"
	"strings"
	"text/template"
	"time"

	pb "a.yandex-team.ru/infra/nanny/go/proto/nanny_repo"
	"a.yandex-team.ru/infra/spnotifier/clients/nanny"
	"a.yandex-team.ru/infra/spnotifier/processors"
	"a.yandex-team.ru/infra/spnotifier/processors/common/evictionpods"
	"a.yandex-team.ru/infra/spnotifier/providers/pods"
)

const TemplateText = `{{ range .Results }}
====={{ .Header }}
{{ .Description }}
{{ end }}`

type EvictionRequestedProcessor interface {
	processors.NotificationProcessor
	AddIfFits(*pods.NannyPod) bool
}

type Data struct {
	Results []*processors.NotificationProcessorResult
}

type ProcessorConfig struct {
	WhiteList                         []string
	DeployURL                         string
	NannyURL                          string
	MaxEvictionRequestedPodsTableSize int
	ManualEvictionPeriod              time.Duration
	EvictionExpiredThreshold          time.Duration
	HandsupExpiredThreshold           time.Duration
}

type Processor struct {
	Service           *nanny.Service
	Cfg               *ProcessorConfig
	Pods              map[string]*pods.NannyPod
	ReplicationPolicy *pb.ReplicationPolicy

	data *Data
}

const TemplateName = "EvictionRequested"

var Template = template.Must(template.New(TemplateName).Parse(TemplateText))

func makePod(pod *pods.NannyPod, deployURL string, manualEvictionPeriod time.Duration) *evictionpods.Pod {
	loc, err := time.LoadLocation("Europe/Moscow")
	if err != nil {
		panic(err)
	}

	evictionRequestedTS := pod.Eviction.LastUpdated.In(loc)
	evictionForcedTS := evictionRequestedTS.Add(manualEvictionPeriod)
	return &evictionpods.Pod{
		HostName:              pod.Info.HostName,
		Cluster:               pod.Info.Cluster,
		PodID:                 pod.Info.PodID,
		DeployURL:             deployURL,
		EvictionRequestedTime: evictionRequestedTS.Format("02 Jan 15:04 MST"),
		ForcedEvictionTime:    evictionForcedTS.Format("02 Jan 15:04 MST"),
	}
}

func (p *Processor) getDescription() (string, error) {
	buf := new(bytes.Buffer)
	err := Template.Execute(buf, p.data)
	if err != nil {
		return "", err
	}
	return buf.String(), nil
}

func (p *Processor) getInvocationText() string {
	invocations := []string{}
	for _, result := range p.data.Results {
		invocations = append(invocations, result.InvocationText)
	}
	text := fmt.Sprintf(
		"В сервисе обнаружены поды с запрошенной эвакуацией:\n - %s",
		strings.Join(invocations, "\n - "),
	)
	return text
}

func (p Processor) Process() (*processors.NotificationProcessorResult, error) {
	notificationsEnabled := false
	replicationPolicyAnnotations := p.ReplicationPolicy.GetMeta().GetAnnotations()
	if replicationPolicyAnnotations != nil {
		if v, ok := replicationPolicyAnnotations["notification_policy"]; ok && v == "enabled" {
			notificationsEnabled = true
		}
	}
	uncontrollablePodPolicy := p.ReplicationPolicy.GetSpec().GetUncontrollablePodPolicy()

	evictionProcessors := []EvictionRequestedProcessor{
		// order from most specific to least specific
		&HandsupPolicyProcessor{
			ServiceID: p.Service.ID,
			Cfg: HandsupPolicyConfig{
				NannyURL:                          p.Cfg.NannyURL,
				DeployURL:                         p.Cfg.DeployURL,
				MaxEvictionRequestedPodsTableSize: p.Cfg.MaxEvictionRequestedPodsTableSize,
				HandsupPolicyThreshold:            p.Cfg.HandsupExpiredThreshold,
				ManualEvictionPeriod:              p.Cfg.ManualEvictionPeriod,
			},
			UncontrollablePodPolicy: uncontrollablePodPolicy,
		},
		&EvictionExpiredProcessor{
			Service: p.Service,
			Cfg: EvictionExpiredConfig{
				NannyURL:                          p.Cfg.NannyURL,
				DeployURL:                         p.Cfg.DeployURL,
				MaxEvictionRequestedPodsTableSize: p.Cfg.MaxEvictionRequestedPodsTableSize,
				ManualEvictionPeriod:              p.Cfg.ManualEvictionPeriod,
				EvictionExpiredThreshold:          p.Cfg.EvictionExpiredThreshold,
			},
		},
		&NotificationPolicyProcessor{
			ServiceID: p.Service.ID,
			Cfg: NotificationPolicyConfig{
				NannyURL:                          p.Cfg.NannyURL,
				DeployURL:                         p.Cfg.DeployURL,
				MaxEvictionRequestedPodsTableSize: p.Cfg.MaxEvictionRequestedPodsTableSize,
				ManualEvictionPeriod:              p.Cfg.ManualEvictionPeriod,
			},
			NotificationsEnabled: notificationsEnabled,
		},
	}

	for _, pod := range p.Pods {
		if !pod.IsEvictionRequested() {
			continue
		}

		for _, processor := range evictionProcessors {
			if !processor.IsEnabled() {
				continue
			}
			if processor.AddIfFits(pod) {
				break
			}
		}

	}

	p.data = &Data{}
	needsRendering := false
	isProblem := false
	for _, processor := range evictionProcessors {
		result, err := processor.Process()
		if err != nil {
			return nil, fmt.Errorf("eviction expired processor failed: %w", err)
		}
		if result.NeedsRendering {
			needsRendering = true
			p.data.Results = append(p.data.Results, result)
		}
		if result.IsProblem {
			isProblem = true
		}
	}

	if !needsRendering {
		return &processors.NotificationProcessorResult{
			NeedsRendering: false,
		}, nil
	}

	description, err := p.getDescription()
	if err != nil {
		return nil, fmt.Errorf("eviction requested template render failed: %w", err)
	}
	return &processors.NotificationProcessorResult{
		NeedsRendering:  true,
		NeedsInvocation: true,
		Name:            TemplateName,
		Header:          "Поды с запрошенной эвакуацией",
		Description:     description,
		InvocationText:  p.getInvocationText(),
		IsProblem:       isProblem,
	}, nil
}

func (p *Processor) IsEnabled() bool {
	return processors.ValueInWhiteList(p.Cfg.WhiteList, p.Service.ID)
}
