package processors

import (
	"bytes"
	"fmt"
	"regexp"
	"sort"
	"strings"
	"text/template"
)

type NotificationProcessorResult struct {
	Name            string
	Header          string
	Description     string
	InvocationText  string
	NeedsRendering  bool
	NeedsInvocation bool
	IsProblem       bool
}

type NotificationProcessor interface {
	Process() (*NotificationProcessorResult, error)
	IsEnabled() bool
}

func ValueInWhiteList(wl []string, value string) bool {
	if len(wl) == 0 {
		return true
	}

	for _, r := range wl {
		matched, err := regexp.MatchString(r, value)
		if err != nil {
			panic(fmt.Errorf("regexp %s matching failed: %w", r, err))
		}
		if matched {
			return true
		}
	}

	return false
}

const TemplateText = `===={{ .Header }}
{{ range $idx, $result := .Results }}
	<a href="#{{ .Name }}">{{- inc $idx }}. {{ .Header }}</a>
{{ end }}
Детали приведены ниже
{{ range .Results }}
	===<p id="{{ .Name }}">{{- .Header}}<p>
	{{ .Description }}
{{ end }}

==== Мне не нравятся эти тикеты
Мы начали внедрять инфраструктуру подобных уведомлений не так давно, она может быть несовершенна и мы с радостью её улучшим. Если кажется, что наши критерии сломанности подов слишком жесткие, нотификации бесят, инструкция непонятная или что-то ещё плохо - оставьте, пожалуйста, конструктивный наброс комментом к этому тикету и призовите дежурного, нажав на кнопку "Призвать дежурного"`

var templateFuncMap = template.FuncMap{
	"inc": func(i int) int {
		return i + 1
	},
}

var Template = template.Must(template.New("General").Funcs(templateFuncMap).Parse(TemplateText))

type TemplateData struct {
	Results []*NotificationProcessorResult
	Header  string
}

type Processor struct {
	Processors []NotificationProcessor
	Header     string
	Title      string
}

func (p *Processor) getInvocationText(results []*NotificationProcessorResult) string {
	var invocationTexts []string
	for _, result := range results {
		invocationTexts = append(invocationTexts, result.InvocationText)
	}
	return strings.Join(invocationTexts, "\n\n")
}

func (p *Processor) getDesciption(results []*NotificationProcessorResult) (string, error) {
	buf := new(bytes.Buffer)
	err := Template.Execute(buf, TemplateData{
		Results: results,
		Header:  p.Header,
	})
	if err != nil {
		return "", err
	}
	return buf.String(), nil
}

func (p *Processor) Process() (*NotificationProcessorResult, error) {
	results := []*NotificationProcessorResult{}

	needsInvocation := false
	isProblem := false
	for _, processor := range p.Processors {
		result, err := processor.Process()
		if err != nil {
			return nil, err
		}
		if result.NeedsRendering {
			results = append(results, result)
		}
		if result.NeedsInvocation {
			needsInvocation = true
		}
		if result.IsProblem {
			isProblem = true
		}
	}

	if len(results) == 0 {
		return &NotificationProcessorResult{
			NeedsRendering: false,
		}, nil
	}

	sort.Slice(results, func(i, j int) bool {
		return results[i].Name < results[j].Name
	})

	description, err := p.getDesciption(results)
	if err != nil {
		return nil, fmt.Errorf("template render failed: %w", err)
	}

	return &NotificationProcessorResult{
		NeedsRendering:  true,
		NeedsInvocation: needsInvocation,
		Header:          p.Title,
		Description:     description,
		InvocationText:  p.getInvocationText(results),
		IsProblem:       isProblem,
	}, nil
}

func (p *Processor) IsEnabled() bool {
	return true
}
