package pods

import (
	"errors"
	"fmt"

	"google.golang.org/protobuf/proto"

	"a.yandex-team.ru/infra/spnotifier/clients/yp"
	"a.yandex-team.ru/yp/go/proto/ypapi"
	"a.yandex-team.ru/yt/go/yson"
)

type DeployPodInfo struct {
	PodID    string
	Cluster  string
	NodeID   string
	HostName string
}

type DeployPod struct {
	StageID     string
	Info        *DeployPodInfo
	Eviction    *Eviction
	Maintenance *Maintenance
}

func (p *DeployPod) IsEvictionRequested() bool {
	return isEvictionRequested(p.Eviction, p.Maintenance)
}

type DeployLables struct {
	StageID string `yson:"stage_id"`
}

func getStageID(ypPod *ypapi.TPod) (*string, error) {
	labels := ypPod.GetLabels().GetAttributes()
	for _, label := range labels {
		if *label.Key != "deploy" {
			continue
		}

		var deployLabels DeployLables
		err := yson.Unmarshal(label.Value, &deployLabels)
		if err != nil {
			return nil, errors.New("cannot decode deploy label")
		}

		if deployLabels.StageID == "" {
			return nil, errors.New("deploy/stage_id is empty")
		}

		return &deployLabels.StageID, nil
	}

	return nil, errors.New("label deploy/stage_id doesn't exist")
}

func makeDeployPod(ypPod *yp.YpPod) (*DeployPod, error) {
	stageID, err := getStageID(ypPod.Pod)
	if err != nil {
		return nil, err
	}

	pod := DeployPod{
		Info: &DeployPodInfo{
			PodID:   ypPod.Pod.Meta.Id,
			Cluster: ypPod.Cluster,
			NodeID:  *ypPod.Pod.Spec.NodeId,
		},
		Eviction:    getPodEviction(ypPod.Pod),
		Maintenance: getPodMaintenance(ypPod.Pod),
		StageID:     *stageID,
	}
	pod.Info.HostName = fmt.Sprintf("%s.%s.yp-c.yandex.net", pod.Info.PodID, pod.Info.Cluster)

	return &pod, nil
}

type DeployClusterSnapshot struct {
	pods map[string]map[string]*DeployPod
	ch   chan *yp.YpPod
}

func (s *DeployClusterSnapshot) GetFilter() string {
	return "string([/labels/deploy_engine]) IN (\"MCRSC\", \"RSC\")"
}

func (s *DeployClusterSnapshot) GetSelectors() []string {
	return []string{
		"/labels",
		"/meta",
		"/status/eviction",
		"/status/maintenance",
		"/status/scheduling",
		"/status/iss_conf_summaries",
		"/spec/node_id",
	}
}

func (s *DeployClusterSnapshot) FillPod(pod *yp.YpPod) error {
	deployPod, err := makeDeployPod(pod)
	if err != nil {
		return err
	}
	if deployPod == nil {
		// skip without error
		return nil
	}
	podsMap, ok := s.pods[deployPod.StageID]
	if !ok {
		podsMap = map[string]*DeployPod{}
		s.pods[deployPod.StageID] = podsMap
	}
	podsMap[deployPod.Info.HostName] = deployPod
	return nil
}

func (s *DeployClusterSnapshot) AddItem(m proto.Message, cluster string) error {
	pod, ok := m.(*ypapi.TPod)
	if !ok {
		return fmt.Errorf("failed to cast proto message to TPod")
	}
	ypPod := yp.YpPod{Pod: pod, Cluster: cluster}
	s.ch <- &ypPod

	return nil
}

func (s *DeployClusterSnapshot) MakeEmptyItem() proto.Message {
	return &ypapi.TPod{}
}

func (s *DeployClusterSnapshot) GetChan() chan *yp.YpPod {
	return s.ch
}
