package pods

import (
	"errors"
	"fmt"
	"time"

	"google.golang.org/protobuf/proto"

	"a.yandex-team.ru/infra/spnotifier/clients/yp"
	"a.yandex-team.ru/yp/go/proto/ypapi"
	"a.yandex-team.ru/yt/go/yson"
)

var ActiveState = "ACTIVE"

type NannyPodInfo struct {
	PodID    string
	Cluster  string
	NodeID   string
	Itype    string
	HostName string
}

type ShutdownLock struct {
	StartedAt time.Time
}

type NannyPod struct {
	ServiceID    string
	State        string
	Info         *NannyPodInfo
	Eviction     *Eviction
	Maintenance  *Maintenance
	ShutdownLock *ShutdownLock
}

func (p *NannyPod) IsEvictionRequested() bool {
	return isEvictionRequested(p.Eviction, p.Maintenance)
}

func getPodTargetActiveConf(ypPod *ypapi.TPod) (*string, error) {
	for instance, summary := range ypPod.GetStatus().GetIssConfSummaries() {
		if *summary.TargetState == ActiveState {
			return &instance, nil
		}
	}

	return nil, errors.New("don't track pods without targetState = ACTIVE")

}

type ShutdownLockLabel struct {
	StartedAt int64 `yson:"started_at"`
}

func getPodShutdownLock(ypPod *ypapi.TPod) (*ShutdownLock, error) {
	labels := ypPod.GetLabels().GetAttributes()
	for _, label := range labels {
		if *label.Key != "yp_lite_shutdown_lock" {
			continue
		}

		var shutdownLock ShutdownLockLabel
		if err := yson.Unmarshal(label.Value, &shutdownLock); err != nil {
			return nil, errors.New("cannot unmarshal shutdown lock")
		}

		if shutdownLock.StartedAt != 0 {
			return &ShutdownLock{
				StartedAt: time.Unix(shutdownLock.StartedAt, 0),
			}, nil
		}
	}

	return nil, nil
}

func makeNannyPod(ypPod *yp.YpPod) (*NannyPod, error) {
	labels := ypPod.Pod.GetLabels().GetAttributes()
	serviceID := ""
	for _, label := range labels {
		if *label.Key == "nanny_service_id" {
			if err := yson.Unmarshal(label.Value, &serviceID); err != nil {
				return nil, fmt.Errorf("cannot unmarshal nanny_service_id label: %w", err)
			}
		}
	}
	if serviceID == "" {
		return nil, errors.New("label nanny_service_id doesn't exist or it is empty")
	}

	targetActiveConf, err := getPodTargetActiveConf(ypPod.Pod)
	if err != nil {
		// skip this pod
		return nil, nil
	}

	summaries := ypPod.Pod.GetStatus().GetAgent().GetIssSummary().GetStateSummaries()
	if summaries == nil {
		return nil, errors.New("summaries field is empty")
	}
	instanceSummary, ok := summaries[*targetActiveConf]
	if !ok || instanceSummary == nil {
		return nil, errors.New("instance summary value is empty or doesn't exist")
	}

	shutdownLock, err := getPodShutdownLock(ypPod.Pod)
	if err != nil {
		return nil, err
	}

	itype := "unknown"

	pod := NannyPod{
		Info: &NannyPodInfo{
			PodID:   ypPod.Pod.Meta.Id,
			Cluster: ypPod.Cluster,
			NodeID:  *ypPod.Pod.Spec.NodeId,
			Itype:   itype,
		},
		State:        *instanceSummary.CurrentState,
		Eviction:     getPodEviction(ypPod.Pod),
		Maintenance:  getPodMaintenance(ypPod.Pod),
		ServiceID:    serviceID,
		ShutdownLock: shutdownLock,
	}
	pod.Info.HostName = fmt.Sprintf("%s.%s.yp-c.yandex.net", pod.Info.PodID, pod.Info.Cluster)

	return &pod, nil
}

type NannyClusterSnapshot struct {
	pods map[string]map[string]*NannyPod
	ch   chan *yp.YpPod
}

func (s *NannyClusterSnapshot) GetFilter() string {
	return "[/labels/deploy_engine] = \"YP_LITE\""
}

func (s *NannyClusterSnapshot) GetSelectors() []string {
	return []string{
		"/labels",
		"/meta",
		"/status/eviction",
		"/status/maintenance",
		"/status/scheduling",
		"/status/iss_conf_summaries",
		"/status/agent/iss_summary",
		"/spec/node_id",
	}
}

func (s *NannyClusterSnapshot) FillPod(pod *yp.YpPod) error {
	nannyPod, err := makeNannyPod(pod)
	if err != nil {
		return err
	}
	if nannyPod == nil {
		// skip without error
		return nil
	}
	podsMap, ok := s.pods[nannyPod.ServiceID]
	if !ok {
		podsMap = map[string]*NannyPod{}
		s.pods[nannyPod.ServiceID] = podsMap
	}
	podsMap[nannyPod.Info.HostName] = nannyPod
	return nil
}

func (s *NannyClusterSnapshot) AddItem(m proto.Message, cluster string) error {
	pod, ok := m.(*ypapi.TPod)
	if !ok {
		return fmt.Errorf("failed to cast proto message to TPod")
	}
	ypPod := yp.YpPod{Pod: pod, Cluster: cluster}
	s.ch <- &ypPod

	return nil
}

func (s *NannyClusterSnapshot) MakeEmptyItem() proto.Message {
	return &ypapi.TPod{}
}

func (s *NannyClusterSnapshot) GetChan() chan *yp.YpPod {
	return s.ch
}
