package pods

import (
	"context"
	"fmt"
	"log"
	"time"

	"golang.org/x/sync/errgroup"

	"a.yandex-team.ru/infra/spnotifier/clients/yp"
	"a.yandex-team.ru/yp/go/proto/ypapi"
)

type ClusterSnapshot interface {
	yp.ListObjectsReciever
	FillPod(*yp.YpPod) error
	GetSelectors() []string
	GetFilter() string
	GetChan() chan *yp.YpPod
}

type PodsData struct {
	// service: HostName: Pod
	Nanny  map[string]map[string]*NannyPod
	Deploy map[string]map[string]*DeployPod
}

type ProviderConfig struct {
	BatchSize        int
	Clusters         []string
	YpRequestTimeout time.Duration
}

type Provider struct {
	Cfg     *ProviderConfig
	YpToken string
}

func (p *Provider) selectPods(ctx context.Context, client *yp.YpClient, snapshot ClusterSnapshot) error {
	errg, ctx := errgroup.WithContext(ctx)

	for i := range p.Cfg.Clusters {
		cluster := p.Cfg.Clusters[i] // to use in closure
		errg.Go(func() error {
			return client.ListObjects(
				ctx,
				&yp.ListObjectsOptions{
					Selectors:  snapshot.GetSelectors(),
					Filter:     snapshot.GetFilter(),
					Cluster:    cluster,
					ObjectType: ypapi.EObjectType_OT_POD,
				},
				snapshot,
			)
		})
	}

	go func() {
		_ = errg.Wait()
		close(snapshot.GetChan())
	}()

	for pod := range snapshot.GetChan() {
		err := snapshot.FillPod(pod)
		if err != nil {
			log.Printf("skipping pod %s: %v", pod.Pod.GetMeta().GetId(), err)
			continue
		}
	}

	return errg.Wait()
}

func (p *Provider) GetPods(ctx context.Context) (*PodsData, error) {
	client, err := yp.NewClient(p.Cfg.Clusters, int32(p.Cfg.BatchSize), p.Cfg.YpRequestTimeout, p.YpToken)
	if err != nil {
		return nil, fmt.Errorf("yp client creation failed: %w", err)
	}
	defer client.Close()

	errg, ctx := errgroup.WithContext(ctx)

	nannyChan := make(chan map[string]map[string]*NannyPod)
	errg.Go(func() error {
		defer close(nannyChan)

		snapshot := &NannyClusterSnapshot{
			pods: map[string]map[string]*NannyPod{},
			ch:   make(chan *yp.YpPod),
		}
		err := p.selectPods(ctx, client, snapshot)
		if err != nil {
			return fmt.Errorf("nanny pods request failed: %w", err)
		}
		nannyChan <- snapshot.pods
		return nil
	})
	deployChan := make(chan map[string]map[string]*DeployPod)
	errg.Go(func() error {
		defer close(deployChan)

		snapshot := &DeployClusterSnapshot{
			pods: map[string]map[string]*DeployPod{},
			ch:   make(chan *yp.YpPod),
		}
		err := p.selectPods(ctx, client, snapshot)
		if err != nil {
			return fmt.Errorf("deploy pods request failed: %w", err)
		}
		deployChan <- snapshot.pods
		return nil
	})

	nannyPods := <-nannyChan
	deployPods := <-deployChan

	if err := errg.Wait(); err != nil {
		return nil, err
	}

	return &PodsData{
		Nanny:  nannyPods,
		Deploy: deployPods,
	}, nil
}
