package stages

import (
	"context"
	"fmt"
	"time"

	"golang.org/x/sync/errgroup"
	"google.golang.org/protobuf/proto"

	"a.yandex-team.ru/infra/spnotifier/clients/yp"
	"a.yandex-team.ru/yp/go/proto/ypapi"
)

type Project struct {
	ID        string
	AccountID string
	OwnerID   string
}

type Stage struct {
	ID        string
	ProjectID string
	AccountID string
	OwnerID   string
}

type ProviderConfig struct {
	BatchSize        int
	YpRequestTimeout time.Duration
}

type Provider struct {
	Cfg     *ProviderConfig
	YpToken string
}

type StagesSnapshot struct {
	stages map[string]*Stage
}

func (r *StagesSnapshot) MakeEmptyItem() proto.Message {
	return &ypapi.TStage{}
}

func (r *StagesSnapshot) AddItem(m proto.Message, cluster string) error {
	ypStage, ok := m.(*ypapi.TStage)
	if !ok {
		return fmt.Errorf("failed to cast proto message to TStage")
	}
	stage := &Stage{
		ID:        ypStage.GetMeta().GetId(),
		ProjectID: ypStage.GetMeta().GetProjectId(),
	}
	r.stages[stage.ID] = stage

	return nil
}

type ProjectsSnapshot struct {
	projects map[string]*Project
}

func (r *ProjectsSnapshot) MakeEmptyItem() proto.Message {
	return &ypapi.TProject{}
}

func (r *ProjectsSnapshot) AddItem(m proto.Message, cluster string) error {
	ypProject, ok := m.(*ypapi.TProject)
	if !ok {
		return fmt.Errorf("failed to cast proto message to TProject")
	}
	project := &Project{
		ID:        ypProject.GetMeta().GetId(),
		AccountID: ypProject.GetSpec().GetAccountId(),
		OwnerID:   ypProject.GetMeta().GetOwnerId(),
	}
	r.projects[project.ID] = project

	return nil
}

func (p *Provider) GetStages(ctx context.Context) (map[string]*Stage, error) {
	client, err := yp.NewClient([]string{"xdc"}, int32(p.Cfg.BatchSize), p.Cfg.YpRequestTimeout, p.YpToken)
	if err != nil {
		return nil, fmt.Errorf("yp client creation failed: %w", err)
	}
	defer client.Close()

	errg, ctx := errgroup.WithContext(ctx)

	projectsChan := make(chan map[string]*Project)
	errg.Go(func() error {
		defer close(projectsChan)

		snapshot := ProjectsSnapshot{
			projects: make(map[string]*Project),
		}
		err := client.ListObjects(ctx,
			&yp.ListObjectsOptions{
				Selectors: []string{
					"/meta/id",
					"/meta/owner_id",
					"/spec/account_id",
				},
				Cluster:    "xdc",
				ObjectType: ypapi.EObjectType_OT_PROJECT,
			},
			&snapshot)

		if err != nil {
			return fmt.Errorf("projects request failed: %w", err)
		}

		projectsChan <- snapshot.projects
		return nil
	})

	stagesChan := make(chan map[string]*Stage)
	errg.Go(func() error {
		defer close(stagesChan)

		snapshot := StagesSnapshot{
			stages: make(map[string]*Stage),
		}
		err := client.ListObjects(ctx,
			&yp.ListObjectsOptions{
				Selectors: []string{
					"/meta/id",
					"/meta/project_id",
				},
				Cluster:    "xdc",
				ObjectType: ypapi.EObjectType_OT_STAGE,
			},
			&snapshot)

		if err != nil {
			return fmt.Errorf("stages request failed: %w", err)
		}

		stagesChan <- snapshot.stages
		return nil
	})

	stages := <-stagesChan
	projects := <-projectsChan

	if err := errg.Wait(); err != nil {
		return nil, err
	}

	for _, stage := range stages {
		project, ok := projects[stage.ProjectID]
		if !ok {
			return nil, fmt.Errorf("no project with id %s found", stage.ProjectID)
		}
		stage.AccountID = project.AccountID
		stage.OwnerID = project.OwnerID
	}

	return stages, nil
}
