package states

import (
	"context"
	"encoding/json"
	"fmt"

	"golang.org/x/sync/errgroup"

	"a.yandex-team.ru/yt/go/yt"
	"a.yandex-team.ru/yt/go/yt/ytrpc"
)

type ProviderConfig struct {
	Proxy      string
	NannyPath  string
	DeployPath string
}

type Provider struct {
	Cfg     *ProviderConfig
	YtToken string
}

type States struct {
	Nanny  map[string]*NannyState  // key: serviceID
	Deploy map[string]*DeployState // key: stageID
}

type TableRow interface {
	GetState() []byte
	GetID() string
}

type Snapshot interface {
	GetPath() string
	MakeTableRow() TableRow
	MakeItem() interface{}
	AddItem(id string, i interface{}) error
}

func (p *Provider) getStatesSnapshot(ctx context.Context, client *yt.Client, snapshot Snapshot) error {
	offset := 0
	limit := 1000
	for {
		reader, err := (*client).SelectRows(ctx, fmt.Sprintf("* from [%s] offset %d limit %d", snapshot.GetPath(), offset, limit), nil)
		if err != nil {
			return fmt.Errorf("cannot select rows from %s: %w", snapshot.GetPath(), err)
		}

		emptyResponse := true
		for reader.Next() {
			emptyResponse = false
			r := snapshot.MakeTableRow()
			err = reader.Scan(r)
			if err != nil {
				return fmt.Errorf("cannot scan one of rows: %w", err)
			}

			i := snapshot.MakeItem()
			if err := json.Unmarshal(r.GetState(), i); err != nil {
				return fmt.Errorf("unmarshal for state failed: %w", err)
			}
			if err := snapshot.AddItem(r.GetID(), i); err != nil {
				return err
			}
		}

		if emptyResponse {
			break
		}

		offset += limit
	}

	return nil
}

func (p *Provider) GetStates(ctx context.Context) (*States, error) {
	client, err := ytrpc.NewClient(&yt.Config{
		Proxy: p.Cfg.Proxy,
		Token: p.YtToken,
	})
	if err != nil {
		return nil, fmt.Errorf("yt client creation failed: %w", err)
	}

	errg, ctx := errgroup.WithContext(ctx)

	nannyChan := make(chan map[string]*NannyState)
	errg.Go(func() error {
		defer close(nannyChan)

		snapshot := NannySnapshot{
			path:   p.Cfg.NannyPath,
			states: make(map[string]*NannyState),
		}
		if err := p.getStatesSnapshot(ctx, &client, &snapshot); err != nil {
			return fmt.Errorf("[nanny] cannot get states: %w", err)
		}
		nannyChan <- snapshot.states
		return nil
	})

	deployChan := make(chan map[string]*DeployState)
	errg.Go(func() error {
		defer close(deployChan)

		snapshot := DeploySnapshot{
			path:   p.Cfg.DeployPath,
			states: make(map[string]*DeployState),
		}
		if err := p.getStatesSnapshot(ctx, &client, &snapshot); err != nil {
			return fmt.Errorf("[deploy] cannot get states: %w", err)
		}

		deployChan <- snapshot.states
		return nil
	})

	nannyStates := <-nannyChan
	deployStates := <-deployChan

	if err := errg.Wait(); err != nil {
		return nil, err
	}

	return &States{
		Nanny:  nannyStates,
		Deploy: deployStates,
	}, nil
}
