package nanny

import (
	"fmt"
	"time"

	pb "a.yandex-team.ru/infra/nanny/go/proto/nanny_repo"
	"a.yandex-team.ru/infra/spnotifier/clients/nanny"
	"a.yandex-team.ru/infra/spnotifier/processors"
	"a.yandex-team.ru/infra/spnotifier/processors/nanny/evictionrequested"
	"a.yandex-team.ru/infra/spnotifier/providers/pods"
	"a.yandex-team.ru/yp/go/proto/ypapi"
)

type EvictionRequestedTestCase struct{}

func (t *EvictionRequestedTestCase) GetName() string {
	return "nanny_evictionrequested"
}

func CreateEvictionRequestedPod(name string) *pods.NannyPod {
	return &pods.NannyPod{
		ServiceID: "test",
		State:     "ACTIVE",
		Info:      &pods.NannyPodInfo{HostName: name},
		Eviction: &pods.Eviction{
			State: ypapi.EEvictionState_ES_REQUESTED,
		},
	}
}

func CreateEvictionExpiredPod(name string) *pods.NannyPod {
	pod := CreateEvictionRequestedPod(name)
	pod.Eviction.LastUpdated = time.Now().Add(-time.Hour * 48)
	return pod
}

func CreateHandupPolicyPod(name string) *pods.NannyPod {
	pod := CreateEvictionRequestedPod(name)
	pod.Eviction.LastUpdated = time.Now().Add(-time.Hour * 3)
	pod.ShutdownLock = &pods.ShutdownLock{
		StartedAt: time.Now().Add(-time.Hour * 2),
	}
	return pod
}

func CreateNotificationPolicyPod(name string) *pods.NannyPod {
	pod := CreateEvictionRequestedPod(name)
	pod.Eviction.LastUpdated = time.Now().Add(-time.Minute)
	return pod
}

func GetReplicationPolicy() *pb.ReplicationPolicy {
	return &pb.ReplicationPolicy{
		Meta: &pb.ReplicationPolicyMeta{
			Annotations: map[string]string{},
		},
		Spec: &pb.ReplicationPolicySpec{},
	}
}

func PatchReplicationPolicyNotificationsEnabled(policy *pb.ReplicationPolicy) *pb.ReplicationPolicy {
	policy.Meta.Annotations["notification_policy"] = "enabled"
	return policy
}

func PatchReplicationPolicyHandsup(policy *pb.ReplicationPolicy) *pb.ReplicationPolicy {
	policy.Spec.UncontrollablePodPolicy = pb.ReplicationPolicySpec_UPP_HANDSUP
	return policy
}

func (t *EvictionRequestedTestCase) Run() (*processors.NotificationProcessorResult, error) {
	pods := map[string]*pods.NannyPod{}
	i := 1
	for i <= 4 {
		name := fmt.Sprintf("pod-%d", i)
		pods[name] = CreateEvictionExpiredPod(name)
		i += 1
	}
	for i <= 8 {
		name := fmt.Sprintf("pod-%d", i)
		pods[name] = CreateHandupPolicyPod(name)
		i += 1
	}
	for i <= 12 {
		name := fmt.Sprintf("pod-%d", i)
		pods[name] = CreateNotificationPolicyPod(name)
		i += 1
	}

	p := evictionrequested.Processor{
		Service: &nanny.Service{
			ID: "test",
			InfoAttrs: &nanny.InfoAttrsContent{Content: &nanny.InfoAttrs{
				EnvType: "",
			}},
		},
		Cfg: &evictionrequested.ProcessorConfig{
			DeployURL:                         "https://deploy.yandex-team.ru",
			NannyURL:                          "https://nanny.yandex-team.ru",
			MaxEvictionRequestedPodsTableSize: 2,
			ManualEvictionPeriod:              time.Hour * 96,
			EvictionExpiredThreshold:          time.Hour * 24,
			HandsupExpiredThreshold:           time.Hour,
		},
		Pods: pods,
		ReplicationPolicy: PatchReplicationPolicyNotificationsEnabled(
			PatchReplicationPolicyHandsup(
				GetReplicationPolicy(),
			),
		),
	}

	return p.Process()
}
