package main

import (
	"bufio"
	"context"
	"flag"
	"fmt"
	"log"
	"os"
	"strings"

	"a.yandex-team.ru/infra/spnotifier/providers/states"
	"a.yandex-team.ru/yt/go/ypath"
	"a.yandex-team.ru/yt/go/yt"
	"a.yandex-team.ru/yt/go/yt/ytrpc"
)

const YtPathTemplate = "//home/spnotifier/production/%s"

func getCurrentState(id, pkName, table string, client yt.Client, row states.TableRow) {
	reader, err := client.SelectRows(
		context.Background(),
		fmt.Sprintf(
			"* from [%s] where %s=%q",
			fmt.Sprintf(YtPathTemplate, table),
			pkName,
			id,
		),
		nil,
	)
	if err != nil {
		log.Fatalf("unable to read current state: %v\n", err)
	}

	if ok := reader.Next(); !ok {
		log.Fatalln("unable to find row for this id")
	}
	err = reader.Scan(row)
	if err != nil {
		log.Fatalf("unable to scan row: %v\n", err)
	}
}

func main() {
	nanny := flag.String("nanny", "", "nanny service id")
	deploy := flag.String("deploy", "", "deploy stage id")
	flag.Parse()

	token := os.Getenv("YT_TOKEN")
	if len(token) == 0 {
		log.Fatalln("YT_TOKEN env var must be set")
	}
	ytClient, err := ytrpc.NewClient(&yt.Config{
		Proxy: "hahn",
		Token: token,
	})
	if err != nil {
		log.Fatalf("yt client creation failed: %v\n", err)
	}

	var row, emptyRow states.TableRow
	var id, pk, table string
	switch {
	case len(*nanny) != 0 && len(*deploy) != 0:
		log.Fatalln("both --nanny and --deploy are specified, expected one of them")
	case len(*nanny) != 0:
		row = &states.NannyTableRow{}
		emptyRow = &states.NannyTableRow{ServiceID: *nanny, State: []byte("{\"startreker_execution\":null}")}
		pk = "service_id"
		table = "nanny"
		id = *nanny
	case len(*deploy) != 0:
		row = &states.DeployTableRow{}
		emptyRow = &states.DeployTableRow{StageID: *deploy, State: []byte("{\"startreker_execution\":null}")}
		pk = "stage_id"
		table = "deploy"
		id = *deploy
	default:
		log.Fatalln("expected one of --nanny or --deploy")
	}

	getCurrentState(id, pk, table, ytClient, row)
	fmt.Printf("current state: %s\n\nContinue? y/n: ", row.GetState())

	r := bufio.NewReader(os.Stdin)
	ans, err := r.ReadString('\n')
	if err != nil {
		panic(err)
	}
	switch strings.TrimRight(ans, "\n") {
	case "y":
		err = ytClient.InsertRows(
			context.Background(),
			ypath.Path(fmt.Sprintf(YtPathTemplate, table)),
			[]interface{}{emptyRow},
			nil,
		)
		if err != nil {
			log.Fatalf("empty row insertion failed: %v\n", err)
		}
		fmt.Println("done")
	case "n":
		fallthrough
	default:
		fmt.Println("aborted")
		os.Exit(0)
	}
}
