package deploy

import (
	"fmt"
	"strconv"
	"strings"
	"time"

	"a.yandex-team.ru/infra/spnotifier/processors"
	"a.yandex-team.ru/infra/spnotifier/processors/deploy/evictionrequested"
	"a.yandex-team.ru/infra/spnotifier/providers/stages"
	"a.yandex-team.ru/infra/spnotifier/ticket"
	"a.yandex-team.ru/infra/temporal/clients/abc"
	"a.yandex-team.ru/infra/temporal/workflows/startreker/processor"

	"a.yandex-team.ru/infra/spnotifier/providers/pods"
	"a.yandex-team.ru/infra/spnotifier/providers/states"
)

type EvictionRequestedConfig struct {
	WhiteList                       []string      `yaml:"white_list"`
	EvictionExpiredThreshold        time.Duration `yaml:"eviction_expired_threshold"`
	MaxEvictionExpiredPodsTableSize int           `yaml:"max_eviction_expired_pods_table_size"`
	ManualEvictionPeriod            time.Duration `yaml:"manual_eviction_period"`
}

type Config struct {
	DeployURL         string
	EvictionRequested *EvictionRequestedConfig
	StartrekerQueue   string
}

type Data struct {
	Stage *stages.Stage
	Pods  map[string]*pods.DeployPod
	State *states.DeployState
}

type Controller struct {
	Cfg             *Config
	Data            *Data
	TemporalToken   string
	AbcToken        string
	StartrekerToken string
}

type UnprocessableStageError struct {
	message string
}

func NewUnprocessableStageError(m string) UnprocessableStageError {
	return UnprocessableStageError{m}
}

func (e UnprocessableStageError) Error() string {
	return e.message
}

func getAbcIDFromAccount(account string) (int, error) {
	accountParts := strings.Split(account, ":")
	if len(accountParts) != 3 || accountParts[0] != "abc" || accountParts[1] != "service" {
		return 0, NewUnprocessableStageError("account is not an abc service")
	}
	abcID, err := strconv.Atoi(accountParts[2])
	if err != nil {
		return 0, err
	}

	return abcID, nil
}

func (ctrl *Controller) getOnDutyFromService(account string) ([]string, error) {
	abcID, err := getAbcIDFromAccount(account)
	if err != nil {
		return nil, err
	}

	client := abc.NewClient(&abc.ClientConfig{OauthToken: ctrl.AbcToken})
	logins, err := client.GetOnDutyFromService(abcID)
	if err != nil {
		return nil, err
	}
	return logins, nil
}

func (ctrl *Controller) GetInvocationData(stExec *ticket.StartrekerExecution) (*ticket.InvocationData, error) {
	assignee, err := ticket.GetTicketAssigneeByMacro(
		ctrl.StartrekerToken,
		ctrl.Cfg.StartrekerQueue,
		ctrl.MakeStartrekerWorkflowID(),
	)
	if err != nil {
		return nil, err
	}
	if assignee != "" {
		return &ticket.InvocationData{
			Responsible: &processor.Responsible{
				Kind:   processor.Logins,
				Logins: []string{assignee},
			},
			InvocationReason: ticket.Assignee,
		}, nil
	}

	onDuty, err := ctrl.getOnDutyFromService(ctrl.Data.Stage.AccountID)
	if err != nil {
		return nil, err
	}
	if len(onDuty) != 0 && onDuty[0] != "" {
		return &ticket.InvocationData{
			Responsible: &processor.Responsible{
				Kind:   processor.Logins,
				Logins: onDuty[:1],
			},
			InvocationReason: ticket.ProjectDuty,
		}, nil
	}

	_, err = getAbcIDFromAccount(ctrl.Data.Stage.OwnerID)
	if err != nil {
		// OwnerID is login
		return &ticket.InvocationData{
			Responsible: &processor.Responsible{
				Kind:   processor.Logins,
				Logins: []string{ctrl.Data.Stage.OwnerID},
			},
			InvocationReason: ticket.ProjectOwner,
		}, nil
	}

	// OwnerID is an abc service
	onDuty, err = ctrl.getOnDutyFromService(ctrl.Data.Stage.OwnerID)
	if err != nil {
		return nil, err
	}
	if len(onDuty) != 0 && onDuty[0] != "" {
		return &ticket.InvocationData{
			Responsible: &processor.Responsible{
				Kind:   processor.Logins,
				Logins: onDuty[:1],
			},
			InvocationReason: ticket.ProjectDuty,
		}, nil
	}

	return nil, fmt.Errorf("no suitable invocation data found")
}

func (ctrl *Controller) GetNotificationResult() (*processors.NotificationProcessorResult, error) {
	notificationProcessors := []processors.NotificationProcessor{}

	evictionrequestedProcessor := &evictionrequested.Processor{
		Cfg: &evictionrequested.ProcessorConfig{
			DeployURL:                       ctrl.Cfg.DeployURL,
			WhiteList:                       ctrl.Cfg.EvictionRequested.WhiteList,
			MaxEvictionExpiredPodsTableSize: ctrl.Cfg.EvictionRequested.MaxEvictionExpiredPodsTableSize,
			ManualEvictionPeriod:            ctrl.Cfg.EvictionRequested.ManualEvictionPeriod,
			EvictionExpiredThreshold:        ctrl.Cfg.EvictionRequested.EvictionExpiredThreshold,
		},
		StageID: ctrl.Data.Stage.ID,
		Pods:    ctrl.Data.Pods,
	}
	if evictionrequestedProcessor.IsEnabled() {
		notificationProcessors = append(notificationProcessors, evictionrequestedProcessor)
	}

	deployProcessor := &processors.Processor{
		Processors: notificationProcessors,
		Header:     "В стейдже обнаружены:",
		Title:      fmt.Sprintf("Не работает переселение подов в стейдже %s", ctrl.Data.Stage.ID),
	}

	result, err := deployProcessor.Process()
	if err != nil {
		return nil, fmt.Errorf("deploy processor failed: %w", err)
	}
	return result, nil
}

func (ctrl *Controller) GetID() string {
	return ctrl.Data.Stage.ID
}

func (ctrl *Controller) GetStartrekerExecution() *ticket.StartrekerExecution {
	if ctrl.Data.State == nil {
		ctrl.Data.State = &states.DeployState{}
	}
	return ctrl.Data.State.StartrekerExecution
}

func (ctrl *Controller) SetStartrekerExecution(stExec *ticket.StartrekerExecution) {
	ctrl.Data.State.StartrekerExecution = stExec
}

func (ctrl *Controller) MakeStartrekerWorkflowID() string {
	return fmt.Sprintf("spnotifier-deploy-%s", ctrl.Data.Stage.ID)
}

func (ctrl *Controller) GetState() interface{} {
	return ctrl.Data.State
}

func (ctrl *Controller) GetYtTableRow(data []byte) interface{} {
	return states.DeployTableRow{
		StageID: ctrl.Data.Stage.ID,
		State:   data,
	}
}
