package nanny

import (
	"errors"
	"fmt"
	"math/rand"
	"strings"
	"time"

	"a.yandex-team.ru/infra/spnotifier/clients/nanny"
	"a.yandex-team.ru/infra/spnotifier/processors"
	"a.yandex-team.ru/infra/spnotifier/processors/nanny/evictionrequested"
	"a.yandex-team.ru/infra/spnotifier/ticket"
	"a.yandex-team.ru/infra/temporal/clients/abc"
	"a.yandex-team.ru/infra/temporal/workflows/startreker/processor"

	"a.yandex-team.ru/infra/spnotifier/providers/pods"
	"a.yandex-team.ru/infra/spnotifier/providers/states"

	pb "a.yandex-team.ru/infra/nanny/go/proto/nanny_repo"
)

type EvictionRequestedConfig struct {
	WhiteList                         []string      `yaml:"white_list"`
	EvictionExpiredThreshold          time.Duration `yaml:"eviction_expired_threshold"`
	HandsupExpiredThreshold           time.Duration `yaml:"handsup_expired_threshold"`
	MaxEvictionRequestedPodsTableSize int           `yaml:"max_eviction_requested_pods_table_size"`
	ManualEvictionPeriod              time.Duration `yaml:"manual_eviction_period"`
}

type Config struct {
	DeployURL              string
	NannyURL               string
	EvictionRequested      *EvictionRequestedConfig
	MaxResponsibleToSummon int
	StartrekerQueue        string
}

type Data struct {
	Service           *nanny.Service
	Pods              map[string]*pods.NannyPod
	State             *states.NannyState
	ReplicationPolicy *pb.ReplicationPolicy
}

type Controller struct {
	Cfg             *Config
	Data            *Data
	NannyToken      string
	AbcToken        string
	StartrekerToken string
}

func (ctrl *Controller) getServiceLatestRuntimeAttrsAuthors() ([]string, error) {
	client := nanny.NewClient(ctrl.Cfg.NannyURL, ctrl.NannyToken)
	runtimeAttrsHistory, err := client.GetServiceRuntimeAttrsHistory(ctrl.Data.Service.ID, 10)
	if err != nil {
		return nil, err
	}

	var uniqueLogins []string
	loginsMap := make(map[string]interface{})
	for _, info := range runtimeAttrsHistory {
		if len(uniqueLogins) > ctrl.Cfg.MaxResponsibleToSummon {
			break
		}

		login := info.ChangeInfo.Author
		if _, ok := loginsMap[login]; !ok && !strings.Contains(login, "robot") {
			uniqueLogins = append(uniqueLogins, login)
			loginsMap[login] = nil
		}
	}

	return uniqueLogins, nil
}

func (ctrl *Controller) getOnDutyFromService() ([]string, error) {
	client := abc.NewClient(&abc.ClientConfig{OauthToken: ctrl.AbcToken})
	logins, err := client.GetOnDutyFromService(ctrl.Data.Service.InfoAttrs.Content.AbcGroup)
	if err != nil {
		return nil, err
	}
	return logins, nil
}

func (ctrl *Controller) getAbcMembers() ([]string, error) {
	var abcID int

	infoAttrs := ctrl.Data.Service.InfoAttrs.Content
	if infoAttrs.DutySchedule.ABCServiceID != 0 {
		abcID = infoAttrs.DutySchedule.ABCServiceID
	} else {
		abcID = infoAttrs.AbcGroup
	}

	client := abc.NewClient(&abc.ClientConfig{OauthToken: ctrl.AbcToken})
	logins, err := client.GetServiceMembers(abcID)
	if err != nil {
		return nil, err
	}

	rand.Seed(time.Now().UnixNano())
	rand.Shuffle(len(logins), func(i, j int) { logins[i], logins[j] = logins[j], logins[i] })

	if len(logins) < ctrl.Cfg.MaxResponsibleToSummon {
		return logins, nil
	}
	return logins[:ctrl.Cfg.MaxResponsibleToSummon], nil
}

func (ctrl *Controller) GetInvocationData(stExec *ticket.StartrekerExecution) (*ticket.InvocationData, error) {
	assignee, err := ticket.GetTicketAssigneeByMacro(
		ctrl.StartrekerToken,
		ctrl.Cfg.StartrekerQueue,
		ctrl.MakeStartrekerWorkflowID(),
	)
	if err != nil {
		return nil, err
	}
	if assignee != "" {
		return &ticket.InvocationData{
			Responsible: &processor.Responsible{
				Kind:   processor.Logins,
				Logins: []string{assignee},
			},
			InvocationReason: ticket.Assignee,
		}, nil
	}

	dutySchedule := ctrl.Data.Service.InfoAttrs.Content.DutySchedule
	if dutySchedule.ID != 0 {
		return &ticket.InvocationData{
			Responsible: &processor.Responsible{
				Kind:          processor.AbcSchedule,
				AbcScheduleID: dutySchedule.ID,
			},
			InvocationReason: ticket.DutySchedule,
		}, nil
	}

	onDuty, err := ctrl.getOnDutyFromService()
	if err != nil {
		return nil, err
	}
	if len(onDuty) != 0 && onDuty[0] != "" {
		return &ticket.InvocationData{
			Responsible: &processor.Responsible{
				Kind:   processor.Logins,
				Logins: onDuty[:1],
			},
			InvocationReason: ticket.RandomDutyFromService,
		}, nil
	}

	authors, err := ctrl.getServiceLatestRuntimeAttrsAuthors()
	if err != nil {
		return nil, err
	}
	if len(authors) != 0 {
		return &ticket.InvocationData{
			Responsible: &processor.Responsible{
				Kind:   processor.Logins,
				Logins: authors,
			},
			InvocationReason: ticket.ServiceRuntimeAttrsAuthor,
		}, nil
	}

	// always summon same abc members
	if stExec != nil {
		lastInvocationData := stExec.LastInvocationData
		if lastInvocationData.InvocationReason == ticket.AbcMember {
			return lastInvocationData, nil
		}
	}

	members, err := ctrl.getAbcMembers()
	if err != nil {
		return nil, err
	}
	if len(members) != 0 {
		return &ticket.InvocationData{
			Responsible: &processor.Responsible{
				Kind:   processor.Logins,
				Logins: members,
			},
			InvocationReason: ticket.AbcMember,
		}, nil
	}

	return nil, errors.New("no responsibe found")
}

func (ctrl *Controller) GetNotificationResult() (*processors.NotificationProcessorResult, error) {
	notificationProcessors := []processors.NotificationProcessor{}

	evictionrequestedProcessor := &evictionrequested.Processor{
		Cfg: &evictionrequested.ProcessorConfig{
			DeployURL:                         ctrl.Cfg.DeployURL,
			NannyURL:                          ctrl.Cfg.NannyURL,
			WhiteList:                         ctrl.Cfg.EvictionRequested.WhiteList,
			MaxEvictionRequestedPodsTableSize: ctrl.Cfg.EvictionRequested.MaxEvictionRequestedPodsTableSize,
			ManualEvictionPeriod:              ctrl.Cfg.EvictionRequested.ManualEvictionPeriod,
			EvictionExpiredThreshold:          ctrl.Cfg.EvictionRequested.EvictionExpiredThreshold,
			HandsupExpiredThreshold:           ctrl.Cfg.EvictionRequested.HandsupExpiredThreshold,
		},
		Service:           ctrl.Data.Service,
		Pods:              ctrl.Data.Pods,
		ReplicationPolicy: ctrl.Data.ReplicationPolicy,
	}
	if evictionrequestedProcessor.IsEnabled() {
		notificationProcessors = append(notificationProcessors, evictionrequestedProcessor)
	}

	nannyProcessor := &processors.Processor{
		Processors: notificationProcessors,
		Header:     "В сервисе обнаружены:",
	}

	result, err := nannyProcessor.Process()
	if err != nil {
		return nil, fmt.Errorf("nanny processor failed: %w", err)
	}

	title := fmt.Sprintf("Нотификации о сервисе %s", ctrl.Data.Service.ID)
	if result.IsProblem {
		title = fmt.Sprintf("Не работает переселение подов в сервисе %s", ctrl.Data.Service.ID)
	}
	result.Header = title
	return result, nil
}

func (ctrl *Controller) GetID() string {
	return ctrl.Data.Service.ID
}

func (ctrl *Controller) GetStartrekerExecution() *ticket.StartrekerExecution {
	if ctrl.Data.State == nil {
		ctrl.Data.State = &states.NannyState{}
	}
	return ctrl.Data.State.StartrekerExecution
}

func (ctrl *Controller) SetStartrekerExecution(stExec *ticket.StartrekerExecution) {
	ctrl.Data.State.StartrekerExecution = stExec
}

func (ctrl *Controller) MakeStartrekerWorkflowID() string {
	return fmt.Sprintf("spnotifier-nanny-%s", ctrl.Data.Service.ID)
}

func (ctrl *Controller) GetState() interface{} {
	return ctrl.Data.State
}

func (ctrl *Controller) GetYtTableRow(data []byte) interface{} {
	return states.NannyTableRow{
		ServiceID: ctrl.Data.Service.ID,
		State:     data,
	}
}
