package servicecontrollers

import (
	"context"
	"encoding/json"
	"errors"
	"fmt"
	"log"
	"sync"
	"time"

	"go.temporal.io/sdk/client"

	"a.yandex-team.ru/infra/spnotifier/clients/errorbooster"
	"a.yandex-team.ru/infra/spnotifier/processors"
	"a.yandex-team.ru/infra/spnotifier/servicecontrollers/deploy"
	"a.yandex-team.ru/infra/spnotifier/ticket"
	ytuploader "a.yandex-team.ru/infra/spnotifier/uploaders/yt"
	"a.yandex-team.ru/infra/temporal/workflows/startreker/processor"
)

type StartrekerConfig struct {
	TicketQueue           string        `yaml:"ticket_queue"`
	Tags                  []string      `yaml:"tags"`
	RetryInvocationPeriod time.Duration `yaml:"retry_invocation_period"`
	DutyScheduleID        int           `yaml:"duty_schedule_id"`
	TaskQueue             string        `yaml:"task_queue"`
}

type ServiceController interface {
	GetID() string
	GetNotificationResult() (*processors.NotificationProcessorResult, error)
	GetStartrekerExecution() *ticket.StartrekerExecution
	SetStartrekerExecution(*ticket.StartrekerExecution)
	GetInvocationData(*ticket.StartrekerExecution) (*ticket.InvocationData, error)
	MakeStartrekerWorkflowID() string
	GetState() interface{}
	GetYtTableRow([]byte) interface{}
}

type Config struct {
	StartrekerConfig *StartrekerConfig
	Summon           bool
	YtProxy          string
	YtPath           string
	WorkflowPostfix  string
}

type ControllerRunner struct {
	Ctrl ServiceController
	Cfg  *Config

	YtUploaderChan chan *ytuploader.YtRowInfo
	TemporalClient client.Client
	EBClient       *errorbooster.Client
}

func (r *ControllerRunner) getTicketData(result *processors.NotificationProcessorResult, stExec *ticket.StartrekerExecution) (*ticket.TicketData, error) {
	var invocationData *ticket.InvocationData
	var invocationSettings *processor.RetryInvocationSettings
	if !result.NeedsInvocation {
		invocationData = &ticket.InvocationData{
			InvocationReason: ticket.None,
			Responsible: &processor.Responsible{
				Kind:   processor.Logins,
				Logins: []string{},
			},
		}
		invocationSettings = &processor.RetryInvocationSettings{
			Kind: processor.Once,
		}
	} else {
		var err error
		invocationData, err = r.Ctrl.GetInvocationData(stExec)
		if err != nil {
			return nil, err
		}
		invocationSettings = &processor.RetryInvocationSettings{
			Kind:   processor.AckPeriod,
			Period: r.Cfg.StartrekerConfig.RetryInvocationPeriod,
		}
	}

	if !r.Cfg.Summon {
		log.Printf(
			"[%s] summon = false, printing responsible instead: %v",
			r.Ctrl.GetID(),
			invocationData.Responsible,
		)

		invocationData.Responsible = &processor.Responsible{
			Kind:   processor.Logins,
			Logins: []string{},
		}
	}

	return &ticket.TicketData{
		InvocationData:     invocationData,
		InvocationSettings: invocationSettings,
		Result:             result,
	}, nil
}

func (r *ControllerRunner) processTicket(result *processors.NotificationProcessorResult) (*ticket.StartrekerExecution, error) {
	startrekerExecution := r.Ctrl.GetStartrekerExecution()
	ticketCtrl := ticket.Controller{
		Cfg: &ticket.ControllerConfig{
			TaskQueue:             r.Cfg.StartrekerConfig.TaskQueue,
			TicketQueue:           r.Cfg.StartrekerConfig.TicketQueue,
			Tags:                  r.Cfg.StartrekerConfig.Tags,
			RetryInvocationPeriod: r.Cfg.StartrekerConfig.RetryInvocationPeriod,
			DutyScheduleID:        r.Cfg.StartrekerConfig.DutyScheduleID,
			WorkflowPostfix:       r.Cfg.WorkflowPostfix,
		},
		StartrekerExec: startrekerExecution,
		TemporalClient: r.TemporalClient,
	}

	ctx := context.Background()
	if result.NeedsRendering {
		ticketData, err := r.getTicketData(result, startrekerExecution)
		if err != nil {
			return nil, err
		}
		stExec, err := ticketCtrl.CreateOrUpdateTicket(
			ctx,
			ticketData,
			r.Ctrl.MakeStartrekerWorkflowID(),
		)
		startrekerExecution = stExec
		if err != nil {
			return nil, fmt.Errorf("ticket updating failed: %w", err)
		}

	} else {
		err := ticketCtrl.CloseTicketIfExists(ctx, ticket.CloseReasonServiceOk)
		if err != nil {
			return nil, err
		}
		startrekerExecution = nil
	}

	return startrekerExecution, nil
}

func (r *ControllerRunner) saveState() error {
	rawState, err := json.Marshal(r.Ctrl.GetState())
	if err != nil {
		return fmt.Errorf("cannot marshal state: %w", err)
	}
	row := r.Ctrl.GetYtTableRow(rawState)
	r.YtUploaderChan <- &ytuploader.YtRowInfo{YtPath: r.Cfg.YtPath, Value: row}

	return nil
}

func (r *ControllerRunner) process() error {
	result, err := r.Ctrl.GetNotificationResult()
	if err != nil {
		return err
	}

	stExec, err := r.processTicket(result)
	if err != nil {
		return fmt.Errorf("ticket processing failed: %w", err)
	}
	r.Ctrl.SetStartrekerExecution(stExec)

	err = r.saveState()
	if err != nil {
		return err
	}

	return nil
}

func (r *ControllerRunner) Run(wg *sync.WaitGroup) {
	defer wg.Done()
	log.Printf("[%s] starting controller\n", r.Ctrl.GetID())

	err := r.process()
	if err != nil {
		switch {
		case errors.As(err, &deploy.UnprocessableStageError{}):
			log.Printf("[%s] unable to process stage: %v, skipping", r.Ctrl.GetID(), err)
		default:
			msg := fmt.Sprintf("[%s] processing error: %v", r.Ctrl.GetID(), err)
			log.Println(msg)
			if err := r.EBClient.Send(msg); err != nil {
				log.Printf("failed to send to errorbooster: %v", err)
			}
		}
	}

	log.Printf("[%s] finished\n", r.Ctrl.GetID())
}
