package ticket

import (
	"context"
	"fmt"
	"time"

	"go.temporal.io/sdk/client"
	"go.temporal.io/sdk/workflow"

	"a.yandex-team.ru/infra/spnotifier/processors"
	"a.yandex-team.ru/infra/temporal/clients/startrek"
	"a.yandex-team.ru/infra/temporal/workflows/startreker/processor"
)

const (
	DutyInvocationText        = "Привет!\nНужна твоя помощь, как дежурного nanny"
	CloseReasonNowTestingText = "Вы пометили сервис как тестинг, закрываем тикет."
	CloseReasonServiceOk      = "Для вашего сервиса больше нет уведомлений, закрываем тикет."
	CloseReasonStageOk        = "Для вашего стейджа больше нет уведомлений, закрываем тикет."
	CloseReasonServiceDeleted = "Ваш сервис удален, закрываем тикет."
)

type InvocationReason int

const (
	DutySchedule InvocationReason = iota
	ServiceRuntimeAttrsAuthor
	AbcMember
	RandomDutyFromService
	ProjectDuty
	ProjectOwner
	None
	Assignee
)

type ControllerConfig struct {
	TaskQueue             string
	TicketQueue           string
	Tags                  []string
	RetryInvocationPeriod time.Duration
	DutyScheduleID        int
	WorkflowPostfix       string
}

type InvocationData struct {
	Responsible *processor.Responsible
	InvocationReason
}

type StartrekerExecution struct {
	Workflow           *workflow.Execution
	LastInvocationData *InvocationData
}

type TicketData struct {
	InvocationData     *InvocationData
	InvocationSettings *processor.RetryInvocationSettings
	Result             *processors.NotificationProcessorResult
}

type Controller struct {
	Cfg            *ControllerConfig
	StartrekerExec *StartrekerExecution
	TemporalClient client.Client
}

func (ctrl *Controller) CommentTicket(ctx context.Context, message string) error {
	return ctrl.TemporalClient.SignalWorkflow(ctx,
		ctrl.StartrekerExec.Workflow.ID,
		ctrl.StartrekerExec.Workflow.RunID,
		"CommentProblem",
		processor.CommentProblemSignal{
			Comment: startrek.Comment{
				Text: message,
			},
		})
}

func (ctrl *Controller) CloseTicketIfExists(ctx context.Context, reason string) error {
	if ctrl.StartrekerExec == nil {
		return nil
	}
	if err := ctrl.CommentTicket(ctx, reason); err != nil {
		return err
	}
	err := ctrl.TemporalClient.SignalWorkflow(ctx,
		ctrl.StartrekerExec.Workflow.ID,
		ctrl.StartrekerExec.Workflow.RunID,
		"CloseProblem",
		nil)
	if err != nil {
		return err
	}
	ctrl.StartrekerExec = nil
	return nil
}

func (ctrl *Controller) getInvocationText(ticketData *TicketData) string {
	var text string
	switch ticketData.InvocationData.InvocationReason {
	case DutySchedule:
		text = "Вы были выбраны, так как являетесь текущим дежурным в ABC сервисе, ((https://clubs.at.yandex-team.ru/infra-cloud/1955 который указан в няне)). "
	case RandomDutyFromService:
		text = "Вы были выбраны, так как являетесь дежурным этого сервиса. Из-за того, что у вас не ((https://clubs.at.yandex-team.ru/infra-cloud/1955 указан календарь дежурств)), мы выбрали случайный календарь из указанного в няне ABC-сервиса. "
	case ServiceRuntimeAttrsAuthor:
		text = "Вы были выбраны, так как недавно выкатывали изменения в этом сервисе. Это очень неточная эвристика, поэтому рекомендуем ((https://clubs.at.yandex-team.ru/infra-cloud/1955 указать календарь дежурств)). "
	case AbcMember:
		text = "Мы выбрали несколько человек из ABC сервиса, потому что не знаем, кто сможет этой проблемой заняться. Пожалуйста, ((https://clubs.at.yandex-team.ru/infra-cloud/1955 укажите календарь дежурств)). "
	case ProjectDuty:
		text = "Вы были выбраны, так как являетесь текущим дежурным в ABC сервисе проекта, к которому относится данный стейдж. "
	case ProjectOwner:
		text = "Вы были выбраны, так как являетесь владельцем проекта, в который входит данный стейдж. Мы не смогли определить дежурного, так как либо не указан ABC сервис проекта, либо в нем отстутсвует календарь дежурств. "
	case Assignee:
		text = "Вы были выбраны, так как являетесь текущим исполнителем тикета. "
	}

	return fmt.Sprintf("Обратите внимание на тикет. %s\n\n%s", text, ticketData.Result.InvocationText)
}

func (ctrl *Controller) formProblem(ticketData *TicketData) processor.Problem {
	problem := processor.Problem{
		Ticket: startrek.Ticket{
			Summary:     ticketData.Result.Header,
			Description: ticketData.Result.Description,
			Queue:       &startrek.Queue{Key: ctrl.Cfg.TicketQueue},
			Tags:        ctrl.Cfg.Tags,
		},
		InvocationSettings: processor.InvocationSettings{
			Responsible:             *ticketData.InvocationData.Responsible,
			RetryInvocationSettings: *ticketData.InvocationSettings,
			Text:                    ctrl.getInvocationText(ticketData),
		},
		InfraDutyInvocationSettings: &processor.InfraDutyInvocationSettings{
			AbcScheduleID: ctrl.Cfg.DutyScheduleID,
			Text:          "",
		},
	}
	return problem
}

func (ctrl *Controller) CreateOrUpdateTicket(ctx context.Context, ticketData *TicketData, WorkflowID string) (*StartrekerExecution, error) {
	problem := ctrl.formProblem(ticketData)

	if ctrl.StartrekerExec == nil {
		wo := client.StartWorkflowOptions{
			ID:        fmt.Sprintf("%s%s", WorkflowID, ctrl.Cfg.WorkflowPostfix),
			TaskQueue: ctrl.Cfg.TaskQueue,
		}
		wr, err := ctrl.TemporalClient.ExecuteWorkflow(ctx, wo, processor.ProcessWorkflow, problem)
		if err != nil {
			return nil, fmt.Errorf("workflow execution failed: %w", err)
		}

		ctrl.StartrekerExec = &StartrekerExecution{Workflow: &workflow.Execution{
			RunID: wr.GetRunID(),
			ID:    wr.GetID(),
		}}
	} else {
		err := ctrl.TemporalClient.SignalWorkflow(ctx,
			ctrl.StartrekerExec.Workflow.ID,
			ctrl.StartrekerExec.Workflow.RunID,
			"UpdateProblem",
			processor.UpdateProblemSignal{Problem: problem, Notify: false})
		if err != nil {
			return nil, err
		}
	}

	if ticketData.InvocationSettings.Kind == processor.AckPeriod {
		err := ctrl.TemporalClient.SignalWorkflow(ctx,
			ctrl.StartrekerExec.Workflow.ID,
			ctrl.StartrekerExec.Workflow.RunID,
			"AckSummon",
			nil)
		if err != nil {
			return nil, err
		}
	}

	ctrl.StartrekerExec.LastInvocationData = ticketData.InvocationData
	return ctrl.StartrekerExec, nil
}
