package main

import (
	"errors"
	"fmt"
	"os"
	"time"

	"gopkg.in/yaml.v2"

	"a.yandex-team.ru/infra/spnotifier/clients/errorbooster"
	"a.yandex-team.ru/infra/spnotifier/servicecontrollers"
	"a.yandex-team.ru/infra/spnotifier/servicecontrollers/deploy"
	"a.yandex-team.ru/infra/spnotifier/servicecontrollers/nanny"
)

const ConfigPathEnvironmentVariable = "CONFIG_PATH"

type YpConfig struct {
	BatchSize      int           `yaml:"batch_size"`
	Clusters       []string      `yaml:"clusters"`
	RequestTimeout time.Duration `yaml:"request_timeout"`
}

type NannyConfig struct {
	URL                             string `yaml:"url"`
	GetServicesBatchSize            int    `yaml:"get_services_batch_size"`
	GetReplicationPoliciesBatchSize int    `yaml:"get_replication_policies_batch_size"`
}

type DeployConfig struct {
	URL string `yaml:"url"`
}

type TemporalConfig struct {
	URL             string `yaml:"url"`
	TaskQueue       string `yaml:"task_queue"`
	NamespaceID     string `yaml:"namespace_id"`
	WorkflowPostfix string `yaml:"workflow_postfix"`
}

type YtConfig struct {
	Proxy      string `yaml:"proxy"`
	NannyPath  string `yaml:"nanny_path"`
	DeployPath string `yaml:"deploy_path"`
}

type NannyProcessors struct {
	EvictionRequested *nanny.EvictionRequestedConfig `yaml:"eviction_requested"`
}

type DeployProcessors struct {
	EvictionRequested *deploy.EvictionRequestedConfig `yaml:"eviction_requested"`
}

type Processors struct {
	Nanny  *NannyProcessors  `yaml:"nanny"`
	Deploy *DeployProcessors `yaml:"deploy"`
}

type Startreker struct {
	Nanny                  *servicecontrollers.StartrekerConfig `yaml:"nanny"`
	Deploy                 *servicecontrollers.StartrekerConfig `yaml:"deploy"`
	MaxResponsibleToSummon int                                  `yaml:"max_responsible_to_summon"`
	Summon                 bool                                 `yaml:"summon"`
}

type Config struct {
	IterationPeriod time.Duration        `yaml:"iteration_period"`
	Yp              *YpConfig            `yaml:"yp"`
	Yt              *YtConfig            `yaml:"yt"`
	Nanny           *NannyConfig         `yaml:"nanny"`
	Deploy          *DeployConfig        `yaml:"deploy"`
	Temporal        *TemporalConfig      `yaml:"temporal"`
	Processors      *Processors          `yaml:"processors"`
	Startreker      *Startreker          `yaml:"startreker"`
	ErrorBooster    *errorbooster.Config `yaml:"errorbooster"`
}

func getConfig() (*Config, error) {
	fileName, exists := os.LookupEnv(ConfigPathEnvironmentVariable)
	if !exists {
		return nil, errors.New("environment variable for config path is not set")
	}

	configRaw, err := os.ReadFile(fileName)
	if err != nil {
		return nil, fmt.Errorf("unable to read config file: %w", err)
	}
	var config Config
	err = yaml.UnmarshalStrict(configRaw, &config)
	if err != nil {
		return nil, fmt.Errorf("unmarshalling error: %w", err)
	}

	return &config, nil
}
