package main

import (
	"context"
	"fmt"
	"log"
	"sync"

	"go.temporal.io/sdk/client"
	"golang.org/x/sync/errgroup"

	pb "a.yandex-team.ru/infra/nanny/go/proto/nanny_repo"
	"a.yandex-team.ru/infra/spnotifier/clients/errorbooster"
	"a.yandex-team.ru/infra/spnotifier/clients/nanny"
	"a.yandex-team.ru/infra/spnotifier/providers/pods"
	"a.yandex-team.ru/infra/spnotifier/providers/stages"
	"a.yandex-team.ru/infra/spnotifier/providers/states"
	"a.yandex-team.ru/infra/spnotifier/servicecontrollers"
	deployctrl "a.yandex-team.ru/infra/spnotifier/servicecontrollers/deploy"
	nannyctrl "a.yandex-team.ru/infra/spnotifier/servicecontrollers/nanny"
	ytuploader "a.yandex-team.ru/infra/spnotifier/uploaders/yt"
	temporalclient "a.yandex-team.ru/infra/temporal/swat/client"
	"a.yandex-team.ru/yt/go/yt"
	"a.yandex-team.ru/yt/go/yt/ytrpc"
)

type Data struct {
	states              *states.States
	pods                *pods.PodsData
	services            []*nanny.Service
	replicationPolicies map[string]*pb.ReplicationPolicy
	stages              map[string]*stages.Stage
}

type Controller struct {
	ctx      context.Context
	cfg      *Config
	ebClient *errorbooster.Client

	ypToken         string
	ytToken         string
	nannyToken      string
	temporalToken   string
	abcToken        string
	startrekerToken string
}

func (ctrl *Controller) getPodsData(ctx context.Context, c chan *pods.PodsData) error {
	defer close(c)
	log.Println("collecting pods...")

	provider := pods.Provider{
		Cfg: &pods.ProviderConfig{
			BatchSize:        ctrl.cfg.Yp.BatchSize,
			Clusters:         ctrl.cfg.Yp.Clusters,
			YpRequestTimeout: ctrl.cfg.Yp.RequestTimeout,
		},
		YpToken: ctrl.ypToken,
	}
	podsData, err := provider.GetPods(ctx)
	if err != nil {
		return fmt.Errorf("pods provider failed: %w", err)
	}
	log.Printf("collected pods: nanny services - %d, deploy stages - %d", len(podsData.Nanny), len(podsData.Deploy))

	c <- podsData
	return nil
}

func (ctrl *Controller) getStages(ctx context.Context, c chan map[string]*stages.Stage) error {
	defer close(c)
	log.Println("collecting stages...")

	provider := stages.Provider{
		Cfg: &stages.ProviderConfig{
			BatchSize:        ctrl.cfg.Yp.BatchSize,
			YpRequestTimeout: ctrl.cfg.Yp.RequestTimeout,
		},
		YpToken: ctrl.ypToken,
	}
	stagesData, err := provider.GetStages(ctx)
	if err != nil {
		return fmt.Errorf("stages provider failed: %w", err)
	}
	log.Printf("collected %d stages", len(stagesData))

	c <- stagesData
	return nil
}

func (ctrl *Controller) getServicesData(ctx context.Context, c chan []*nanny.Service) error {
	defer close(c)
	log.Println("collecting services...")

	client := nanny.NewClient(ctrl.cfg.Nanny.URL, ctrl.nannyToken)
	servicesData, err := client.GetServices(ctx, ctrl.cfg.Nanny.GetServicesBatchSize)
	if err != nil {
		return fmt.Errorf("services collecting failed: %w", err)
	}
	log.Printf("collected %d services\n", len(servicesData))

	c <- servicesData
	return nil
}

func (ctrl *Controller) getStates(ctx context.Context, c chan *states.States) error {
	defer close(c)
	log.Println("collecting states...")

	provider := states.Provider{
		Cfg: &states.ProviderConfig{
			Proxy:      ctrl.cfg.Yt.Proxy,
			NannyPath:  ctrl.cfg.Yt.NannyPath,
			DeployPath: ctrl.cfg.Yt.DeployPath,
		},
		YtToken: ctrl.ytToken,
	}

	statesData, err := provider.GetStates(ctx)
	if err != nil {
		return fmt.Errorf("states collecting failed: %w", err)
	}
	log.Printf("collected states: deploy - %d, nanny - %d\n", len(statesData.Deploy), len(statesData.Nanny))

	c <- statesData
	return nil
}

func (ctrl *Controller) getReplicationPolicies(ctx context.Context, c chan map[string]*pb.ReplicationPolicy) error {
	defer close(c)
	log.Println("collecting replication policies...")

	client := nanny.NewClient(ctrl.cfg.Nanny.URL, ctrl.nannyToken)
	policiesData, err := client.GetReplicationPolicies(ctx, ctrl.cfg.Nanny.GetReplicationPoliciesBatchSize)
	if err != nil {
		return fmt.Errorf("replication policies collecting failed: %w", err)
	}
	log.Printf("collected %d replication policies", len(policiesData))

	c <- policiesData
	return nil
}

func (ctrl *Controller) startNannyControllers(wg *sync.WaitGroup, data *Data, temporalClient client.Client, ytChan chan *ytuploader.YtRowInfo) {
	servicesWg := &sync.WaitGroup{}

	for _, service := range data.services {
		if service.RuntimeAttrs.MetaInfo.Annotaions.DeployEngine != "YP_LITE" {
			continue
		}
		rp, ok := data.replicationPolicies[service.ID]
		if !ok {
			continue
		}

		nannyCtrl := nannyctrl.Controller{
			Cfg: &nannyctrl.Config{
				EvictionRequested:      ctrl.cfg.Processors.Nanny.EvictionRequested,
				NannyURL:               ctrl.cfg.Nanny.URL,
				DeployURL:              ctrl.cfg.Deploy.URL,
				MaxResponsibleToSummon: ctrl.cfg.Startreker.MaxResponsibleToSummon,
				StartrekerQueue:        ctrl.cfg.Startreker.Nanny.TicketQueue,
			},
			Data: &nannyctrl.Data{
				Service:           service,
				ReplicationPolicy: rp,
				State:             data.states.Nanny[service.ID],
				Pods:              data.pods.Nanny[service.ID],
			},
			NannyToken:      ctrl.nannyToken,
			AbcToken:        ctrl.abcToken,
			StartrekerToken: ctrl.startrekerToken,
		}

		runner := servicecontrollers.ControllerRunner{
			Ctrl: &nannyCtrl,
			Cfg: &servicecontrollers.Config{
				StartrekerConfig: ctrl.cfg.Startreker.Nanny,
				YtProxy:          ctrl.cfg.Yt.Proxy,
				YtPath:           ctrl.cfg.Yt.NannyPath,
				WorkflowPostfix:  ctrl.cfg.Temporal.WorkflowPostfix,
				Summon:           ctrl.cfg.Startreker.Summon,
			},
			YtUploaderChan: ytChan,
			TemporalClient: temporalClient,
			EBClient:       ctrl.ebClient,
		}

		servicesWg.Add(1)
		go runner.Run(servicesWg)
	}

	servicesWg.Wait()
	wg.Done()
}

func (ctrl *Controller) startDeployControllers(wg *sync.WaitGroup, data *Data, temporalClient client.Client, ytChan chan *ytuploader.YtRowInfo) {
	stagesWg := &sync.WaitGroup{}

	for _, stage := range data.stages {
		deployCtrl := deployctrl.Controller{
			Cfg: &deployctrl.Config{
				DeployURL:         ctrl.cfg.Deploy.URL,
				EvictionRequested: ctrl.cfg.Processors.Deploy.EvictionRequested,
				StartrekerQueue:   ctrl.cfg.Startreker.Nanny.TicketQueue,
			},
			Data: &deployctrl.Data{
				Pods:  data.pods.Deploy[stage.ID],
				State: data.states.Deploy[stage.ID],
				Stage: stage,
			},
			TemporalToken:   ctrl.temporalToken,
			AbcToken:        ctrl.abcToken,
			StartrekerToken: ctrl.startrekerToken,
		}

		runner := servicecontrollers.ControllerRunner{
			Ctrl: &deployCtrl,
			Cfg: &servicecontrollers.Config{
				StartrekerConfig: ctrl.cfg.Startreker.Deploy,
				YtProxy:          ctrl.cfg.Yt.Proxy,
				YtPath:           ctrl.cfg.Yt.DeployPath,
				WorkflowPostfix:  ctrl.cfg.Temporal.WorkflowPostfix,
				Summon:           ctrl.cfg.Startreker.Summon,
			},
			YtUploaderChan: ytChan,
			TemporalClient: temporalClient,
			EBClient:       ctrl.ebClient,
		}

		stagesWg.Add(1)
		go runner.Run(stagesWg)
	}

	stagesWg.Wait()
	wg.Done()
}

func (ctrl *Controller) getData() (*Data, error) {
	errg, ctx := errgroup.WithContext(ctrl.ctx)

	podsDataChan := make(chan *pods.PodsData)
	errg.Go(func() error { return ctrl.getPodsData(ctx, podsDataChan) })

	servicesDataChan := make(chan []*nanny.Service)
	errg.Go(func() error { return ctrl.getServicesData(ctx, servicesDataChan) })

	statesChan := make(chan *states.States)
	errg.Go(func() error { return ctrl.getStates(ctx, statesChan) })

	replicationPoliciesChan := make(chan map[string]*pb.ReplicationPolicy)
	errg.Go(func() error { return ctrl.getReplicationPolicies(ctx, replicationPoliciesChan) })

	stagesChan := make(chan map[string]*stages.Stage)
	errg.Go(func() error { return ctrl.getStages(ctx, stagesChan) })

	data := &Data{}
	data.states = <-statesChan
	data.pods = <-podsDataChan
	data.services = <-servicesDataChan
	data.replicationPolicies = <-replicationPoliciesChan
	data.stages = <-stagesChan

	if err := errg.Wait(); err != nil {
		return nil, err
	}

	return data, nil
}

func (ctrl *Controller) run() error {
	data, err := ctrl.getData()
	if err != nil {
		return err
	}

	ytClient, err := ytrpc.NewClient(&yt.Config{
		Proxy: ctrl.cfg.Yt.Proxy,
		Token: ctrl.ytToken,
	})
	if err != nil {
		return fmt.Errorf("yt client creation failed: %w", err)
	}

	temporalClient, err := temporalclient.NewSdkClient(ctrl.cfg.Temporal.URL, ctrl.temporalToken, ctrl.cfg.Temporal.NamespaceID)
	if err != nil {
		return fmt.Errorf("temporal client creation failed: %w", err)
	}
	defer temporalClient.Close()

	ytUploadChan := make(chan *ytuploader.YtRowInfo)
	ytUploader := ytuploader.Uploader{
		Client: ytClient,
		Ch:     ytUploadChan,
	}
	ytUploader.Start()

	wg := &sync.WaitGroup{}

	wg.Add(1)
	go ctrl.startNannyControllers(wg, data, temporalClient, ytUploadChan)

	wg.Add(1)
	go ctrl.startDeployControllers(wg, data, temporalClient, ytUploadChan)

	log.Println("waiting until all service controllers finished")
	wg.Wait()

	close(ytUploadChan)
	log.Println("waiting until yt uploader finished")
	if err := ytUploader.Wait(); err != nil {
		return fmt.Errorf("yt uploader failed: %w", err)
	}

	return nil
}
