# A script that generates jinja panels for stagectl
import argparse
import difflib
import requests
import sys


def get_panel_url(type, action, name):
    return 'https://yasm.yandex-team.ru/srvambry/tmpl/{}/{}/content?key={}'.format(type, action, name)

YP_DOWNTIME_HASH = "9b8ee2a2b559b3afff751623e5e79546"

CACHED_OBJECT_TYPE = ["docker", "resources", "sandbox_resources"]

SANDBOX_RESOURCES_UPDATABLE = [
    #source = "sandbox"
    "pod_agent_binary",
    "tvm_layer",
    "logbroker_agent_layer",
]

SANDBOX_RESOURCES_FIXED_IN_CONFIG = [
    #source = "config"
    "pod_agent_layer",
    "tvm_base_layer",
    "instancectl_binary",
    "juggler_binary",
    "logrotate_binary",
    "dru_layer",
    "gdb_layer"
]

GRAPHS_PANEL_NAME = "stage-controller"
GRAPHS_PANEL_GET_URL = get_panel_url("panels", "get", GRAPHS_PANEL_NAME)
GRAPHS_PANEL_UPDATE_URL = get_panel_url("panels", "update", GRAPHS_PANEL_NAME)

AUTH_PANEL_NAME = "auth-controller"
AUTH_PANEL_GET_URL = get_panel_url("panels", "get", AUTH_PANEL_NAME)
AUTH_PANEL_UPDATE_URL = get_panel_url("panels", "update", AUTH_PANEL_NAME)

RESOURCES_PANEL_NAME = "stage-controller-resources"
RESOURCES_PANEL_GET_URL = get_panel_url("panels", "get", RESOURCES_PANEL_NAME)
RESOURCES_PANEL_UPDATE_URL = get_panel_url("panels", "update", RESOURCES_PANEL_NAME)

ALERTS_PANEL_NAME = "yd-stagectl-alerts"
ALERTS_PANEL_GET_URL = get_panel_url("alerts", "get", ALERTS_PANEL_NAME)
ALERTS_PANEL_UPDATE_URL = get_panel_url("alerts", "update", ALERTS_PANEL_NAME)
ALERTS_PANEL_APPLY_URL = 'https://yasm.yandex-team.ru/srvambry/tmpl/alerts/apply/{}'.format(ALERTS_PANEL_NAME)

RADAR_PANEL_NAME = "deploy-speed-radar"
RADAR_PANEL_GET_URL = get_panel_url("panels", "get", GRAPHS_PANEL_NAME)
RADAR_PANEL_UPDATE_URL = get_panel_url("panels", "update", GRAPHS_PANEL_NAME)

class Ctype:
    PROD = "prod"
    PRESTABLE = "prestable"
    TESTING = "testing"
    ACCEPTANCE = "acceptance"
    ALL = [PROD, PRESTABLE, TESTING, ACCEPTANCE]


class PerClusterObjects:
    RS = {
        'id': 'replica-set',
        "graph_name": "replica_sets",
        "description": "Replica sets",
        "abbreviation": "rs",
    }
    ES = {
        'id': 'endpoint-set',
        "graph_name": "endpoint_sets",
        "description": "Endpoint sets",
        "abbreviation": "es",
    }
    HPA = {
        'id': 'hpa',
        "graph_name": "horizontal_pod_autoscalers",
        "description": "HPAs",
        "abbreviation": "hpa"
    }
    DR = {
        'id': 'dynamic-resources',
        "graph_name": "dynamic_resources",
        "description": "Dynamic resources",
        "abbreviation": "dr"
    }
    #not in production yet....
    RC = {
        'id': 'resource_caches',
        "graph_name": "resource_caches",
        "description": "Resource caches",
        "abbreviation": "rc"
    }
    ALL = [RS, ES, HPA, DR]
    WATCHES_ONLY = [RS, ES]


managed_clusters = {
    Ctype.PROD: ['vla', 'man', 'sas', 'myt', 'iva'],
    Ctype.PRESTABLE: ['man-pre'],
    Ctype.TESTING: ['man-pre', 'sas-test'],
    Ctype.ACCEPTANCE: ['vla', 'man', 'sas', 'myt', 'iva'],
}


def print_with_indent(lines, indent, result):
    for line in lines:
        result.append(" " * indent * 4 + line)


def print_signal(title, graph, indent, result, color=None, y_axis=None, width=None, active=None):
    print_with_indent(['{'], indent, result)
    print_with_indent([
        '"title": "{}",'.format(title),
        '"name": "{}",'.format(graph),
    ], indent + 1, result)
    if color:
        print_with_indent([
            '"color": "{}",'.format(color)
        ], indent + 1, result)
    if width:
        print_with_indent([
            '"width": {},'.format(width)
        ], indent + 1, result)
    if y_axis:
        print_with_indent([
            '"yAxis": {},'.format(y_axis)
        ], indent + 1, result)
    if active:
        print_with_indent([
            '"active": true,'
        ], indent + 1, result)
    print_with_indent([
        '"tag": "<< tag >>",',
        '"host": "ASEARCH",'
    ], indent + 1, result)

    print_with_indent(['},'], indent, result)


class Alert(object):
    def __init__(self, name, warn, crit, flaps=None):
        self.name = name
        self.warn = warn
        self.crit = crit
        self.flaps = flaps
        self.cluster = None

    def print_info(self, signal, indent, result):
        # per-cluster alerts cannot format clusters in place
        print_with_indent(['<% do info.append({'], indent, result)
        print_with_indent([
            '"name": {},'.format(self.name),
            '"signal": {},'.format(signal),
            '"warn": {},'.format(self.warn),
            '"crit": {},'.format(self.crit),
        ], indent + 1, result)
        if self.flaps:
            print_with_indent([
                '"flaps": {},'.format(self.flaps),
            ], indent + 1, result)
        print_with_indent(['"tags": ['], indent + 1, result)
        print_with_indent([
            '"stagectl",',
            '"yp-downtime-{}",'.format(YP_DOWNTIME_HASH),
        ], indent + 2, result)
        print_with_indent(['],'], indent + 1, result)
        print_with_indent(['}) %>'], indent, result)
        print_with_indent(['<% if cluster is defined %>'], indent, result)
        print_with_indent([
            '<% do info[-1]["tags"].append("yp-" + cluster + "-downtime-{}") %>'.format(YP_DOWNTIME_HASH)
        ], indent + 1, result)
        print_with_indent([
            '<% do info[-1]["tags"].append("a_geo_" + cluster) %>'
        ], indent + 1, result)
        print_with_indent(['<% endif %>'], indent, result)
        print_with_indent(['<% if ctype is defined %>'], indent, result)
        print_with_indent([
            '<% do info[-1]["tags"].append("stagectl-" + ctype) %>'
        ], indent + 1, result)
        print_with_indent(['<% endif %>'], indent, result)



class Signal(object):
    def __init__(self, title, name, alert=None, color=None, y_axis=None, width=None, active=None):
        self.title = title
        self.name = name
        self.alert = alert
        self.color = color
        self.y_axis = y_axis
        self.width = width
        self.active = active

    def print_yasm(self, indent, result):
        print_signal(self.title, self.name, indent, result, self.color, self.y_axis, self.width, self.active)

    def print_alert_info(self, indent, result):
        if self.alert is not None:
            self.alert.print_info('"' + self.name + '"', indent, result)


class PerClusterSignal(object):
    def __init__(self, title, prefix, suffix, alert=None):
        self.title = title
        self.prefix = prefix
        self.suffix = suffix
        self.alert = alert

    def print_yasm(self, indent, result):
        print_with_indent(['<% for cluster in clusters %>'], indent, result)
        print_signal(self.title, ".".join([self.prefix, '<< cluster >>', self.suffix]), indent + 1, result)
        print_with_indent(['<% endfor %>'], indent, result)

    def print_alert_info(self, indent, result):
        if self.alert is not None:
            print_with_indent(['<% for cluster in clusters %>'], indent, result)
            # Jinja is unable to render <<cluster>> inside <% do %>
            signal = "['{}.', cluster, '.{}']|join()".format(self.prefix, self.suffix)
            self.alert.print_info(signal, indent + 1, result)
            print_with_indent(['<% endfor %>'], indent, result)


class Chart(object):
    def __init__(self, id, title, signals, width=1, stacked=None):
        self.id = id
        self.title = title
        self.signals = signals
        self.width = width
        self.stacked = stacked

    def print_yasm(self, indent, result, next_row=False):
        print_with_indent(['{'], indent, result)
        print_with_indent([
            '"id": "{}",'.format(self.id),
            '"type": "graphic",',
            '"title": "{}",'.format(self.title),
        ], indent + 1, result)
        if self.stacked:
            print_with_indent([
                '"stacked": true,'
            ], indent + 1, result)
        print_with_indent([
            '"width": {},'.format(self.width),
            '"height": 1,',
            '"row": << table[\'row\'] >>,',
            '"col": << table[\'col\'] >>,',
            '"signals": [',
        ], indent + 1, result)
        for signal in self.signals:
            signal.print_yasm(indent + 2, result)
        print_with_indent(['],'], indent + 1, result)
        print_with_indent(['},'], indent, result)
        print_with_indent(['<< inc_col() >>'], indent, result)
        if next_row:
            print_with_indent(['<< inc_row() >>'], indent, result)


def object_type_statistics_chart(object_type, unready_signal=None):
    signals = [
        PerClusterSignal("Managed count on << cluster >>", "unistat-" + object_type["graph_name"], "managed_objects_count_axxx"),
        PerClusterSignal("Garbage count on << cluster >>", "unistat-" + object_type["graph_name"], "garbage_objects_count_axxx"),
        PerClusterSignal("Running operations count on << cluster >>", "unistat-" + object_type["graph_name"], "running_operations_count_axxx"),
        PerClusterSignal("Failed count on << cluster >>", "unistat-" + object_type["graph_name"], "objects_with_failed_operations_axxx"),
        Signal("Garbage limit", f"unistat-{object_type['graph_name']}.garbage_limit_axxx"),
        Signal("Garbage limit (first sync cycle)", f"unistat-{object_type['graph_name']}.initial_garbage_limit_axxx"),
    ]
    if unready_signal is not None:
        signals.append(unready_signal)
    return Chart(object_type['id'], "{} statistics".format(object_type["description"]), signals)

common_charts = [
    Chart('cpu', "CPU (cores, max)", [
        Signal("user", "quant(portoinst-cpu_usage_cores_hgram,max)", color='#0089BA'),
        Signal("system", "quant(portoinst-cpu_usage_system_cores_hgram,max)", color='#6574BD'),
        Signal("wait", "quant(portoinst-cpu_wait_cores_hgram,max)", color='#A3569C'),
        Signal("limit", "quant(portoinst-cpu_limit_cores_thhh,max)", color='#CA5C75', y_axis=1, width=2),
        Signal("guarantee", "quant(portoinst-cpu_guarantee_cores_thhh,max)", color='#D4584F', y_axis=1, width=2),

    ], stacked=True),
    Chart('memory', "Memory usage (mb)", [
        Signal("Total Max (JVM)", "conv(unistat-jvm.memory.total.max_axxx, Mi)", color='#D65DB1', y_axis=1, width=2, active=True),
        Signal("Total Used (JVM)", "conv(sum(unistat-jvm.memory.heap.used_axxx, unistat-jvm.memory.non-heap.used_axxx), Mi)", color='#FF6F91', y_axis=1, width=2, active=True,
               alert=Alert('"heap_usage"', [35000, 39000], crit=[39000, None], flaps={
                   "critical": 3000,
                   "boost": 0,
                   "stable": 600
               })),
        Signal("Anon Usage", "mul(portoinst-anon_usage_gb_txxx, 1024)", color='#FF9671', y_axis=1, width=2, active=True),
        Signal("Anon Limit", "conv(quant(portoinst-anon_limit_slot_hgram,max), Mi)", color='#FFC75F', y_axis=1, width=2, active=True),
        Signal("Memory Limit", "conv(quant(portoinst-memory_limit_slot_hgram,max), Mi)", color='#FFC75F', y_axis=1, width=2, active=True),
        Signal("Memory Guarantee", "conv(quant(portoinst-memory_guarantee_slot_hgram, max), Mi)", color='#4E8397', y_axis=1, width=2, active=True),
    ], stacked=True),
    Chart('threads', "Threads", [
        Signal("Blocked", "unistat-jvm.thread-states.blocked.count_axxx"),
        Signal("Total count", "unistat-jvm.thread-states.count_axxx"),
        Signal("Waiting", "unistat-jvm.thread-states.waiting.count_axxx"),
        Signal("Timed waiting", "unistat-jvm.thread-states.timed_waiting.count_axxx"),
        Signal("Runnable", "jvm.gc.stw_pause_5sec_dhhh"),
    ]),
    Chart('memory GC jvm', "Memory GC jvm (mb)", [
        Signal("Metaspace", "conv(unistat-jvm.memory.pools.Metaspace.used_axxx, Mi)", color='#845EC2', active=True),
        Signal("Compressed Class Space", "conv(unistat-jvm.memory.pools.Compressed-Class-Space.used_axxx, Mi)", color='#0081CF', active=True),
        Signal("G1 Eden Space", "conv(unistat-jvm.memory.pools.G1-Eden-Space.used_axxx, Mi)", color='#0089BA', active=True),
        Signal("G1 Old Gen", "conv(unistat-jvm.memory.pools.G1-Old-Gen.used_axxx, Mi)", color='#008E9B', active=True),
        Signal("G1 Survivor Space", "conv(unistat-jvm.memory.pools.G1-Survivor-Space.used_axxx, Mi)", color='#008F7A', active=True),
    ], stacked=True),
    Chart('oom', "OOM (sum)", [
        Signal("count", "portoinst-ooms_summ", color='#37bff2'),
    ]),
    Chart('gc-swt', "GC STW", [
        Signal("1sec", "div(quant(unistat-jvm.gc.stw_pause_1sec_dhhh, max), 1000000)",
               alert=Alert('"stw_pause"', warn=[1, 2], crit=[2, None])),
        Signal("5sec", "div(quant(unistat-jvm.gc.stw_pause_5sec_dhhh, max), 5000000)"),
    ]),
    Chart('logs', "Logs count", [
        Signal("Error", "unistat-log.events.error.count_dmmm", color='#e85b4e', active=True,
               alert=Alert('"log_events_error_count"', warn=[1, 5], crit=[5, None], flaps={
                    "critical": 900,
                    "boost": 0,
                    "stable": 150
        })),
        Signal("Warn", "unistat-log.events.warn.count_dmmm", color='#f6ab31', active=True),
        Signal("Info", "unistat-log.events.info.count_dmmm", color='#169833', active=True),
        Signal("Debug", "unistat-log.events.debug.count_dmmm", color='#c95edd', active=True),
        Signal("Trace", "unistat-log.events.trace.count_dmmm", color='#37bff2', active=True),
    ], stacked=True),
    Chart('page-fault', "Page faults", [
        Signal("count", "porto_page_faults"),
    ]),
    Chart('net', "Network", [
        Signal("RX Mb/s", "conv(portoinst-net_iface_veth_rx_bytes_tmmv, Mi)"),
        Signal("TX Mb/s", "conv(portoinst-net_iface_veth_tx_bytes_tmmv, Mi)"),
        Signal("RX Packets/s", "portoinst-net_iface_veth_rx_packets_tmmv"),
        Signal("TX Packets/s", "portoinst-net_iface_veth_tx_packets_tmmv"),
        Signal("RX drops/s", "portoinst-net_iface_veth_rx_drops_tmmv"),
        Signal("TX drops/s", "portoinst-net_iface_veth_tx_drops_tmmv"),
        Signal("Guarantee Mb/s", "min(portoinst-net_guarantee_mb_summ)"),
        Signal("Limit Mb/s", "min(portoinst-net_limit_mb_summ)"),
    ]),
    Chart('unistat', "Unistat", [
        Signal("/unistat calls", "unistat-unistat_count_dmmm"),
        Signal("Metrics count", "unistat-metrics_count_axxx"),
        Signal("Metrics collection time (ms)", "unistat-unistat_execution_time_ms_dmmm"),
    ]),
]

stagectl_charts = [
    Chart('leadership', "Replica leadership status (boolean)", [
        Signal("YT lock acquired", "unistat-stagectl.leader_lock_acquired_ammx",
               alert=Alert('"yt_leader_lock"', warn=[0, 0], crit=[2, None])),
        Signal("Processing is allowed", "unistat-stagectl.processing_allowed_axxx"),
    ]),
    Chart('aux', "Epoch / Restarts", [
        Signal("Epoch", "unistat-stagectl.current_epoch_axxx"),
    ]),
    Chart('errors', "Errors", [
        Signal("Engine: failed iterations", "unistat-engine.failed_iterations_dmmm",
               alert=Alert('"engine_failed_iterations"', warn=[1, 1], crit=[1, None])),
        Signal("Engine: failed Project/Stages load from YP", "unistat-engine.yp_objects_load_errors_dmmm",
               alert=Alert('"engine_exceptions_count"', warn=[1, 9], crit=[10, None])),
        Signal("Engine: main loop hung", "unistat-engine.hung_iteration_dmmm",
               alert=Alert('"engine_main_loop_hung_count"', warn=[1, 1], crit=[1, None])),
        Signal("Engine: failed stage child objects GC", "unistat-engine.child_objects_gc_errors_dmmm",
               alert=Alert('"engine_child_objects_gc"', warn=[1, 1], crit=[1, None])),
        Signal("Engine: failed to wait for SerialExecutor tasks completion", "unistat-engine.serial_executor_tasks_wait_timeouts_dmmm"),
        Signal("All LOG errors (from StageCtl /unistat )", "unistat-log.events.error.count_dmmm"),
        Signal("All LOG errors", "unistat-all_log_errors_count_dmmm"),

        Signal("Failed add_deploy_child calls", "unistat-relations.failed_add_deploy_child_count_dmmm"),
        Signal("Failed relation remove", "unistat-relations.failed_remove_relation_count_dmmm"),
        Signal("Failed full relations reload", "unistat-relations.failed_relations_reload_count_dmmm"),
        Signal("Infra resources config validation errors", "unistat-config.validation.errors_dmmm",
               alert=Alert('"infra_resources_invalid_config"', warn=[1, 1], crit=[1, None])),
        PerClusterSignal("Failed masters discovery requests on << cluster >>", "unistat-yp", "failed_requests_count_dmmm",
                         alert=Alert("['failed_yp_masters_discovery_requests_count_on_', cluster]|join('_')", warn=[1, 1], crit=[2, None])),
        Signal("Failed masters discovery requests on xdc", "unistat-yp.xdc.failed_requests_count_dmmm",
               alert=Alert('"failed_yp_masters_discovery_requests_count_on_xdc"', warn=[1, 1], crit=[2, None]))
    ]),
    Chart('cycle-duration', "Sync cycles duration (milliseconds)", [
        Signal("Engine: last iteration", "unistat-engine.last_iteration_duration_ms_axxx"),
        Signal("Engine: interval between iterations", "unistat-engine.interval_between_iterations_ms_axxx"),
        Signal("Engine: project/stages load time from YP", "unistat-engine.yp_objects_load_time_ms_axxx"),
        Signal("Engine: All stage identifiers load time", "unistat-engine.yp_stage_ids_load_time_ms_axxx"),
        Signal("Engine: stages processing", "unistat-engine.last_stages_processing_time_ms_axxx"),
        Signal("Engine: Docker/Sandbox resolving", "unistat-engine.last_resolvers_wait_time_ms_axxx"),
        Signal("Engine: stage child objects processing", "unistat-engine.last_child_objects_processing_time_ms_axxx"),
        Signal("Engine: stage child objects GC", "unistat-engine.last_child_objects_gc_time_ms_axxx"),
        Signal("Engine: stage child objects update to YP", "unistat-engine.last_child_objects_update_time_ms_axxx"),
        Signal("Last relations reload", "unistat-relations.reload_time_ms_axxx"),
        Signal("SerialExecutor: Callbacks total execution time", "unistat-serial_executor.callbacks_total_execution_time_ms_dmmm"),
    ]),
    Chart('engine-cycles', "Engine cycles", [
      Signal("Engine: finished iterations", "unistat-engine.finished_iterations_dmmm"),
    ]),
    Chart('object-errors', "Object errors", [
        PerClusterSignal("updateObject: {} on << cluster >>".format(item["description"]), "unistat-" + item["graph_name"], "modify_operation_errors_count_dmmm") for item in PerClusterObjects.ALL
    ] + [
        Signal("updateObject: MultiCluster replica sets", "unistat-mcrs.modify_operation_errors_count_dmmm"),
        Signal("Failed Stage.Status updates", "unistat-stages.send_status_requests_failed_dmmm"),
    ] + [
        PerClusterSignal("watchObject: {} on << cluster >>".format(item["description"]), "unistat-" + item["graph_name"], "watch_objects_errors_dmmm") for item in PerClusterObjects.WATCHES_ONLY
    ] + [
        Signal("watchObject: MultiCluster replica sets", "unistat-mcrs.watch_objects_errors_dmmm"),
        Signal("watchObject: Stages", "unistat-stages.watch_objects_errors_dmmm"),
    ] + [
        PerClusterSignal("getObjects: {} on << cluster >>".format(item["description"]), "unistat-" + item["graph_name"], "get_objects_errors_dmmm") for item in PerClusterObjects.WATCHES_ONLY
    ] + [
        Signal("getObjects: MultiCluster replica sets", "unistat-mcrs.get_objects_errors_dmmm"),
        Signal("getObjects: Stages", "unistat-stages.get_objects_errors_dmmm"),
    ]),
    Chart('objects-cycle-duration', "Objects load time (milliseconds)", [
        PerClusterSignal("{} on << cluster >>".format(item["description"]), "unistat-" + item["graph_name"], "yp_objects_load_time_ms_axxx") for item in PerClusterObjects.ALL
    ] + [
        Signal("MultiCluster replica sets", "unistat-mcrs.yp_objects_load_time_ms_axxx"),
        Signal("Stages", "unistat-stages.yp_objects_load_time_ms_axxx"),
        Signal("Projects", "unistat-projects.yp_objects_load_time_ms_axxx"),
    ]),
    Chart('object-operations', "Object write operations", [
        PerClusterSignal("{} on << cluster >>".format(item["description"]), "unistat-" + item["graph_name"], "modify_operation_count_dmmm") for item in PerClusterObjects.ALL
    ] + [
        Signal("MultiCluster replica sets", "unistat-mcrs.modify_operation_count_dmmm"),
    ]),
    Chart('object-updates', "Object updates", [
        PerClusterSignal("{} on << cluster >>".format(item["description"]), "unistat-" + item["graph_name"], "updated_objects_count_dxxm") for item in PerClusterObjects.WATCHES_ONLY
    ] + [
        Signal("MultiCluster replica sets", "unistat-mcrs.updated_objects_count_dxxm"),
        Signal("Stages (all updates)", "unistat-stages.updated_objects_count_dxxm"),
        Signal("Stages (/spec updates)", "unistat-stages.stage_spec_updates_count_dmmm"),
        #Signal("Projects", "unistat-projects.updated_objects_count_dxxm")
    ]),
    Chart('object-full-reloads', "Object full reloads (selectObjects requests)", [
        PerClusterSignal("{} on << cluster >>".format(item["description"]), "unistat-" + item["graph_name"], "full_reload_count_dmmm") for item in PerClusterObjects.WATCHES_ONLY
    ] + [
        Signal("MultiCluster replica sets", "unistat-mcrs.full_reload_count_dmmm"),
        Signal("Stages", "unistat-stages.full_reload_count_dmmm"),
        #Signal("Projects", "unistat-projects.full_reload_count_dmmm")
    ]),
    Chart('object-watch-time', "watchObjects (milliseconds)", [
        PerClusterSignal("{} on << cluster >>".format(item["description"]), "unistat-" + item["graph_name"], "watch_objects_time_ms_axxx") for item in PerClusterObjects.WATCHES_ONLY
    ] + [
        Signal("MultiCluster replica sets", "unistat-mcrs.watch_objects_time_ms_axxx"),
        Signal("Stages", "unistat-stages.watch_objects_time_ms_axxx"),
        #Signal("Projects", "unistat-projects.watch_objects_time_ms_axxx")
    ]),
    Chart('object-getobjects-time', "getObjects (milliseconds)", [
        PerClusterSignal("{} on << cluster >>".format(item["description"]), "unistat-" + item["graph_name"], "get_objects_time_ms_axxx") for item in PerClusterObjects.WATCHES_ONLY
    ] + [
        Signal("MultiCluster replica sets", "unistat-mcrs.get_objects_time_ms_axxx"),
        Signal("Stages", "unistat-stages.get_objects_time_ms_axxx"),
        #Signal("Projects", "unistat-projects.get_objects_time_ms_axxx")
    ]),
    Chart('resource-delay', "Infra resource delays (seconds)", [
      Signal("Delay resolving {resource} resource".format(resource=resource), "unistat-resources.{resource}.delay_seconds_axxx".format(resource=resource),
             alert=Alert('"{}_delay"'.format(resource), warn=[1800, 3600], crit=[3600, None]))
      for resource in SANDBOX_RESOURCES_UPDATABLE
    ]),
    Chart('gc', "Garbage collection stopped (boolean)", [
        PerClusterSignal("{} on << cluster >>".format(item["description"]), "unistat-" + item["graph_name"], "garbage_limit_exceeded_axxx",
                         alert=Alert("['{}', cluster, 'garbage']|join('_')".format(item["abbreviation"]),
                                   warn=[0.5, 0.75], crit=[0.75, None])) for item in PerClusterObjects.ALL
    ] + [
        Signal("Garbage stopped on mcrs", "unistat-mcrs.garbage_limit_exceeded_axxx"),
    ]),
    Chart('list-delay', "Object listing delays (seconds)", [
        PerClusterSignal("Delay listing {} on << cluster >>".format(item["description"]),
                       "unistat-" + item["graph_name"], "list_delay_seconds_axxx",
                       alert=Alert("['{}', cluster, 'delay']|join('_')".format(item["abbreviation"]),
                                   warn=[360, 360], crit=[360, None]))
        for item in PerClusterObjects.ALL
    ] + [
        Signal("Delay listing mcrs on xdc", "unistat-mcrs.list_delay_seconds_axxx",
               alert=Alert('"mcrs_delay"', warn=[180, 180], crit=[180, None])),
        Signal("Delay since last successful stages sync", "unistat-stages.list_delay_seconds_axxx",
               alert=Alert('"stage_delay"', warn=[180, 180], crit=[180, None])),
        Signal("Engine: delay since current iteration start", "trunc(div(unistat-engine.current_iteration_duration_ms_axxx, 1000))",
               alert=Alert('"current_engine_iteration_time_seconds"', warn=[180, 180], crit=[180, None])),
        Signal("SerialExecutor: Current action delay", "trunc(div(unistat-serial_executor.current_action_delay_ms_axxx, 1000))",
               alert=Alert('"current_serial_executor_callback_execution_time_seconds"', warn=[20, 30], crit=[30, None])),
    ]),
    Chart('stage-status-updates', "Stage status updates statistics", [
        Signal("Queue size (stages scheduled for update)", "unistat-stages.status_update_scheduled_axxx"),
        Signal("In progress", "unistat-stages.status_update_in_progress_axxx"),
        Signal("Requests (total)", "unistat-stages.send_status_requests_total_dmmm"),
        Signal("RPS limit", "unistat-stages.send_status_rps_limit_axxx"),
    ]),
    Chart('stage', "Stages statistics", [
        Signal("Total", "unistat-stages.total_count_axxx"),
        Signal("Valid", "unistat-stages.valid_count_axxx"),
        Signal("Invalid", "unistat-stages.invalid_count_axxx"),
        Signal("Parsing failed", "unistat-stages.parsing_failed_count_axxx"),
        Signal("Stages with failed status update", "unistat-stages.failed_send_status_count_axxx",
               alert=Alert('"stage_status_failed_send"', warn=[3, 7], crit=[7, None])),
    ]),
    Chart('deploy-unit', "Deploy units statistics", [
        Signal("Total deploy untis", "unistat-stages.total_deploy_unit_count_axxx"),
        Signal("Deploy units with ready prerequisites", "unistat-stages.prerequisites_ready_deploy_unit_count_axxx"),
        Signal("Deploy units with unready prerequisites", "unistat-stages.prerequisites_unready_deploy_unit_count_axxx",
               alert=Alert('"prerequisites"', warn=[350, 650], crit=[650, None])),
    ]),
    Chart('mcrs', "Multi cluster replica sets statistics", [
        Signal("Managed count", "unistat-mcrs.managed_objects_count_axxx"),
        Signal("Failed operations count", "unistat-mcrs.objects_with_failed_operations_axxx"),
        Signal("Running operations count", "unistat-mcrs.running_operations_count_axxx"),
        Signal("Garbage count", "unistat-mcrs.garbage_objects_count_axxx",
               alert=Alert('"mcrs_garbage"', warn=[10.0, 20.0], crit=[20.0, None], flaps={
                   'stable': 90,
                   'critical': 270
               })),
        Signal("Garbage stopped", "unistat-mcrs.garbage_limit_exceeded_axxx"),
        Signal("Garbage limit", "unistat-mcrs.garbage_limit_axxx"),
        Signal("Garbage limit (first sync cycle)", "unistat-mcrs.initial_garbage_limit_axxx"),
    ]),
    Chart('serial-executor', "SerialExecutor", [
        Signal("Active (running) futures", "unistat-serial_executor.active_futures_axxx"),
        Signal("Submitted futures (total)", "unistat-serial_executor.submitted_futures_total_dmmm"),
        Signal("Failed futures", "unistat-serial_executor.failed_futures_dmmm"),
        Signal("Retries (for failed futures)", "unistat-serial_executor.failed_futures_retries_dmmm"),
        Signal("Active (running) actions", "unistat-serial_executor.active_actions_axxx"),
        Signal("Submitted actions (total)", "unistat-serial_executor.submitted_actions_total_dmmm"),
        Signal("Failed actions", "unistat-serial_executor.failed_actions_dmmm"),
    ]),

    object_type_statistics_chart(PerClusterObjects.RS, Signal("Unready (both RS and MCRS)", "unistat-stages.unready_replica_set_count_axxx",
                alert=Alert('"unready_replica_sets"', warn=[1000, 1500], crit=[1500, None]))),
    object_type_statistics_chart(PerClusterObjects.ES,
         Signal("Unready", "unistat-stages.unready_endpoint_set_count_axxx",
                alert=Alert('"endpoint_set_updating"', warn=[25, 50], crit=[50, None]))),
    object_type_statistics_chart(PerClusterObjects.HPA, None),
    object_type_statistics_chart(PerClusterObjects.DR, Signal("Unready dynamic resource", "unistat-stages.unready_dynamic_resource_count_axxx")),
    #not in production yet....
    # object_type_statistics_chart(PerClusterObjects.RC, None),
    Chart('yp', "YP statistics", [
        Signal("Alive masters on xdc", "unistat-yp.xdc.alive_masters_count_annv",
               alert=Alert('"low_count_of_alive_yp_masters_on_xdc"', warn=[3, 3], crit=[0, 2])),
        PerClusterSignal("Alive masters on << cluster >>",
                         "unistat-yp", "alive_masters_count_annv",
                         alert=Alert("['low_count_of_alive_yp_masters_on_', cluster]|join('_')", warn=[3, 3], crit=[0, 2])),
        Signal("Masters discovery requests on xdc", "unistat-yp.xdc.requests_count_dmmm"),
        PerClusterSignal("Masters discovery requests on << cluster >>", "unistat-yp", "requests_count_dmmm"),
    ]),
    Chart('relations', "Relations (Stage -> RS/MCRS)", [
        Signal("Missed relation events for existed RS", "unistat-relations.missed_relation_events_count_dmmm"),
        Signal("add_deploy_child calls", "unistat-relations.add_deploy_child_count_dmmm"),
        Signal("Added relations count", "unistat-relations.success_add_deploy_child_count_dmmm"),
        Signal("Removed relations count", "unistat-relations.remove_relation_count_dmmm"),
        Signal("Full reload count", "unistat-relations.relations_reload_count_dmmm"),
    ]),
] + common_charts


resource_charts = [
    Chart('objects', "Objects count", [
        Signal("Cached in memory: " + cached_object_type, f"unistat-cache.{cached_object_type}.count_axxx") for cached_object_type in CACHED_OBJECT_TYPE
    ] + [
        Signal("Stages", "unistat-stages.total_count_axxx"),
        Signal("Deploy untis", "unistat-stages.total_deploy_unit_count_axxx"),
        Signal("MultiCluster RS", "unistat-mcrs.managed_objects_count_axxx"),
        Signal("Docker Images count", "unistat-docker.images_count_axxx"),
        Signal("Relations count", "unistat-relations.count_axxx"),
    ]),
    Chart('yt', "YT requests", [
        Signal("Total", "unistat-yt.total_transactions_count_dmmm"),
        Signal("Completed", "unistat-yt.completed_transactions_count_dmmm"),
        Signal("Failed", "unistat-yt.failed_transactions_count_dmmm"),
    ]),
] +[
    Chart(cached_object_type + "_cache", f"Cache: {cached_object_type}", [
        Signal(signal, f"unistat-cache.{cached_object_type}.{signal}_dmmm") for signal in ["put", "remove", "get_hit", "get_miss"]
    ]) for cached_object_type in CACHED_OBJECT_TYPE
] + [
    Chart(f'{obj_type}-http', f"Http request statistics: {obj_type}", [
        Signal("Send requests count", f"unistat-http.{obj_type}.request_dmmm"),
        Signal("Failed requests count", f"unistat-http.{obj_type}.json-parse-error_dmmm"),
        Signal("Failed response parsing count", f"unistat-http.{obj_type}.json-parse-error_dmmm"),
    ]) for obj_type in ["docker", "sandbox"]
] + [
    Chart('docker', "Docker resolver statistics", [
        Signal("Images scheduled", "unistat-docker.scheduled_requests_count_ammx"),
        Signal("Active requests count", "unistat-docker.active_requests_count_ammx"),
        Signal("Send requests count", "unistat-docker.send_requests_count_dmmm"),
        Signal("Failed requests count", "unistat-docker.failed_requests_count_dmmm"),
        Signal("Succeed requests count", "unistat-docker.succeed_requests_count_dmmm"),
        Signal("Force resolve requests count", "unistat-docker.force_resolve_count_dmmm"),
    ])
]

auth_charts = [
    Chart('leadership', "Replica leadership status (boolean)", [
        Signal("YT lock acquired", "unistat-authctl.leader_lock_acquired_ammx",
               alert=Alert('"yt_leader_lock"', warn=[0, 0], crit=[2, None])),
        Signal("Processing is allowed", "unistat-authctl.processing_allowed_axxx"),
    ]),
    Chart('epoch', "Epoch", [
        Signal("Epoch", "unistat-authctl.current_epoch_axxx"),
    ]),
    Chart('errors', "Errors", [
        Signal("Failed IDM nodes sync cycles", "unistat-nodes.failed_iterations_dmmm"),

        Signal("IDM errors (reply != OK or exception)", "unistat-idm.errors_count_dmmm"),
        Signal("Failed outgoing IDM API requests", "unistat-idm.failed_api_requests_dmmm"),

        Signal("Failed Staff API requests", "unistat-staff.failed_api_requests_dmmm"),
        Signal("Failed staff groups cache cycles", "unistat-staff.failed_iterations_dmmm"),

        Signal("Failed Nanny cycles", "unistat-nanny.failed_iterations_dmmm"),
        Signal("Failed Nanny API requests", "unistat-nanny.failed_api_requests_dmmm"),
        Signal("Failed Nanny pod_set group updates", "unistat-nanny.failed_group_updates_dmmm"),
        Signal("Exception in AuthAttrs sync thread", "unistat-nanny.sync_queue_exceptions_dmmm"),
        Signal("Failed AuthAttrs updates", "unistat-nanny.failed_auth_attrs_updates_dmmm"),

        Signal("Failed roles cache update cycles", "unistat-roles.failed_iterations_dmmm"),
        Signal("Failed /get-project-acl", "unistat-idm.get_project_acl_failed_requests_count_dmmm"),

        Signal("YP errors", "unistat-yp_errors_count_axxx"),
        Signal("YP getObject errors (Groups)", "unistat-groups.get_objects_errors_dmmm"),
        Signal("YP getObject errors (Stages)", "unistat-stages.get_objects_errors_dmmm"),
        Signal("YP getObject errors (Nanny services)", "unistat-nanny_services.get_objects_errors_dmmm"),
        Signal("YP getObject errors (Projects)", "unistat-projects.get_objects_errors_dmmm"),

        PerClusterSignal("Failed masters discovery requests on << cluster >>", "unistat-yp", "failed_requests_count_dmmm",
                         alert=Alert("['failed_yp_masters_discovery_requests_count_on_', cluster]|join('_')", warn=[1, 1], crit=[2, None])),

        Signal("Failed masters discovery requests on xdc", "unistat-yp.xdc.failed_requests_count_dmmm",
               alert=Alert('"failed_yp_masters_discovery_requests_count_on_xdc"', warn=[1, 1], crit=[2, None])),

        Signal("Failed YT lock acquire attempts", "unistat-authctl.failed_leader_lock_acquire_attempts_dmmm"),
        Signal("All LOG errors", "unistat-log.events.error.count_dmmm"),
    ]),
    Chart('cycle-duration', "Sync cycles duration (milliseconds)", [
        Signal("YP Objects load (all)", "unistat-nodes.yp_objects_load_time_ms_axxx"),
        Signal("YP Objects load: Projects", "unistat-projects.yp_objects_load_time_ms_axxx"),
        Signal("YP Objects load: Stages", "unistat-stages.yp_objects_load_time_ms_axxx"),
        Signal("YP Objects load: Nanny services", "unistat-nanny_services.yp_objects_load_time_ms_axxx"),
        Signal("YP Objects load: Groups", "unistat-groups.yp_objects_load_time_ms_axxx"),
        Signal("IDM Nodes sync cycles", "unistat-nodes.last_iteration_duration_ms_axxx"),
        Signal("Roles cache", "unistat-roles.last_iteration_duration_ms_axxx"),
        Signal("Nanny pod_set ACLs (seconds)", "trunc(div(unistat-nanny.last_iteration_duration_ms_axxx,1000))"),
    ]),
    Chart('cycle-delays', "Sync cycle delays (seconds)", [
        Signal("IDM Nodes", "trunc(div(unistat-nodes.current_iteration_duration_ms_axxx,1000))"),
        Signal("Nanny pos_set ACLs", "trunc(div(unistat-nanny.current_iteration_duration_ms_axxx,1000))"),
        Signal("Nanny auth_attrs", "trunc(div(unistat-nanny.current_nanny_services_sync_iteration_duration_ms_axxx,1000))"),
        Signal("Roles cache", "trunc(div(unistat-roles.current_iteration_duration_ms_axxx,1000))"),
        Signal("Staff groups", "trunc(div(unistat-staff.current_iteration_duration_ms_axxx,1000))"),
    ]),
    Chart('objects', "Objects", [
        Signal("IDM role nodes", "unistat-total_roles_axxx"),
        Signal("Nodes added", "unistat-nodes.roles_added_dmmm"),
        Signal("Nodes removed", "unistat-nodes.roles_removed_dmmm"),
        Signal("Nodes relocated", "unistat-nodes.roles_relocated_dmmm"),
        Signal("AuthCtl start: loaded IDM nodes", "unistat-idm.loaded_nodes_count_axxx"),
        Signal("AuthCtl start: total IDM nodes", "unistat-idm.total_nodes_count_axxx"),
        Signal("Active role nodes (role is assigned to any user/group)", "unistat-roles.active_axxx"),

        Signal("Role <-> User/Group links (role subjects)", "unistat-total_roles_subjects_axxx"),
        Signal("Projects", "unistat-projects.yp_objects_count_axxx"),
        Signal("Projects: with AccountId == TMP", "unistat-tmp_quota_usage_axxx"),
        Signal("Projects: without OWNER", "unistat-roles.projects_with_missed_owner_axxx"),
        Signal("Stages", "unistat-stages.yp_objects_count_axxx"),
        Signal("Stages: with role overrides", "unistat-stages_count_axxx"),
        Signal("Stages: with Stage.AccountId != Project.AccountId", "unistat-quota_mismatch_axxx"),

        Signal("Nanny services", "unistat-nanny.nanny_services_axxx"),
        Signal("Nanny services loaded in current cycle", "unistat-nanny.current_nanny_service_offset_axxx"),
        Signal("Nanny services (YP)", "unistat-nanny_services.yp_objects_count_axxx"),

        Signal("Staff groups", "unistat-staff_groups_count_axxx"),
        Signal("YP users", "unistat-yp.users_count_axxx"),
        Signal("YP groups: All", "unistat-yp.all_groups_count_axxx"),
        Signal("YP groups: Deploy", "unistat-groups.yp_objects_count_axxx"),
        Signal("YP groups: IDM (synced staff/abc groups)", "unistat-yp.idm_groups_count_axxx"),

        PerClusterSignal("YP nanny pod_set groups: << cluster >>", "unistat-yp", "nanny_groups_count_axxx"),
    ]),
    Chart('queues', "Queue sizes", [
      Signal("YP groups sync", "unistat-yp_async_queue_size_axxx"),
      Signal("IDM role nodes to add", "unistat-nodes.new_roles_count_axxx"),
      Signal("IDM role nodes to remove", "unistat-nodes.garbage_roles_count_axxx"),
      Signal("IDM role nodes to move (change project)", "unistat-nodes.moved_roles_count_axxx"),
      Signal("IDM add node batches", "unistat-nodes.queue_size_to_add_node_axxx"),
      Signal("IDM remove node batches", "unistat-nodes.queue_size_to_remove_node_axxx"),
      Signal("Nanny services sync", "unistat-nanny.sync_queue_size_axxx"),
      Signal("Nanny services sync in progress", "unistat-nanny.sync_in_progress_queue_size_axxx"),
    ]),
    Chart('cycles', "Sync cycles count", [
        Signal("IDM Nodes", "unistat-nodes.finished_iterations_dmmm"),
        Signal("Nanny pod_set ACLs", "unistat-nanny.finished_iterations_dmmm"),
        Signal("Nanny auth_attrs", "unistat-nanny.finished_nanny_services_sync_iterations_dmmm"),
        Signal("Roles cache", "unistat-roles.finished_iterations_dmmm"),
        Signal("Staff groups", "unistat-staff.finished_iterations_dmmm"),
    ]),
    Chart('requests', "Requests", [
        Signal("Incoming /get-project-acl", "unistat-idm.get_project_acl_requests_count_dmmm"),
        Signal("Nanny API requests", "unistat-nanny.api_requests_dmmm"),
        Signal("Staff API requests", "unistat-staff.api_requests_dmmm"),
        Signal("IDM API requests", "unistat-idm.api_requests_dmmm"),
    ]),
    Chart('nanny-requests', "Nanny requests", [
        Signal("Started group updates", "unistat-nanny.group_updates_dmmm"),
        Signal("Succeeded group updates", "unistat-nanny.succeeded_group_updates_dmmm"),
        Signal("AuthAttrs updates", "unistat-nanny.auth_attrs_updates_dmmm"),
        Signal("API: get service", "unistat-nanny.api.get_service_dmmm"),
        Signal("API: get services range (limit = 200)", "unistat-nanny.api.get_services_range_dmmm"),
        Signal("API: get all services (ListSummaries)", "unistat-nanny.api.get_all_services_dmmm"),
        Signal("API: update service", "unistat-nanny.api.update_service_dmmm"),
    ]),
    Chart('idm', "Incoming IDM requests", [
        Signal("/info", "unistat-idm.info_requests_count_dmmm"),
        Signal("/add-role", "unistat-idm.add_role_requests_count_dmmm"),
        Signal("/remove-role", "unistat-idm.remove_role_requests_count_dmmm"),
        Signal("/get-all-roles", "unistat-idm.get_all_roles_requests_count_dmmm"),
        Signal("/add-batch-memberships", "unistat-idm.add_batch_memberships_requests_count_dmmm"),
        Signal("/remove-batch-memberships", "unistat-idm.remove_batch_memberships_requests_count_dmmm"),
        Signal("/get-memberships", "unistat-idm.get_memberships_requests_count_dmmm"),
    ]),
    Chart('idm-errors', "Failed imcoming IDM requests", [
        Signal("/info", "unistat-idm.info_failed_requests_count_dmmm"),
        Signal("/add-role", "unistat-idm.add_role_failed_requests_count_dmmm"),
        Signal("/remove-role", "unistat-idm.remove_role_failed_requests_count_dmmm"),
        Signal("/get-all-roles", "unistat-idm.get_all_roles_failed_requests_count_dmmm"),
        Signal("/add-batch-memberships", "unistat-idm.add_batch_memberships_failed_requests_count_dmmm"),
        Signal("/remove-batch-memberships", "unistat-idm.remove_batch_memberships_failed_requests_count_dmmm"),
        Signal("/get-memberships", "unistat-idm.get_memberships_failed_requests_count_dmmm"),
    ]),
    Chart('object-updates', "Object updates", [
        Signal("Groups", "unistat-groups.updated_objects_count_dxxm"),
        Signal("Stages", "unistat-stages.updated_objects_count_dxxm"),
        Signal("Nanny services", "unistat-nanny_services.updated_objects_count_dxxm"),
        Signal("Projects", "unistat-projects.updated_objects_count_dxxm")
    ]),
    Chart('object-full-reloads', "Object full reloads (selectObjects requests)", [
        Signal("Groups", "unistat-groups.full_reload_count_dmmm"),
        Signal("Stages", "unistat-stages.full_reload_count_dmmm"),
        Signal("Nanny services", "unistat-nanny_services.full_reload_count_dmmm"),
        Signal("Projects", "unistat-projects.full_reload_count_dmmm"),
        Signal("YP watches errors (Groups)", "unistat-groups.watch_objects_errors_dmmm"),
        Signal("YP watches errors (Stages)", "unistat-stages.watch_objects_errors_dmmm"),
        Signal("YP watches errors (Nanny services)", "unistat-nanny_services.watch_objects_errors_dmmm"),
        Signal("YP watches errors (Projects)", "unistat-projects.watch_objects_errors_dmmm"),
    ]),
    Chart('object-watch-time', "watchObjects (milliseconds)", [
        Signal("Groups", "unistat-groups.watch_objects_time_ms_axxx"),
        Signal("Stages", "unistat-stages.watch_objects_time_ms_axxx"),
        Signal("Nanny services", "unistat-nanny_services.watch_objects_time_ms_axxx"),
        Signal("Projects", "unistat-projects.watch_objects_time_ms_axxx")
    ]),
    Chart('object-getobjects-time', "getObjects (milliseconds)", [
        Signal("Groups", "unistat-groups.get_objects_time_ms_axxx"),
        Signal("Stages", "unistat-stages.get_objects_time_ms_axxx"),
        Signal("Nanny services", "unistat-nanny_services.get_objects_time_ms_axxx"),
        Signal("Projects", "unistat-projects.get_objects_time_ms_axxx")
    ]),
    Chart('objects-select', "selectObjects (milliseconds)", [
        Signal("Staff groups load", "unistat-staff_groups_load_time_ms_axxx"),
        Signal("Nanny: YP groups load from all clusters", "unistat-nanny.yp_objects_load_time_ms_axxx"),
        Signal("Initial IDM nodes load (seconds)", "trunc(div(unistat-nodes.startup_time_ms,1000))"),
    ]),
] + common_charts

RADAR_STAGES_XDC = [
    "deploy-speed-sandboxbased-stage1",
    "deploy-speed-docker-zerodiff1",
    "deploy-speed-sequential",
    "deploy-speed-docker-sequential",
    "deploy-speed-sandbox-zerodiff-acceptance"
]

RADAR_STAGES_XDC_MC = [
    "deploy-speed-multi-cluster-replica-set"
]

RADAR_STAGES_SAS_TEST = [
    "sequential1",
    "sequential2",
    "sandboxbased-stage1",
]

RADAR_STAGES_SAS_TEST_MC = [
    "multi-cluster-replica-set"
]

def get_radar_chars(stage_cluster, ctype, stages, is_mcrs):
    result = []
    for stage in stages:
        signals = []
        signals.append(Signal("Stage ready", f"unistat-{stage_cluster}.{stage}.stage_ready_seconds_axxx"))
        if is_mcrs:
            signals.append(Signal(f"mcrs ready", f"unistat-{stage_cluster}.{stage}.rs_ready_time_seconds_axxx"))
        for cluster in managed_clusters[ctype]:
            if not is_mcrs:
                signals.append(Signal(f"{cluster}: rs ready", f"unistat-{cluster}.{stage}.rs_ready_time_seconds_axxx"))
            signals.append(Signal(f"{cluster}: workload started", f"unistat-{cluster}.{stage}.deploy_time_seconds_axxx"))
            if not is_mcrs:
                signals.append(Signal(f"{cluster}: rs updated", f"unistat-{cluster}.{stage}.rs_update_time_seconds_axxx"))
        if is_mcrs:
            signals.append(Signal(f"mcrs updated", f"unistat-{stage_cluster}.{stage}.rs_update_time_seconds_axxx"))

        result.append(Chart(f'{stage_cluster}_{stage}', f"{stage_cluster} {stage} (seconds)", signals))
    return result

radar_charts = get_radar_chars("xdc", Ctype.PROD, RADAR_STAGES_XDC, False) + \
               get_radar_chars("xdc", Ctype.PROD, RADAR_STAGES_XDC_MC, True) + \
               get_radar_chars("sas-test", Ctype.TESTING, RADAR_STAGES_SAS_TEST, False) + \
               get_radar_chars("sas-test", Ctype.TESTING, RADAR_STAGES_SAS_TEST_MC, True)

def generate_panel(itype, title, charts):
    result = [
        '<% set ctype = ctype|default("{}") %>'.format(Ctype.PROD),
        '<% set itype = itype|default("{}") %>'.format(itype)
    ]

    max_rows = 4

    for ctype in Ctype.ALL:
        result.append('''
<% if ctype == "{}" %>
    <% set clusters = {} %>
<% endif %>
'''.format(ctype, managed_clusters[ctype]))

    result.append('''
<% set tag = 'itype=' ~ (itype if itype is defined)
    ~ ( ';ctype=' ~ ctype if ctype is defined ) %>

<% set table = dict(col=1, row=1) %>
<% set total_rows = {} %>
<% macro inc_col(rows=1) -%>
    <% if table['col'] >= total_rows %>
        <% if table.update({{'col': table['row'] + rows}}) %><% endif %>
    <% endif %>
    <% if table.update({{'row': table['col'] % total_rows + rows}}) %><% endif %>
<%- endmacro %>
<% macro inc_row(rows=1) -%>
    <% if table.update({{'row': table['row'] + rows, 'col': 1}}) %><% endif %>
<%- endmacro %>
'''.format(max_rows))

    result.append(f'''
{{
    "abc": "drug",
    "user": "amich",
    "title": "{title} (<< ctype >>)",
    "type": "panel",
    "editors": [
        "reddi",
        "alonger",
        "dmitriyt",
        "amich",
        "kkembo",
        "panefgen",
        "vonzeppelin",
        "staroverovad"
    ],
    "charts": [
''')

    total_width = 0
    for chart in charts:
        total_width += chart.width
        chart.print_yasm(2, result)

    result.append('''
    ]
}
''')

    return "\n".join(result)


def generate_alerts_data():
    result = []
    result.append('''
[
    <% for ctype in ['prod', 'prestable'] %>''')

    for ctype in Ctype.ALL:
        result.append('''
        <% if ctype == "{}" %>
            <% set clusters = {} %>
        <% endif %>'''.format(ctype, managed_clusters[ctype]))

    result.append('        <% set info = [] %>')

    for chart in stagectl_charts:
        for signal in chart.signals:
            signal.print_alert_info(2, result)

    # this sorry single trend alert does not fit anywhere
    result.append('''
        <% if ctype == "prod" %>
            <% do info.append({
                "name": "valid_stage_count_decrease",
                "signal": "unistat-stages.valid_count_axxx",
                "trend": "down",
                "interval": 60,
                "warn_perc": 20,
                "crit_perc": 40,
                "value_modify": {
                    "type": "max",
                    "window": 60
                }
            }) %>
        <% endif %>

        <% if ctype == "prod" %>
            <% set updating_rs = "sum(unistat-replica_sets.iva.running_operations_count_axxx,unistat-replica_sets.man.running_operations_count_axxx,unistat-replica_sets.myt.running_operations_count_axxx,unistat-replica_sets.sas.running_operations_count_axxx,unistat-replica_sets.vla.running_operations_count_axxx)" %>
        <% elif ctype == "test" %>
            <% set updating_rs = "sum(unistat-replica_sets.sas-test.running_operations_count_axxx, unistat-replica_sets.man-pre.running_operations_count_axxx)" %>
        <% elif ctype == "prestable" %>
            <% set updating_rs = "unistat-replica_sets.man-pre.running_operations_count_axxx" %>
        <% else %>
            <% set updating_rs = "unistat-replica_sets.man-pre.running_operations_count_axxx" %>
        <% endif %>

        <% do info.append({
            "name": "rs_updating",
            "signal": updating_rs,
            "warn": [25, 80],
            "crit": [80, None]
        }) %>

        <% for s in info %>
            {
                "name": "<<ctype>>.<< s.name >>",
                "abc": "drug",
                "signal": "<< s.signal >>",
                "mgroups": ["ASEARCH"],
                "tags": {
                    "ctype": "<< ctype >>",
                    "itype": "stagectl"
                },
                "juggler_check": {
                    "namespace": "deploy",
                    "host": "yd-stagectl-<<ctype>>",
                    "service": "<< s.name >>",
                    "aggregator": "logic_or",
                    "aggregator_kwargs": {
                        "unreach_service": [
                            {
                                "check": "yasm_alert:virtual-meta"
                            }
                        ],
                        "nodata_mode": "force_crit",
                        "unreach_mode": "force_ok"
                    },
                    "tags": ["send_notification"],
                    "active_kwargs": None,
                    "notifications": [
                        {
                            "template_name": "on_status_change",
                            "template_kwargs": {
                                "status": ["WARN"],
                                "login": ["amich"],
                                "method": ["telegram"]
                            }
                        },
                        {
                            "template_name": "on_status_change",
                            "template_kwargs": {
                                "status": ["CRIT"],
                                "login": ["yd-monitorings"],
                                "method": ["telegram"]
                            }
                        },
                        {
                            "template_name": "on_status_change",
                            "template_kwargs": {
                                "status": ["CRIT"],
                                "login": ["amich"],
                                "method": ["email"]
                            }
                        }
                        <% if ctype == "prod" %>
                        ,{
                            "template_name": "on_status_change",
                            "template_kwargs": {
                                "status": ["CRIT"],
                                "login": ["@svc_QLOUD:qloud-support-2nd-line"],
                                "method": ["phone"]
                            }
                        }
                        <% endif %>
                    ],
                    <% if "flaps" in s %>
                    "flaps": {
                        "critical": << s.flaps.critical | default(0)>>,
                        "boost": << s.flaps.boost | default(0)>>,
                        "stable": << s.flaps.stable | default(0)>>
                    },
                    <% elif "trend" not in s %>
                    "flaps": {
                        "critical": 1800,
                        "boost": 0,
                        "stable": 360
                    },
                    <% endif %>
                    <% if "tags" in s %>
                    "tags": [
                        <% for tag in s.tags | sort %>
                          "<< tag >>",
                        <% endfor %>
                    ]
                    <% endif %>
                },
                <% if "trend" in s %>
                    "trend": << s.trend >>,
                    "interval": << s.interval >>,
                    "warn_perc": << s.warn_perc >>,
                    "crit_perc": << s.crit_perc >>,
                    "value_modify": << s.value_modify >>
                <% else %>
                    "warn": << s.warn >>,
                    "crit": << s.crit >>,
                <% endif %>
            },
        <% endfor %>
    <% endfor %>
]
''')

    return "\n".join(result)


class Modes:
    DRY_RUN = "dry_run"
    UPDATE = "update"
    DIFF = "diff"
    ALL = [DRY_RUN, UPDATE, DIFF]


class Panels:
    GRAPHS = "graphs"
    AUTH = "auth"
    ALERTS = "alerts"
    RESOURCES = "resources"
    RADAR = "radar"
    ALL = [GRAPHS, ALERTS, RESOURCES, AUTH, RADAR]


def verify_return_code(response, action):
    if response.status_code != 200:
        print("Could not {}, returned code {}: {}\n{}".format(
            action, response.status_code, response.reason, response.text))
        sys.exit(1)

def update_graphs_panel(data):
    verify_return_code(requests.post(GRAPHS_PANEL_UPDATE_URL, data=data), "update graphs panel")

def update_resources_panel(data):
    verify_return_code(requests.post(RESOURCES_PANEL_UPDATE_URL, data=data), "update resources panel")

def update_auth_panel(data):
    verify_return_code(requests.post(AUTH_PANEL_UPDATE_URL, data=data), "update auth panel")

def update_radar_panel(data):
    verify_return_code(requests.post(RADAR_PANEL_UPDATE_URL, data=data), "update deploy speed panel")

def update_alerts_panel(data):
    verify_return_code(requests.post(ALERTS_PANEL_UPDATE_URL, data=data), "update alerts panel")
    verify_return_code(requests.post(ALERTS_PANEL_APPLY_URL), "apply new alerts")

if __name__ == "__main__":
    parser = argparse.ArgumentParser(description='Manage stagectl monitoring')
    parser.add_argument("panel", choices=Panels.ALL, help="Select panel")
    parser.add_argument("action", choices=Modes.ALL, help="Action with panel")

    args = parser.parse_args()
    if args.panel == Panels.GRAPHS:
        generated_data = generate_panel("stagectl", "Y.Deploy stage controller", stagectl_charts)
        get_url = GRAPHS_PANEL_GET_URL
        update_action = update_graphs_panel
    elif args.panel == Panels.ALERTS:
        generated_data = generate_alerts_data()
        get_url = ALERTS_PANEL_GET_URL
        update_action = update_alerts_panel
    elif args.panel == Panels.RESOURCES:
        generated_data = generate_panel("stagectl", "Y.Deploy stage controller resources", resource_charts)
        get_url = RESOURCES_PANEL_GET_URL
        update_action = update_resources_panel
    elif args.panel == Panels.AUTH:
        generated_data = generate_panel("authctl", "Y.Deploy auth controller", auth_charts)
        get_url = AUTH_PANEL_GET_URL
        update_action = update_auth_panel
    elif args.panel == Panels.RADAR:
        generated_data = generate_panel("deploy-speed-radar", "Deploy speed metrics", radar_charts)
        get_url = RADAR_PANEL_GET_URL
        update_action = update_radar_panel
    else:
        print("Unknown panel", args.panel)
        sys.exit(1)

    if args.action == Modes.DRY_RUN:
        print(generated_data)
    elif args.action == Modes.UPDATE:
        update_action(generated_data)
    else:
        r = requests.get(get_url)
        verify_return_code(r, "get current panel")
        print("\n".join(difflib.unified_diff(r.text.split("\n"), generated_data.split("\n"))))
