import argparse
import os
import hashlib
import json
import time
import socket
import requests
import logging
import sys
from requests.adapters import HTTPAdapter
from urllib3 import Retry

RETRY_LIMIT = 5
checker_file = ''


def get_initial_data(checked_file):
    with open(checker_file, 'rb') as f:
        dump = json.load(f)
        v = dump.get('resources', {}).get(checked_file, {}).get('verification', {}).get('checksum')

        if v and v.startswith('MD5:'):
            return v[4:]
        else:
            return get_md5(checked_file)


def get_md5(file_name, block_size=8192):
    md5 = hashlib.md5()
    with open(file_name, 'rb') as f:
        block = f.read(block_size)
        while block:
            md5.update(block)
            block = f.read(block_size)
    return md5.hexdigest()


class CheckedFile:
    def __init__(self, file_name):
        self.file_name = file_name
        self.initial_sum = get_initial_data(file_name)
        self.initial_time = -1


class MD5Checker:

    def __init__(self, files, service):
        with open(files, 'rb') as f:
            self.file_names = json.load(f).get('files', {})
        self.service_name = service
        self.modified_files = []
        self.files = [CheckedFile(file_name) for file_name in self.file_names]

    def check_sum(self):
        self.modified_files = []
        for f in self.files:
            current_time = os.path.getmtime(f.file_name)
            if current_time != f.initial_time:
                current_sum = get_md5(f.file_name)
                if current_sum != f.initial_sum:
                    self.modified_files.append(f.file_name)
                else:
                    f.initial_time = current_time
        return len(self.modified_files) == 0

    def check(self):
        try:
            if not self.check_sum():
                return 'CRIT', '{}: files were modified'.format(', '.join(self.modified_files))
            return 'OK', ''
        except (IOError, OSError) as err:
            return 'CRIT', str(err)


def push_to_juggler(url, host, service, status, desc, instance):
    events = {
        "events": [
            {
                "host": host,
                "service": service,
                "instance": instance,
                "status": status,
                "description": desc
            }
        ]
    }
    try:
        with requests.Session() as s:
            retries = Retry(
                total=RETRY_LIMIT,
                read=RETRY_LIMIT,
                connect=RETRY_LIMIT,
                backoff_factor=0.5,
                status_forcelist=[500, 502, 503, 504],
                method_whitelist=frozenset(['POST'])
            )
            s.mount("https://", HTTPAdapter(max_retries=retries))
            s.mount("http://", HTTPAdapter(max_retries=retries))
            return s.post(url, json=events, timeout=5)
    except Exception:
        pass


def main():
    logging.basicConfig(stream=sys.stdout,
                        format='[%(asctime)s] %(levelname)s: %(message)s',
                        level=logging.INFO)

    parser = argparse.ArgumentParser()
    parser.add_argument('-f', '--checked-files', required=True)
    parser.add_argument('-d', '--delay', type=int, default=300)
    parser.add_argument('-i', '--juggler-instance', default='')
    parser.add_argument('-s', '--juggler-service', default='deploy_checksum')
    parser.add_argument('-c', '--checksum-file', default='dump.json')
    parser.add_argument('--host', default=socket.gethostname())
    parser.add_argument('-u', '--juggler-url', default='http://[::1]:31579/events')
    args = parser.parse_args()

    checked_files = os.path.basename(args.checked_files)
    global checker_file
    checker_file = os.path.basename(args.checksum_file)
    checker = MD5Checker(checked_files, args.juggler_service)

    while True:
        logging.info("%s %s %s", args.juggler_service, args.host, args.juggler_instance)
        status, desc = checker.check()
        logging.info("Check result: %s %s", status, desc)
        push_to_juggler(url=args.juggler_url,
                        host=args.host,
                        status=status,
                        instance=args.juggler_instance,
                        desc=desc,
                        service=args.juggler_service)
        logging.info("Pushed to Juggler state %s", status)
        time.sleep(args.delay)


if __name__ == '__main__':
    main()
