#!/usr/bin/env python

import json
import os
import re
import socket
from BaseHTTPServer import BaseHTTPRequestHandler, HTTPServer

# Example: "2021-10-12 17:27:55.680860 [CompletableFutureDelayScheduler] ERROR r.y.i.s.Engine - Engine..."
error_line_pattern = re.compile(r'^\d{4}-\d{2}-\d{2} \d{2}:\d{2}:\d{2}\.\d{6} \[([^\[\]]+)\] ERROR .*$')

# if logger cannot log an error, it is printed into standard err stream and marked with these lines
lines_preceding_the_error = [re.compile(r'Could not log exception$'),
                             re.compile(r'^because of another exception during logging$')]


class LogParser(object):
    def __init__(self, logs_file):
        self.logs_file = logs_file
        self.is_processing = False
        self.errors = []
        self.is_next_line_error = False
        self.last_pointer = 0
        self.last_segment = None

    def get_errors_count(self):
        self.parse()
        return len(self.errors)

    def get_errors(self):
        self.parse()
        return self.errors

    def parse(self):
        if self.is_processing:
            return
        self.is_processing = True
        try:
            with open(self.logs_file) as fp:
                for line in self.read_new(fp):
                    line = line.strip()

                    if self.is_next_line_error:
                        self.errors.append(line)
                        self.is_next_line_error = False
                        continue

                    mather_pause = error_line_pattern.match(line)
                    if mather_pause:
                        self.errors.append(mather_pause.group(0))
                        continue

                    for line_before_error in lines_preceding_the_error:
                        mather_pause = line_before_error.match(line)
                        if mather_pause:
                            self.is_next_line_error = True
                            break
                    if self.is_next_line_error:
                        continue
        finally:
            self.is_processing = False

    def read_new(self, fp, buf_size=8192):
        fp.seek(0, os.SEEK_END)
        file_size = fp.tell()

        if self.last_pointer > file_size:
            self.last_pointer = 0
            self.last_segment = None
            self.is_next_line_error = False

        fp.seek(self.last_pointer)
        while self.last_pointer < file_size:
            length_to_read = min(file_size - self.last_pointer, buf_size)
            buffer = fp.read(length_to_read)

            lines = buffer.splitlines(True)

            if self.last_segment is not None:
                lines[0] = self.last_segment + lines[0]
            for line in lines:
                if len(line):
                    if line[-1] == '\n':
                        yield line
                        self.last_segment = None
                    else:
                        self.last_segment = line
            self.last_pointer += length_to_read


class YasmHandler(BaseHTTPRequestHandler):
    parser = None

    def _set_headers(self):
        self.send_response(200)
        self.send_header('Content-type', 'application/json')
        self.end_headers()

    def do_GET(self):
        if self.path == "/unistat":
            if self.parser.is_processing:
                self.send_response(503)
                self.end_headers()
            else:
                self.get_unistat()
        elif self.path == "/errors":
            self.get_errors()
        else:
            self.send_response(404)
            self.end_headers()

    def get_unistat(self):
        self._set_headers()
        result = [["all_log_errors_count_dmmm", self.parser.get_errors_count()]]
        self.wfile.write(json.dumps(result))

    def get_errors(self):
        self._set_headers()
        result = self.parser.get_errors()
        self.wfile.write(json.dumps(result))

    def do_HEAD(self):
        self._set_headers()


class HTTPServerV6(HTTPServer):
    address_family = socket.AF_INET6


def run(server_class=HTTPServerV6, handler_class=YasmHandler, port=80):
    server_address = ('', port)
    httpd = server_class(server_address, handler_class)
    httpd.serve_forever()


if __name__ == "__main__":
    from sys import argv

    YasmHandler.parser = LogParser(argv[1])
    run(port=int(argv[2]))
