package ru.yandex.infra.stage;

import java.time.Clock;
import java.time.Duration;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.function.BiFunction;
import java.util.function.Function;
import java.util.function.Supplier;
import java.util.stream.Collectors;

import javax.servlet.http.HttpServlet;

import com.codahale.metrics.MetricRegistry;
import com.google.common.collect.ImmutableMap;
import com.google.protobuf.Message;
import com.typesafe.config.Config;
import com.typesafe.config.ConfigFactory;
import com.typesafe.config.ConfigList;
import com.typesafe.config.ConfigMergeable;
import com.typesafe.config.ConfigValueFactory;
import io.jaegertracing.Configuration;
import io.jaegertracing.internal.JaegerTracer;
import io.jaegertracing.internal.samplers.ConstSampler;
import io.opentracing.Tracer;
import io.opentracing.util.GlobalTracer;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.asynchttpclient.AsyncHttpClient;
import org.asynchttpclient.DefaultAsyncHttpClient;
import org.asynchttpclient.DefaultAsyncHttpClientConfig;
import org.eclipse.jetty.server.Handler;
import org.eclipse.jetty.server.Server;
import org.eclipse.jetty.server.ServerConnector;
import org.eclipse.jetty.server.handler.DefaultHandler;
import org.eclipse.jetty.server.handler.HandlerCollection;
import org.eclipse.jetty.servlet.ServletContextHandler;
import org.eclipse.jetty.servlet.ServletHolder;
import org.eclipse.jetty.util.thread.QueuedThreadPool;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.infra.controller.YtSettings;
import ru.yandex.infra.controller.dto.SchemaMeta;
import ru.yandex.infra.controller.dto.StageMeta;
import ru.yandex.infra.controller.metrics.GaugeRegistry;
import ru.yandex.infra.controller.metrics.NamespacedGaugeRegistry;
import ru.yandex.infra.controller.servlets.UnistatServlet;
import ru.yandex.infra.controller.yp.EpochDecoratorRepository;
import ru.yandex.infra.controller.yp.LabelBasedRepository;
import ru.yandex.infra.controller.yp.ObjectBuilderDescriptor;
import ru.yandex.infra.controller.yp.YpObjectTransactionalRepository;
import ru.yandex.infra.stage.cache.Cache;
import ru.yandex.infra.stage.cache.CacheSet;
import ru.yandex.infra.stage.cache.CacheStorage;
import ru.yandex.infra.stage.cache.CacheStorageFactory;
import ru.yandex.infra.stage.cache.CachedObjectType;
import ru.yandex.infra.stage.cache.EmptyCacheStorage;
import ru.yandex.infra.stage.cache.LocalFsCacheStorage;
import ru.yandex.infra.stage.cache.ReadonlyCacheStorage;
import ru.yandex.infra.stage.cache.StorageType;
import ru.yandex.infra.stage.cache.YtCacheStorage;
import ru.yandex.infra.stage.concurrent.SerialExecutor;
import ru.yandex.infra.stage.deployunit.SandboxResourcesGetterImpl;
import ru.yandex.infra.stage.deployunit.SandboxResourcesResolver;
import ru.yandex.infra.stage.deployunit.SandboxResourcesResolverImpl;
import ru.yandex.infra.stage.docker.DockerHttpGetterImpl;
import ru.yandex.infra.stage.docker.DockerImagesResolver;
import ru.yandex.infra.stage.docker.DockerImagesResolverImpl;
import ru.yandex.infra.stage.dto.DownloadableResource;
import ru.yandex.infra.stage.inject.GCLimit;
import ru.yandex.infra.stage.inject.GCSettings;
import ru.yandex.infra.stage.podspecs.ResourceSupplier;
import ru.yandex.infra.stage.podspecs.ResourceSupplierFactory;
import ru.yandex.infra.stage.podspecs.ResourceWithMeta;
import ru.yandex.infra.stage.podspecs.SandboxReleaseGetter;
import ru.yandex.infra.stage.podspecs.SandboxReleaseGetterImpl;
import ru.yandex.infra.stage.podspecs.patcher.EndpointSetSpecPatchersHolderFactory;
import ru.yandex.infra.stage.podspecs.patcher.PatcherContexts;
import ru.yandex.infra.stage.podspecs.patcher.PatcherContextsFactory;
import ru.yandex.infra.stage.podspecs.patcher.PatcherParameters;
import ru.yandex.infra.stage.podspecs.patcher.PodSpecPatchersHolderFactory;
import ru.yandex.infra.stage.podspecs.revision.PatcherType;
import ru.yandex.infra.stage.podspecs.revision.PatchersRevisionsHolderFactory;
import ru.yandex.infra.stage.podspecs.revision.RevisionsHolder;
import ru.yandex.infra.stage.podspecs.revision.model.RevisionScheme;
import ru.yandex.infra.stage.rest.CacheServlet;
import ru.yandex.infra.stage.rest.GarbageForceCollectHandler;
import ru.yandex.infra.stage.rest.GarbageListHandler;
import ru.yandex.infra.stage.rest.GarbageRequestReceiver;
import ru.yandex.infra.stage.rest.RelationsServlet;
import ru.yandex.infra.stage.rest.ResourcesServlet;
import ru.yandex.infra.stage.util.AdaptiveRateLimiter;
import ru.yandex.infra.stage.util.AdaptiveRateLimiterImpl;
import ru.yandex.infra.stage.util.JsonHttpGetter;
import ru.yandex.infra.stage.yp.AsyncYpClientsMap;
import ru.yandex.yp.YpRawObjectService;
import ru.yandex.yp.client.api.Autogen.TSchemaMeta;
import ru.yandex.yp.client.api.Autogen.TStageMeta;
import ru.yandex.yp.client.api.DataModel;
import ru.yandex.yp.client.api.TPodTemplateSpec;
import ru.yandex.yp.client.api.TProjectSpec;
import ru.yandex.yp.client.api.TProjectStatus;
import ru.yandex.yp.client.api.TStageSpec;
import ru.yandex.yp.client.api.TStageStatus;
import ru.yandex.yp.model.YpObjectType;
import ru.yandex.yt.ytclient.proxy.YtClient;
import ru.yandex.yt.ytclient.rpc.RpcCredentials;

import static ru.yandex.infra.controller.util.ConfigUtils.token;
import static ru.yandex.infra.controller.util.YpUtils.ypRawClient;

public final class ConfigUtils {
    public static final String REST_SERVER_CONFIG_KEY = "rest_server";
    public static final String LOGBROKER_PATCHER_CONFIG_PATH = "logbroker";

    public static final BiFunction<Config, String, String> STRING_FROM_CONFIG = Config::getString;
    public static final BiFunction<Config, String, Integer> INTEGER_FROM_CONFIG = Config::getInt;
    public static final BiFunction<Config, String, Config> CONFIG_FROM_CONFIG = Config::getConfig;

    private static final Logger LOG = LoggerFactory.getLogger(ConfigUtils.class);

    private static final ObjectBuilderDescriptor<TStageMeta, StageMeta> STAGE_DESCRIPTOR = new ObjectBuilderDescriptor<>(
            TStageSpec::newBuilder, TStageStatus::newBuilder, StageMeta::fromProto, TStageMeta.getDefaultInstance());
    private static final ObjectBuilderDescriptor<TSchemaMeta, SchemaMeta> PROJECT_DESCRIPTOR = new ObjectBuilderDescriptor<>(
            TProjectSpec::newBuilder, TProjectStatus::newBuilder, SchemaMeta::fromProto, TSchemaMeta.getDefaultInstance());

    private ConfigUtils() {
    }

    public static AsyncYpClientsMap ypClientMap(Config config, GaugeRegistry gaugeRegistry, boolean isReadonlyMode) {
        YpRawObjectService multiObjectService = ypRawClient(config.getConfig("multi_cluster"), config, gaugeRegistry, isReadonlyMode);

        Map<String, Config> clusterConfigs = extractMap(config.getConfig("clusters"), CONFIG_FROM_CONFIG);
        Map<String, YpRawObjectService> clusterClients = EntryStream.of(clusterConfigs)
                .mapValues(clusterConfig -> ypRawClient(clusterConfig, config, gaugeRegistry, isReadonlyMode))
                .toMap();

        return new AsyncYpClientsMap(clusterClients, multiObjectService);
    }

    public static GCLimit getGCLimit(Config rootConfig, String path) {
        var config = rootConfig.getConfig(path);
        return new GCLimit(config.getInt("initial"), config.getInt("regular"));
    }

    public static GCSettings loadGCSettings(Config config) {
        var defaultLimit = ConfigUtils.getGCLimit(config, "default_limit");
        var customLimits = EntryStream.of(extractMap(config.getConfig("custom_limits"), ConfigUtils::getGCLimit))
                .mapKeys(YpObjectType::valueOf)
                .toMap();

        return new GCSettings(defaultLimit, customLimits);
    }

    public static <T> Map<String, T> extractMap(Config config,
                                                BiFunction<Config, String, T> extractor) {
        return extractMap(config, extractor, Function.identity());
    }

    public static <ConfigType, ValueType> Map<String, ValueType> extractMap(Config config,
                                                                            BiFunction<Config, String, ConfigType> extractor,
                                                                            Function<ConfigType, ValueType> parser) {
        return StreamEx.of(config.root().keySet())
                .mapToEntry(key -> extractor.apply(config, key))
                .mapValues(parser)
                .toMap();
    }

    public static Map<String, String> labels(Config config) {
        return extractMap(config, STRING_FROM_CONFIG);
    }

    public static Map<String, Integer> blackboxEnvironments(Config config) {
        return extractMap(config, INTEGER_FROM_CONFIG);
    }

    private static YtClient createYtClient(YtSettings ytSettings) {
        return YtClient.builder()
                .setCluster(ytSettings.getProxy())
                .setRpcCredentials(new RpcCredentials(ytSettings.getUser(), ytSettings.getToken()))
                .setRpcOptions(ytSettings.getRpcOptions())
                .build();
    }

    static CacheStorageFactory getCacheStorageFactory(Config config,
                                                      YtSettings ytSettings,
                                                      GaugeRegistry gaugeRegistry,
                                                      boolean isReadOnlyMode) {
        CacheStorageFactory factory = createCacheStorageFactory(config, ytSettings, gaugeRegistry, isReadOnlyMode);
        if (isReadOnlyMode) {
            return new CacheStorageFactory() {
                @Override
                public <TValue, TProtoValue extends Message> CacheStorage<TProtoValue> createStorage(
                        CachedObjectType<TValue, TProtoValue> cachedObjectType) {
                    final CacheStorage<TProtoValue> rwStorage = factory.createStorage(cachedObjectType);
                    return new ReadonlyCacheStorage<>(rwStorage);
                }
            };
        }
        return factory;
    }

    private static CacheStorageFactory createCacheStorageFactory(Config config,
                                                                 YtSettings ytSettings,
                                                                 GaugeRegistry gaugeRegistry,
                                                                 boolean isReadOnlyMode) {

        StorageType storageType = StorageType.get(config.getString("storage_type"));
        LOG.info("Initializing cache storage: type = {}", storageType);

        switch (storageType) {
            case EMPTY:
                return new CacheStorageFactory() {
                    @Override
                    public <TValue, TProtoValue extends Message> CacheStorage<TProtoValue> createStorage(
                            CachedObjectType<TValue, TProtoValue> cachedObjectType) {
                        return new EmptyCacheStorage<>();
                    }
                };
            case YT_TABLES:
                YtClient ytClient = createYtClient(ytSettings);
                Optional<Duration> readTimeout = config.hasPath("read_request_timeout") ?
                        Optional.of(config.getDuration("read_request_timeout")) : Optional.empty();

                return new CacheStorageFactory() {
                    @Override
                    public <TValue, TProtoValue extends Message> CacheStorage<TProtoValue> createStorage(
                            CachedObjectType<TValue, TProtoValue> cachedObjectType) {
                        return new YtCacheStorage<>(ytClient,
                                config.getString("path"),
                                cachedObjectType,
                                config.getInt("bulk_write_batch_size"),
                                readTimeout,
                                gaugeRegistry);
                    }
                };
            case LOCAL_FILE_SYSTEM:
                return new CacheStorageFactory() {
                    @Override
                    public <TValue, TProtoValue extends Message> CacheStorage<TProtoValue> createStorage(
                            CachedObjectType<TValue, TProtoValue> cachedObjectType) {
                        return new LocalFsCacheStorage<>(config.getString("path"),
                                cachedObjectType,
                                isReadOnlyMode ? Duration.ZERO : config.getDuration("flush_interval"));
                    }
                };
            default:
                throw new RuntimeException("Unknown storage_type 'cache.$(cache.storage).storage_type': " + storageType);
        }
    }

    public static YpObjectTransactionalRepository<StageMeta, TStageSpec, TStageStatus> ypStageRepository(
            YpRawObjectService objectService,
            Config config, int pageSize,
            Supplier<Long> epochGetter,
            String epochKey,
            Optional<String> vcsKey,
            GaugeRegistry gaugeRegistry,
            boolean isReadonlyMode) {
        Map<String, String> labels = ConfigUtils.labels(config.getConfig("stage_repository.labels"));
        LabelBasedRepository<StageMeta, TStageSpec, TStageStatus> basedRepository =
                new LabelBasedRepository<>(YpObjectType.STAGE, labels, vcsKey, objectService, STAGE_DESCRIPTOR, pageSize, gaugeRegistry);
        if (!config.getBoolean("leader_lock.use_lock") || isReadonlyMode) {
            return basedRepository;
        }
        return new EpochDecoratorRepository<>(basedRepository, YpObjectType.STAGE, epochGetter, epochKey);
    }

    public static LabelBasedRepository<SchemaMeta, TProjectSpec, TProjectStatus> ypProjectRepository(
            YpRawObjectService objectService,
            Config config, int pageSize,
            Optional<String> vcsKey, GaugeRegistry gaugeRegistry) {
        Map<String, String> labels = ConfigUtils.labels(config.getConfig("labels"));
        return new LabelBasedRepository<>(YpObjectType.PROJECT, labels, vcsKey, objectService, PROJECT_DESCRIPTOR, pageSize, gaugeRegistry);
    }

    public static AdaptiveRateLimiter adaptiveRateLimiter(Clock clock, Config rateLimiterConfig) {
        return rateLimiterConfig.getBoolean("enabled")
                ? new AdaptiveRateLimiterImpl(clock, rateLimiterConfig) :
                AdaptiveRateLimiter.EMPTY;
    }

    public static DockerImagesResolver dockerResolver(Config config,
                                                      SerialExecutor serialExecutor,
                                                      GaugeRegistry metricsRegistry,
                                                      HttpServiceMetrics serviceMetrics,
                                                      CacheSet caches,
                                                      AdaptiveRateLimiter rateLimiter) {
        var jsonHttpGetter = new JsonHttpGetter(HttpServiceMetrics.Source.DOCKER,
                buildAsyncHttpClient(config.getConfig("http_client").getDuration("request_timeout")),
                serviceMetrics);
        return new DockerImagesResolverImpl(
                serialExecutor,
                new DockerHttpGetterImpl(
                        config.getString("base_url"),
                        getOptional(config, "default_registry_host", STRING_FROM_CONFIG).orElse(null),
                        jsonHttpGetter),
                config.getDuration("initial_retry_timeout"),
                config.getDuration("max_retry_timeout"),
                caches.get(CachedObjectType.DOCKER_IMAGE_CONTENTS),
                config.getBoolean("allow_force_resolve"),
                new NamespacedGaugeRegistry(metricsRegistry, "docker"),
                rateLimiter);
    }

    public static SandboxResourcesResolver sandboxResourcesResolver(Config sandboxConfig,
                                                                    SerialExecutor executor,
                                                                    HttpServiceMetrics serviceMetrics,
                                                                    AdaptiveRateLimiter rateLimiter,
                                                                    Cache<DownloadableResource> cache) {
        var jsonHttpGetter = new JsonHttpGetter(HttpServiceMetrics.Source.SANDBOX,
                buildAsyncHttpClient(sandboxConfig.getDuration("resolver_request_timeout")),
                serviceMetrics);

        String sandboxToken = token(sandboxConfig.getString("token_file"));
        SandboxResourcesGetterImpl sandboxResourcesGetter = new SandboxResourcesGetterImpl(
                sandboxToken,
                sandboxConfig.getString("base_url") + "/api/v1.0/resource/%s",
                cache,
                sandboxConfig.getBoolean("enable_resource_cache"),
                jsonHttpGetter
        );
        return new SandboxResourcesResolverImpl(
                executor,
                sandboxResourcesGetter,
                cache,
                sandboxConfig.getDuration("initial_retry_timeout"),
                sandboxConfig.getDuration("max_retry_timeout"),
                rateLimiter
        );
    }

    public static AclFilter aclFilter(Config config) {
        final String type = config.getString("type");
        switch (type) {
            case "prefix":
                return new AclPrefixFilter(config.getString("prefix"));
            default:
                throw new IllegalArgumentException("Unsupported acl filter type: " + type);
        }
    }

    public static RevisionsHolder<DataModel.TEndpointSetSpec.Builder> endpointSetSpecPatchersRevisionsHolder(Config patcherConfig, String revisionSchemeFileName) {

        var endpointSetSpecPatchersHolder = EndpointSetSpecPatchersHolderFactory.fromContexts();

        var revisionScheme = RevisionScheme.loadFromResource(revisionSchemeFileName);

        var revisionSchemeConfig = patcherConfig.getConfig("revision_scheme");

        return new PatchersRevisionsHolderFactory<>(endpointSetSpecPatchersHolder).from(revisionScheme, revisionSchemeConfig, PatcherType.ENDPOINT_SET_SPEC);
    }

    public static RevisionsHolder<TPodTemplateSpec.Builder> podSpecPatchersRevisionsHolder(Config patcherConfig,
                                                                                           ResourceSupplierFactory factory,
                                                                                           PatcherParameters patcherParameters,
                                                                                           String revisionSchemeFileName) {

        PatcherContexts contexts = PatcherContextsFactory.parseConfig(patcherConfig, factory, patcherParameters);

        var podSpecPatchersHolder = PodSpecPatchersHolderFactory.fromContexts(contexts);

        var revisionScheme = RevisionScheme.loadFromResource(revisionSchemeFileName);

        var revisionSchemeConfig = patcherConfig.getConfig("revision_scheme");

        return new PatchersRevisionsHolderFactory<>(podSpecPatchersHolder).from(revisionScheme, revisionSchemeConfig, PatcherType.POD_SPEC);
    }

    public static Tracer registerTracer(Config config) {
        var isEnabled = config.getBoolean("is_enabled");
        if (!isEnabled) {
            return GlobalTracer.get();
        }
        var builder = new Configuration(config.getString("service_name"))
                .withSampler(Configuration.SamplerConfiguration.fromEnv()
                        .withType(ConstSampler.TYPE)
                        .withParam(1))
                .withReporter(Configuration.ReporterConfiguration.fromEnv()
                        .withLogSpans(config.getBoolean("with_logs"))
                        .withSender(Configuration.SenderConfiguration.fromEnv()
                                .withAgentHost("::1"))
                );

        JaegerTracer tracer = builder.getTracer();
        GlobalTracer.registerIfAbsent(tracer);

        return tracer;
    }

    public static ResourceSupplierFactory resourceSupplierFactory(Config config, SerialExecutor executor,
                                                                  Clock clock, GaugeRegistry registry,
                                                                  Cache<ResourceWithMeta> persistance, HttpServiceMetrics serviceMetrics) {
        return new ResourceSupplierFactory(executor,
                releaseGetter(config, serviceMetrics, persistance),
                config.getDuration(
                "update_interval"),
                config.getDuration("retry_on_start_interval"), clock, registry, persistance);
    }

    private static SandboxReleaseGetter releaseGetter(Config config, HttpServiceMetrics serviceMetrics, Cache<ResourceWithMeta> cache) {
        var jsonHttpGetter = new JsonHttpGetter(HttpServiceMetrics.Source.SANDBOX,
                buildAsyncHttpClient(config.getConfig("http_client").getDuration("request_timeout")),
                serviceMetrics);
        Map<String, String> attributes = labels(config.getConfig("attributes"));
        if (attributes.isEmpty()) {
            throw new IllegalArgumentException("Attributes for sandbox release filtering must be specified");
        }
        return new SandboxReleaseGetterImpl(config.getString("base_url"),
                token(config.getString("token_file")), attributes, cache, jsonHttpGetter);
    }

    private static Server buildServer(Config config, int port, String address, Map<String, HttpServlet> servlets) {
        int workerThreads = config.getInt("worker_threads");
        Server server = new Server(new QueuedThreadPool(workerThreads, workerThreads));

        int selectorThreads = config.getInt("selector_threads");
        ServerConnector connector = new ServerConnector(server, 0, selectorThreads);
        connector.setReuseAddress(true);
        connector.setHost(address);
        connector.setPort(port);
        connector.setIdleTimeout(config.getDuration("connection_timeout", TimeUnit.MILLISECONDS));
        connector.setAcceptQueueSize(config.getInt("accept_backlog_size"));
        server.addConnector(connector);

        ServletContextHandler context = new ServletContextHandler();
        context.setContextPath("/");
        servlets.forEach((name, servlet) -> context.addServlet(new ServletHolder(servlet), name));
        HandlerCollection handlers = new HandlerCollection();
        handlers.setHandlers(new Handler[]{context, new DefaultHandler()});
        server.setHandler(handlers);
        return server;
    }

    public static Optional<String> logbrokerAllocationId(Config config) {
        return allocationId(config, "logbroker_tools_allocation_id");
    }

    public static Optional<String> tvmAllocationId(Config config) {
        return allocationId(config, "tvm_tools_allocation_id");
    }

    public static Optional<String> podAgentAllocationId(Config config) {
        return allocationId(config, "pod_agent_allocation_id");
    }

    private static Optional<String> allocationId(Config config, String pathInConfig) {
        String allocationId = config.getString(pathInConfig);

        if (allocationId.isEmpty()) {
            return Optional.empty();
        }
        return Optional.of(allocationId);
    }

    public static Server publicRest(Config config, MetricRegistry metricRegistry) {
        return buildServer(config, config.getInt("port"), "::", ImmutableMap.of(
                "/unistat", new UnistatServlet(metricRegistry)));
    }

    public static Server privateRest(Config config, Set<GarbageRequestReceiver> garbageRequestReceivers,
                                     Map<String, ResourceSupplier> resourceSuppliers,
                                     CacheSet caches,
                                     Function<String, CacheStorageFactory> storageFactorySupplier,
                                     AsyncYpClientsMap ypClients) {
        return buildServer(config, config.getInt("port") + 1, "localhost", Map.of(
                "/garbage/list", new GarbageListHandler(garbageRequestReceivers),
                "/garbage/force_collect", new GarbageForceCollectHandler(garbageRequestReceivers),
                "/cache/export", new CacheServlet(CacheServlet.RequestType.EXPORT, caches, storageFactorySupplier),
                "/cache/remove", new CacheServlet(CacheServlet.RequestType.REMOVE, caches),
                "/relations/check", new RelationsServlet(ypClients),
                "/resources", new ResourcesServlet(resourceSuppliers)));
    }

    public static List<String> unwrapStringList(ConfigList list) {
        return list.unwrapped().stream()
                .map(item -> (String) item)
                .collect(Collectors.toList());
    }

    public static <T> Optional<T> getOptional(Config config,
                                              String key,
                                              BiFunction<Config, String, T> extractor) {
        var value = config.hasPath(key)
                ? extractor.apply(config, key)
                : null;

        return Optional.ofNullable(value);
    }

    public static void validateSidecarDiskAllocations(List<Optional<String>> allocations) {
        long emptyAllocationsCount = allocations.stream().filter(Optional::isEmpty).count();

        boolean allAreEmpty = (emptyAllocationsCount == allocations.size());

        if ((emptyAllocationsCount != 0) && !allAreEmpty) {
            throw new RuntimeException("Allocations should all be empty or all should be filled");
        }

        Set<Optional<String>> allocationsSet = new HashSet<>(allocations);

        if (!allAreEmpty && allocationsSet.size() < allocations.size()) {
            throw new RuntimeException("Allocations have duplications");
        }
    }

    public static Config logbrokerPatcherConfig(Config patchersConfig) {
        return patchersConfig.getConfig(LOGBROKER_PATCHER_CONFIG_PATH);
    }

    public static <T> Config withValue(Config config,
                                       String key,
                                       T value) {
        return config.withValue(key, ConfigValueFactory.fromAnyRef(value));
    }

    public static <T> Config withValue(Config config,
                                        String key,
                                        Optional<T> valueOptional) {
        return valueOptional.map(
                value -> withValue(config, key, value)
        ).orElse(
                config.withoutPath(key)
        );
    }

    public static Config withConfigValue(ConfigMergeable parentConfig,
                                         String key,
                                         Config configValue) {
        return ConfigFactory.parseMap(Map.of(
                key,
                configValue.root()
        )).withFallback(parentConfig);
    }

    private static AsyncHttpClient buildAsyncHttpClient(Duration requestTimeout) {
        return new DefaultAsyncHttpClient(new DefaultAsyncHttpClientConfig.Builder()
                .setRequestTimeout((int) requestTimeout.toMillis()).build());
    }

    public static GlobalContext getGlobalContext(Config config) {
        var disabledClusters = Set.copyOf(config.getStringList("yp.disabled_clusters"));
        return new GlobalContext(disabledClusters);
    }
}
