package ru.yandex.infra.stage;

import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicInteger;

import ru.yandex.infra.controller.metrics.GaugeRegistry;
import ru.yandex.infra.controller.metrics.GolovanableGauge;

public class HttpServiceMetricsImpl implements HttpServiceMetrics {

    private static final String DOCKER_METRIC = "http.docker.%s";
    private static final String SANDBOX_METRIC = "http.sandbox.%s";

    private static final String REQUEST = "request";
    private static final String HTTP_ERROR = "http-error";
    private static final String JSON_PARSE_ERROR = "json-parse-error";

    private final ConcurrentHashMap<String, AtomicInteger> registryMap = new ConcurrentHashMap<>();

    HttpServiceMetricsImpl(GaugeRegistry metricRegistry) {
        registryMap.put(String.format(DOCKER_METRIC, REQUEST), new AtomicInteger(0));
        registryMap.put(String.format(DOCKER_METRIC, HTTP_ERROR), new AtomicInteger(0));
        registryMap.put(String.format(DOCKER_METRIC, JSON_PARSE_ERROR), new AtomicInteger(0));

        registryMap.put(String.format(SANDBOX_METRIC, REQUEST), new AtomicInteger(0));
        registryMap.put(String.format(SANDBOX_METRIC, HTTP_ERROR), new AtomicInteger(0));
        registryMap.put(String.format(SANDBOX_METRIC, JSON_PARSE_ERROR), new AtomicInteger(0));

        for (Map.Entry<String, AtomicInteger> entry : registryMap.entrySet()) {
            metricRegistry.add(entry.getKey(), new GolovanableGauge<>(entry::getValue, "dmmm"));
        }
    }

    private String sourceFormat(Source source) {
        switch (source) {
            case DOCKER:
                return DOCKER_METRIC;
            case SANDBOX:
                return SANDBOX_METRIC;
            default:
                throw new RuntimeException(String.format("Unknown source metrics: %s", source.name()));
        }
    }

    @Override
    public void addRequest(Source source) {
        registryMap.get(String.format(sourceFormat(source), REQUEST)).incrementAndGet();
    }

    @Override
    public void addErrorResponse(Source source) {
        registryMap.get(String.format(sourceFormat(source), HTTP_ERROR)).incrementAndGet();
    }

    @Override
    public void addParsingError(Source source) {
        registryMap.get(String.format(sourceFormat(source), JSON_PARSE_ERROR)).incrementAndGet();
    }
}
