package ru.yandex.infra.stage;

import java.io.File;
import java.time.Clock;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.atomic.AtomicInteger;

import com.codahale.metrics.MetricRegistry;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Throwables;
import com.google.common.collect.ImmutableList;
import com.typesafe.config.Config;
import com.typesafe.config.ConfigFactory;
import com.typesafe.config.ConfigParseOptions;
import org.eclipse.jetty.server.Server;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.infra.controller.YtSettings;
import ru.yandex.infra.controller.concurrent.DummyLeaderService;
import ru.yandex.infra.controller.concurrent.LeaderService;
import ru.yandex.infra.controller.dto.SchemaMeta;
import ru.yandex.infra.controller.dto.StageMeta;
import ru.yandex.infra.controller.metrics.GaugeRegistry;
import ru.yandex.infra.controller.metrics.GolovanableGauge;
import ru.yandex.infra.controller.metrics.LeaderGaugeRegistry;
import ru.yandex.infra.controller.metrics.NamespacedGaugeRegistry;
import ru.yandex.infra.controller.util.ExitUtils;
import ru.yandex.infra.controller.yp.LabelBasedRepository;
import ru.yandex.infra.controller.yp.YpObjectSettings;
import ru.yandex.infra.controller.yp.YpObjectTransactionalRepository;
import ru.yandex.infra.stage.cache.CacheSet;
import ru.yandex.infra.stage.cache.CacheStorageFactory;
import ru.yandex.infra.stage.cache.CachedObjectType;
import ru.yandex.infra.stage.cache.StorageBasedCache;
import ru.yandex.infra.stage.concurrent.SerialExecutor;
import ru.yandex.infra.stage.deployunit.DeployUnitTimelineManager;
import ru.yandex.infra.stage.deployunit.LogbrokerTopicConfigResolver;
import ru.yandex.infra.stage.deployunit.LogbrokerTopicConfigResolverImpl;
import ru.yandex.infra.stage.deployunit.SandboxResourcesResolver;
import ru.yandex.infra.stage.docker.DockerImagesResolver;
import ru.yandex.infra.stage.inject.ControllerFactoryImpl;
import ru.yandex.infra.stage.inject.GCSettings;
import ru.yandex.infra.stage.inject.ObjectLifeCycleManagerFactory;
import ru.yandex.infra.stage.podspecs.EndpointSetSpecCompositePatcher;
import ru.yandex.infra.stage.podspecs.PodSpecCompositePatcher;
import ru.yandex.infra.stage.podspecs.ResourceSupplierFactory;
import ru.yandex.infra.stage.podspecs.SpecPatcher;
import ru.yandex.infra.stage.podspecs.patcher.PatcherParameters;
import ru.yandex.infra.stage.podspecs.patcher.dynamic_resource.DynamicResourceResolvedResourcesPatcher;
import ru.yandex.infra.stage.podspecs.patcher.logbroker.LogbrokerPatcherUtils;
import ru.yandex.infra.stage.podspecs.revision.RevisionsHolder;
import ru.yandex.infra.stage.protobuf.Converter;
import ru.yandex.infra.stage.util.AdaptiveRateLimiter;
import ru.yandex.infra.stage.yp.AclUpdater;
import ru.yandex.infra.stage.yp.AppendingAclUpdater;
import ru.yandex.infra.stage.yp.AsyncYpClientsMap;
import ru.yandex.infra.stage.yp.EpochDecoratorRepositoryFactory;
import ru.yandex.infra.stage.yp.GarbageApproverRef;
import ru.yandex.infra.stage.yp.LabelRepositoryFactory;
import ru.yandex.infra.stage.yp.RelationController;
import ru.yandex.infra.stage.yp.RelationControllerImpl;
import ru.yandex.infra.stage.yp.RepositoryFactory;
import ru.yandex.yp.client.api.DataModel;
import ru.yandex.yp.client.api.DynamicResource;
import ru.yandex.yp.client.api.TPodTemplateSpec;
import ru.yandex.yp.client.api.TProjectSpec;
import ru.yandex.yp.client.api.TProjectStatus;
import ru.yandex.yp.client.api.TStageSpec;
import ru.yandex.yp.client.api.TStageStatus;
import ru.yandex.yp.model.YpObjectType;

import static ru.yandex.infra.controller.metrics.MetricUtils.buildMetricRegistry;
import static ru.yandex.infra.controller.util.ConfigUtils.token;
import static ru.yandex.infra.stage.ConfigUtils.REST_SERVER_CONFIG_KEY;
import static ru.yandex.infra.stage.ConfigUtils.logbrokerAllocationId;
import static ru.yandex.infra.stage.ConfigUtils.podAgentAllocationId;
import static ru.yandex.infra.stage.ConfigUtils.tvmAllocationId;
import static ru.yandex.infra.stage.ConfigUtils.validateSidecarDiskAllocations;
import static ru.yandex.infra.stage.ConfigUtils.ypProjectRepository;

public class Main {

    private static final Logger LOG = LoggerFactory.getLogger(Main.class);
    private static final String STAGECTL_SERVICE_NAME = "stagectl";

    @VisibleForTesting
    public static final String REVISION_SCHEME_FILE_NAME = "revision_scheme.json";
    private static final String APPLICATION_DEFAULT_CONFIG_FILE_NAME = "application_defaults.conf";
    private static final String METRIC_CONFIG_VALIDATION_ERRORS = "config.validation.errors";

    public static void main(String[] args) {
        // To exit app if main thread throws
        try {
            doMain(args);
        } catch (Exception e) {
            LOG.error("Exception in main:", e);
            ExitUtils.gracefulExit(ExitUtils.EXCEPTION_MAIN);
        }
    }

    private static void doMain(String[] args) {
        Bootstrap bootstrap = startStandbyReplica(args);
        bootstrap.leaderService.ensureLeadership();
        runAsLeader(bootstrap);
    }

    private static Bootstrap startStandbyReplica(String[] args) {
        LOG.info("------------------------------------------------------------------------------------------------------");
        LOG.info("Starting new instance of Stage Controller...");
        Config config = ConfigFactory.parseFile(new File(args[0]), ConfigParseOptions.defaults().setAllowMissing(false))
                .withFallback(ConfigFactory.load(APPLICATION_DEFAULT_CONFIG_FILE_NAME));

        boolean isReadonlyMode = config.getBoolean("readonly_mode");
        if (isReadonlyMode) {
            LOG.warn("!!! STARTING IN READONLY MODE -> ONLY READ OPERATIONS, NO CHANGES TO YP/YT !!!");
        }

        ConfigUtils.registerTracer(config.getConfig("jaeger"));

        Config metricsConfig = config.getConfig("metrics");
        MetricRegistry metricRegistry = buildMetricRegistry(metricsConfig);
        List<String> deployTimelineStages = metricsConfig.getStringList("deploy_timeline_stages");
        DeployUnitTimelineManager.setupMetric(deployTimelineStages, metricRegistry);

        YtSettings ytSettings = ru.yandex.infra.controller.util.ConfigUtils.ytSettings(config.getConfig("yt"));
        LeaderService leaderService = isReadonlyMode ?
                new DummyLeaderService(metricRegistry) :
                ru.yandex.infra.controller.util.ConfigUtils.leaderService(STAGECTL_SERVICE_NAME,
                        config.getConfig("leader_lock"), metricRegistry, ytSettings);

        Converter converter = new Converter(config);

        Server publicRest = ConfigUtils.publicRest(config.getConfig(REST_SERVER_CONFIG_KEY), metricRegistry);
        startServer(publicRest);

        LeaderGaugeRegistry gaugeRegistry = new LeaderGaugeRegistry(metricRegistry, leaderService);
        SerialExecutor serialExecutor = new SerialExecutor("serializer", new NamespacedGaugeRegistry(gaugeRegistry, "serial_executor"));

        return new Bootstrap(config, leaderService, gaugeRegistry, serialExecutor, converter, ytSettings, isReadonlyMode);
    }

    private static void runAsLeader(Bootstrap bootstrap) {
        Config config = bootstrap.config;
        Converter converter = bootstrap.converter;
        GaugeRegistry gaugeRegistry = bootstrap.gaugeRegistry;
        LeaderService leaderService = bootstrap.leaderService;
        SerialExecutor serialExecutor = bootstrap.serialExecutor;
        YtSettings ytSettings = bootstrap.ytSettings;
        boolean isReadonlyMode = bootstrap.isReadonlyMode;

        Clock clock = Clock.systemDefaultZone();

        Config cacheConfig = config.getConfig("cache");
        String cacheStorage = cacheConfig.getString("storage");
        CacheStorageFactory storageFactory = ConfigUtils.getCacheStorageFactory(cacheConfig.getConfig(cacheStorage),
                ytSettings, gaugeRegistry, isReadonlyMode);
        CacheSet caches = new CacheSet(type -> new StorageBasedCache<>(type, storageFactory, gaugeRegistry));

        AsyncYpClientsMap ypClients = ConfigUtils.ypClientMap(config.getConfig("yp"), gaugeRegistry, isReadonlyMode);

        Map<String, Integer> blackboxEnvironments =
                ConfigUtils.blackboxEnvironments(config.getConfig("tvm.blackbox_environments"));

        Config patcherConfig = config.getConfig("pod_specs_patcher");
        Optional<String> logbrokerDiskAllocationId = logbrokerAllocationId(patcherConfig);
        Optional<String> tvmDiskAllocationId = tvmAllocationId(patcherConfig);
        Optional<String> podAgentDiskAllocationId = podAgentAllocationId(patcherConfig);

        validateSidecarDiskAllocations(ImmutableList.of(logbrokerDiskAllocationId, tvmDiskAllocationId, podAgentDiskAllocationId));

        int pageSize = config.getInt("yp.request_page_size");
        String epochKey = config.getString("yp.epoch_key");
        Optional<String> vcsKey = ConfigUtils.getOptional(config, "yp.vcs_key", ConfigUtils.STRING_FROM_CONFIG);

        YpObjectTransactionalRepository<StageMeta, TStageSpec, TStageStatus> ypStageRepository =
                ConfigUtils.ypStageRepository(ypClients.getMultiClusterClient(), config, pageSize,
                        bootstrap.leaderService::getCurrentEpoch, epochKey, vcsKey, gaugeRegistry, isReadonlyMode);

        GaugeRegistry stageRegistry = new NamespacedGaugeRegistry(gaugeRegistry, "stages");
        HttpServiceMetrics serviceMetrics = new HttpServiceMetricsImpl(gaugeRegistry);

        Config ypRateLimiterConfig = config.getConfig("yp.rate_limiter");
        AdaptiveRateLimiter ypRateLimiter = ConfigUtils.adaptiveRateLimiter(clock, ypRateLimiterConfig);
        StageStatusSender stageStatusSender = new StageStatusSenderImpl(
                ypStageRepository,
                serialExecutor,
                stageRegistry,
                converter,
                ypRateLimiter,
                config.getDuration("stage_repository.status_sender.initial_retry_timeout"),
                config.getDuration("stage_repository.status_sender.max_retry_timeout")
        );

        AclUpdater commonAclUpdater = AppendingAclUpdater.configure(config.getConfig("yp_repositories.acl"));
        AclUpdater replicaSetAclUpdater =
                AppendingAclUpdater.configure(config.getConfig("yp_repositories.replica_sets_common_acl"));

        Config logbrokerConfig = config.getConfig("logbroker");

        Config dockerResolverRateLimiterConfig = config.getConfig("docker_resolver.rate_limiter");
        AdaptiveRateLimiter dockerResolverRateLimiter = ConfigUtils.adaptiveRateLimiter(clock, dockerResolverRateLimiterConfig);
        DockerImagesResolver dockerImagesResolver = ConfigUtils.dockerResolver(
                config.getConfig("docker_resolver"),
                serialExecutor,
                gaugeRegistry,
                serviceMetrics,
                caches,
                dockerResolverRateLimiter
        );

        Config sandboxConfig = patcherConfig.getConfig("sandbox");
        ResourceSupplierFactory resourceSupplierFactory =
                ConfigUtils.resourceSupplierFactory(sandboxConfig, serialExecutor, clock, gaugeRegistry,
                        caches.get(CachedObjectType.RESOURCES),
                        serviceMetrics);

        Optional<List<String>> allSidecarDiskAllocationIds = logbrokerDiskAllocationId
                .map(logbrokerAllocation -> ImmutableList.of(
                        logbrokerAllocation,
                        tvmDiskAllocationId.orElseThrow(() -> new RuntimeException("Expected tvmDiskAllocationId, but it is empty")),
                        podAgentDiskAllocationId.orElseThrow(() -> new RuntimeException("Expected podAgentAllocationId, but it is empty"))
                        )
                );

        long releaseGetterTimeoutSeconds = sandboxConfig.getDuration("release_getter_timeout").toSeconds();

        var patcherParameters = PatcherParameters.with(
                podAgentDiskAllocationId,
                logbrokerDiskAllocationId,
                tvmDiskAllocationId,
                allSidecarDiskAllocationIds,
                releaseGetterTimeoutSeconds,
                blackboxEnvironments
        );

        RevisionsHolder<TPodTemplateSpec.Builder> podSpecPatcherRevisionsHolder = ConfigUtils.podSpecPatchersRevisionsHolder(
                patcherConfig,
                resourceSupplierFactory,
                patcherParameters,
                REVISION_SCHEME_FILE_NAME
        );

        RevisionsHolder<DataModel.TEndpointSetSpec.Builder> endpointSetSpecPatcherRevisionsHolder =
                ConfigUtils.endpointSetSpecPatchersRevisionsHolder(
                patcherConfig,
                        REVISION_SCHEME_FILE_NAME
        );
        GCSettings gcSettings = ConfigUtils.loadGCSettings(config.getConfig("yp.garbage_collector_settings"));

        resourceSupplierFactory.startAll(sandboxConfig.getDuration("start_timeout"));

        AtomicInteger configValidationErrors = new AtomicInteger(0);
        gaugeRegistry.add(METRIC_CONFIG_VALIDATION_ERRORS, new GolovanableGauge<>(configValidationErrors::get, "dmmm"));
        if (config.getBoolean("infra_resources.validation_enabled")) {
            try {
                resourceSupplierFactory.validateAll(sandboxConfig.getDuration("start_timeout"));
            } catch (Exception e) {
                configValidationErrors.incrementAndGet();
                LOG.error("Exception during infra resource config validation: ", e);
            }
        }

        GarbageApproverRef garbageApprover = new GarbageApproverRef();

        RepositoryFactory repositoryFactory = config.getBoolean("leader_lock.use_lock") && !isReadonlyMode ?
                new EpochDecoratorRepositoryFactory(
                        ConfigUtils.labels(config.getConfig("yp_repositories.labels")),
                        pageSize,
                        leaderService::getCurrentEpoch, epochKey, ypClients, vcsKey,
                        gaugeRegistry) :
                new LabelRepositoryFactory(ConfigUtils.labels(config.getConfig("yp_repositories.labels")),
                        pageSize, ypClients, vcsKey, gaugeRegistry);

        Map<YpObjectType, YpObjectSettings> ypObjectsCacheSettings = YpObjectSettings.loadFromConfig(config.getConfig("yp.settings_per_object_type"));
        ObjectLifeCycleManagerFactory facadeFactory = new ObjectLifeCycleManagerFactory(serialExecutor,
                gaugeRegistry, clock, garbageApprover, repositoryFactory, gcSettings, ypObjectsCacheSettings);

        LogbrokerTopicConfigResolver logbrokerTopicConfigResolver = new LogbrokerTopicConfigResolverImpl(
                LogbrokerPatcherUtils.LOGBROKER_AGENT_COMMUNAL_TOPIC_DESCRIPTION,
                token(logbrokerConfig.getString("communal_topic_tvm_token_file")));

        StageValidator validator = new StageValidatorImpl(
                ypClients.getClusters(),
                blackboxEnvironments.keySet(),
                logbrokerDiskAllocationId,
                tvmDiskAllocationId,
                podAgentDiskAllocationId,
                podSpecPatcherRevisionsHolder,
                logbrokerConfig.getConfig("destroy_policy")
        );

        Config sandboxRateLimiterConfig = sandboxConfig.getConfig("rate_limiter");
        AdaptiveRateLimiter sandboxRateLimiter = ConfigUtils.adaptiveRateLimiter(clock, sandboxRateLimiterConfig);
        SandboxResourcesResolver sandboxResourcesResolver = ConfigUtils.sandboxResourcesResolver(
                sandboxConfig,
                serialExecutor,
                serviceMetrics,
                sandboxRateLimiter,
                caches.get(CachedObjectType.SANDBOX_RESOURCES)
        );

        Config relationConfig = config.getConfig("yp.relation_controller");
        RelationController relationController = relationConfig.getBoolean("enabled") ?
                new RelationControllerImpl(ypClients.getMultiClusterClient(),
                        new NamespacedGaugeRegistry(gaugeRegistry, "relations"),
                        relationConfig.getBoolean("add_missed_relations"),
                        relationConfig.getBoolean("remove_relations"),
                        relationConfig.getDuration("cache_reset_interval"),
                        relationConfig.getDuration("init_retry_interval"),
                        relationConfig.getDuration("sync_timeout"),
                        relationConfig.getInt("request_page_size")) :
                RelationController.EMPTY;
        GlobalContext globalContext = ConfigUtils.getGlobalContext(config);

        SpecPatcher<TPodTemplateSpec.Builder> podSpecPatcher = new PodSpecCompositePatcher(podSpecPatcherRevisionsHolder);
        SpecPatcher<DataModel.TEndpointSetSpec.Builder> endpointSetSpecPatcher = new EndpointSetSpecCompositePatcher(endpointSetSpecPatcherRevisionsHolder);
        SpecPatcher<DynamicResource.TDynamicResourceSpec.Builder> dynamicResourceSpecPatcher = new DynamicResourceResolvedResourcesPatcher(sandboxResourcesResolver);

        ControllerFactoryImpl factory = new ControllerFactoryImpl(ypClients, clock,
                commonAclUpdater, dockerImagesResolver, converter, validator, stageStatusSender, facadeFactory,
                replicaSetAclUpdater, AclUpdater.IDENTITY, ConfigUtils.aclFilter(config.getConfig("acl_filter")),
                logbrokerTopicConfigResolver, sandboxResourcesResolver, relationController, globalContext,
                podSpecPatcher,
                endpointSetSpecPatcher,
                dynamicResourceSpecPatcher);
        RootControllerImpl rootController = new RootControllerImpl(factory, leaderService, stageRegistry, clock);

        // We have cycle depends. Give ref to fabric and after that init ref
        garbageApprover.init(rootController);

        Server privateRest = ConfigUtils.privateRest(config.getConfig(REST_SERVER_CONFIG_KEY),
                new HashSet<>(factory.listObjectRepositories()), resourceSupplierFactory.getSuppliers(),
                caches,
                storage -> ConfigUtils.getCacheStorageFactory(cacheConfig.getConfig(storage), ytSettings, gaugeRegistry, isReadonlyMode),
                ypClients);
        startServer(privateRest);

        LabelBasedRepository<SchemaMeta, TProjectSpec, TProjectStatus> ypProjectRepository = ypProjectRepository(
                ypClients.getMultiClusterClient(),
                config.getConfig("project_repository"),
                pageSize,
                vcsKey, gaugeRegistry);

        Engine engine = new Engine(
                ypStageRepository,
                ypProjectRepository,
                config.getDuration("engine.update_interval"),
                config.getDuration("engine.main_loop_timeout"),
                config.getDuration("engine.external_resources_wait_timeout"),
                config.getDuration("engine.yp_object_update_wait_timeout"),
                rootController,
                serialExecutor,
                leaderService,
                new HashSet<>(factory.listObjectRepositories()),
                gaugeRegistry,
                ypObjectsCacheSettings);

        engine.start();
    }

    private static void startServer(Server server) {
        try {
            server.start();
        } catch (Exception e) {
            Throwables.throwIfUnchecked(e);
            throw new RuntimeException(e);
        }
    }

    private static class Bootstrap {
        final Config config;
        final LeaderService leaderService;
        final GaugeRegistry gaugeRegistry;
        final SerialExecutor serialExecutor;
        final Converter converter;
        final YtSettings ytSettings;
        final boolean isReadonlyMode;

        public Bootstrap(Config config, LeaderService leaderService,
                         GaugeRegistry gaugeRegistry, SerialExecutor serialExecutor,
                         Converter converter, YtSettings ytSettings, boolean isReadonlyMode) {
            this.config = config;
            this.leaderService = leaderService;
            this.gaugeRegistry = gaugeRegistry;
            this.serialExecutor = serialExecutor;
            this.converter = converter;
            this.ytSettings = ytSettings;
            this.isReadonlyMode = isReadonlyMode;
        }
    }
}
