package ru.yandex.infra.stage;

import java.time.Clock;
import java.time.Duration;
import java.time.Instant;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.atomic.AtomicLong;
import java.util.stream.Collectors;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.bolts.collection.Try;
import ru.yandex.infra.controller.concurrent.LeaderService;
import ru.yandex.infra.controller.dto.SchemaMeta;
import ru.yandex.infra.controller.dto.StageMeta;
import ru.yandex.infra.controller.metrics.GaugeRegistry;
import ru.yandex.infra.controller.metrics.GolovanableGauge;
import ru.yandex.infra.controller.yp.YpObject;
import ru.yandex.infra.stage.deployunit.DeployUnitStats;
import ru.yandex.infra.stage.dto.ClusterAndType;
import ru.yandex.infra.stage.yp.DeployObjectId;
import ru.yandex.infra.stage.yp.GarbageApprover;
import ru.yandex.infra.stage.yp.Retainment;
import ru.yandex.yp.client.api.TProjectSpec;
import ru.yandex.yp.client.api.TProjectStatus;
import ru.yandex.yp.client.api.TStageSpec;
import ru.yandex.yp.client.api.TStageStatus;

public class RootControllerImpl implements RootController, GarbageApprover {
    static final String TOTAL_METRIC = "total_count";
    static final String VALID_METRIC = "valid_count";
    static final String INVALID_METRIC = "invalid_count";
    static final String PARSING_FAILED_METRIC = "parsing_failed_count";
    static final String LIST_DELAY_METRIC = "list_delay_seconds";

    static final String READY_DEPLOY_REPLICA_SET_METRIC = "ready_replica_set_count";
    static final String UNREADY_DEPLOY_REPLICA_SET_METRIC = "unready_replica_set_count";
    static final String READY_ENDPOINT_SET_METRIC = "ready_endpoint_set_count";
    static final String UNREADY_ENDPOINT_SET_METRIC = "unready_endpoint_set_count";
    static final String READY_DYNAMIC_RESOURCE_METRIC = "ready_dynamic_resource_count";
    static final String UNREADY_DYNAMIC_RESOURCE_METRIC = "unready_dynamic_resource_count";
    static final String PREREQUISITES_READY_DEPLOY_UNIT_METRIC = "prerequisites_ready_deploy_unit_count";
    static final String PREREQUISITES_UNREADY_DEPLOY_UNIT_METRIC = "prerequisites_unready_deploy_unit_count";
    static final String TOTAL_DEPLOY_UNIT_METRIC = "total_deploy_unit_count";
    static final String METRIC_STAGE_SPEC_UPDATES_COUNT = "stage_spec_updates_count";

    static final AtomicLong metricStageSpecUpdatesCount = new AtomicLong();

    private static class Stats {
        final Integer total;
        final Integer valid;
        final Integer invalid;
        final Integer parsingFailed;

        public final static Stats EMPTY = new Stats(null, null, null, null);

        Stats(Integer total, Integer valid, Integer invalid, Integer parsingFailed) {
            this.total = total;
            this.valid = valid;
            this.invalid = invalid;
            this.parsingFailed = parsingFailed;
        }
    }

    private static class StatsBuilder {
        int total = 0;
        int valid = 0;
        int invalid = 0;
        int parsingFailed = 0;

        Stats build() {
            return new Stats(total, valid, invalid, parsingFailed);
        }
    }

    private static final Logger LOG = LoggerFactory.getLogger(RootControllerImpl.class);

    private final ValidationControllerFactory factory;
    private final Map<String, ValidationController> validationControllers = new HashMap<>();
    private final LeaderService leaderService;
    private final Clock clock;

    private Set<String> allStageIds = new HashSet<>();

    private volatile Stats stats;
    private volatile DeployUnitStats deployUnitStats;
    private StatsBuilder currentCycleStatsBuilder;
    private DeployUnitStats.Builder currentCycleDeployUnitStatsBuilder;
    private volatile Instant metricLastSuccessfulStagesSyncTimestamp;

    public RootControllerImpl(ValidationControllerFactory factory, LeaderService leaderService, GaugeRegistry registry, Clock clock) {
        this.factory = factory;
        this.clock = clock;
        this.leaderService = leaderService;

        stats = Stats.EMPTY;
        deployUnitStats = DeployUnitStats.EMPTY;
        metricLastSuccessfulStagesSyncTimestamp = clock.instant();

        registry.add(TOTAL_METRIC, () -> stats.total);
        registry.add(VALID_METRIC, () -> stats.valid);
        registry.add(INVALID_METRIC, () -> stats.invalid);
        registry.add(PARSING_FAILED_METRIC, () -> stats.parsingFailed);
        registry.add(LIST_DELAY_METRIC, () -> Duration.between(metricLastSuccessfulStagesSyncTimestamp, clock.instant()).toSeconds());
        registry.add(METRIC_STAGE_SPEC_UPDATES_COUNT, new GolovanableGauge<>(metricStageSpecUpdatesCount::get, "dmmm"));
        registry.add(READY_DEPLOY_REPLICA_SET_METRIC, () -> deployUnitStats.replicaSetReady);
        registry.add(UNREADY_DEPLOY_REPLICA_SET_METRIC, () -> deployUnitStats.replicaSetUnready);
        registry.add(READY_ENDPOINT_SET_METRIC, () -> deployUnitStats.endpointSetReady);
        registry.add(UNREADY_ENDPOINT_SET_METRIC, () -> deployUnitStats.endpointSetUnready);
        registry.add(READY_DYNAMIC_RESOURCE_METRIC, () -> deployUnitStats.dynamicResourcesReady);
        registry.add(UNREADY_DYNAMIC_RESOURCE_METRIC, () -> deployUnitStats.dynamicResourcesUnready);
        registry.add(PREREQUISITES_READY_DEPLOY_UNIT_METRIC, () -> deployUnitStats.prerequisitesReady);
        registry.add(PREREQUISITES_UNREADY_DEPLOY_UNIT_METRIC, () -> deployUnitStats.prerequisitesUnready);
        registry.add(TOTAL_DEPLOY_UNIT_METRIC, () -> deployUnitStats.unitTotal);
    }

    @Override
    public void beginStatisticsCollection() {
        currentCycleStatsBuilder = new StatsBuilder();
        currentCycleDeployUnitStatsBuilder = new DeployUnitStats.Builder();
    }

    @Override
    public void buildStatistics() {
        metricLastSuccessfulStagesSyncTimestamp = clock.instant();

        stats = currentCycleStatsBuilder.build();
        currentCycleStatsBuilder = null;

        deployUnitStats = currentCycleDeployUnitStatsBuilder.build();
        currentCycleDeployUnitStatsBuilder = null;
    }

    @Override
    public void sync(Map<String, Try<YpObject<StageMeta, TStageSpec, TStageStatus>>> currentSpecs,
                     Map<String, Try<YpObject<SchemaMeta, TProjectSpec, TProjectStatus>>> currentProjects) {
        currentSpecs.forEach((stageId, ypObject) -> {
            try {
                ValidationController controller = validationControllers.get(stageId);
                if (controller == null) {
                    LOG.info("New stage {} received", stageId);
                    controller = factory.createValidationController(stageId);
                    validationControllers.put(stageId, controller);
                }

                controller.sync(ypObject, getProjectOpt(currentProjects, ypObject));
                addValidMetric(currentCycleStatsBuilder, controller.getValidConditionType());
                controller.addStats(currentCycleDeployUnitStatsBuilder);
            } catch (Exception exception) {
                LOG.error(String.format("Could not sync stage %s", stageId), exception);
            }
        });

    }

    @Override
    public void processGcForRemovedStages(Set<String> stageIdsWithDeployEngine, Set<String> allStageIds) {
        this.allStageIds = allStageIds;

        List<String> deletedStages = validationControllers.keySet()
                .stream()
                .filter(stageId -> !stageIdsWithDeployEngine.contains(stageId))
                .collect(Collectors.toList());

        deletedStages.forEach(stageId -> {
            try {
                LOG.info("Stage {} not received, removing", stageId);
                var controller = validationControllers.get(stageId);
                controller.shutdown();
                validationControllers.remove(stageId);
            } catch (Exception exception) {
                LOG.error(String.format("Could not shutdown controller for stage %s", stageId), exception);
            }
        });
    }

    private void addValidMetric(StatsBuilder statsBuilder, ValidationController.ValidityType validityType) {
        switch (validityType) {
            case VALID:
                statsBuilder.valid++;
                break;
            case INVALID:
                statsBuilder.invalid++;
                break;
            case PARSING_FAILED:
                statsBuilder.parsingFailed++;
                break;
        }
        statsBuilder.total++;
    }

    @Override
    public Retainment shouldRetain(String objectId, ClusterAndType clusterAndType) {
        if (validationControllers.isEmpty()) {
            return new Retainment(true, "Stage list is empty, will not remove");
        }
        Optional<DeployObjectId> deployIdOpt = DeployObjectId.tryParse(objectId);
        if (deployIdOpt.isPresent()) {
            DeployObjectId deployId = deployIdOpt.get();
            String stageId = deployId.getStageId();
            ValidationController controller = validationControllers.get(stageId);
            if (controller != null) {
                return controller.shouldRetain(deployId, clusterAndType);
            }
            if (allStageIds.contains(stageId)) {
                return new Retainment(true, String.format("Stage '%s' is present with unknown labels", stageId));
            }
            return new Retainment(false, String.format("Stage '%s' is absent", stageId));
        } else {
            return new Retainment(true, "Object id was not generated by Deploy");
        }
    }

    @Override
    public void updateStatuses() {
        if (leaderService.isProcessingAllowed()) {
            validationControllers.forEach((stageId, controller) -> {
                try {
                    controller.updateStatus();
                } catch (Exception exception) {
                    LOG.error(String.format("Failed update status for stage %s", stageId), exception);
                }
            });
        }
    }

    private static Optional<Try<YpObject<SchemaMeta, TProjectSpec, TProjectStatus>>> getProjectOpt(
            Map<String, Try<YpObject<SchemaMeta, TProjectSpec, TProjectStatus>>> projects,
            Try<YpObject<StageMeta, TStageSpec, TStageStatus>> stage) {
        if (stage.isFailure()) {
            return Optional.empty();
        }

        return Optional.ofNullable(projects.get(stage.get().getMeta().getProjectId()));
    }
}
